package require XOTcl

package require BLT
package require swt
package require usb

package require zlib

wm minsize . 1000 700

namespace eval ::mca {
    namespace import ::xotcl::*

    namespace import ::blt::vector
    namespace import ::blt::graph
    namespace import ::blt::tabnotebook

    proc validate {max value} {
        if {![regexp {^[0-9]*$} $value]} {
            return 0
        } elseif {$value > $max} {
            return 0
        } elseif {[string length $value] > 4} {
            return 0
        } else {
            return 1
        }
    }

# -------------------------------------------------------------------------

    Class Display

# -------------------------------------------------------------------------

    Display instproc usbCmd {command} {
        global usb_handle

        if {[catch {$usb_handle writeRaw [usb::convert $command]} result]} {
            puts {Error during write}
            puts $result
        }
    }

# -------------------------------------------------------------------------

    Display instproc usbCmdRead {command width size} {
        global usb_handle

        my usbCmd $command

        set usb_data {}
        if {[catch {$usb_handle readHex $width $size} result]} {
            puts {Error during read}
            puts $result
            set result {}
        }

        my set data $result
    }

# -------------------------------------------------------------------------

    Display instproc usbCmdReadEpt {command size} {
        global usb_handle

        my usbCmd $command

        set usb_data {}
        if {[catch {$usb_handle readEpt $size} result]} {
            puts {Error during read}
            puts $result
            set result {}
        }

        my set data $result
    }

# -------------------------------------------------------------------------

    Class CfgDisplay -superclass Display -parameter {
        {master}
    }

# -------------------------------------------------------------------------

    CfgDisplay instproc init {} {

        my reset

        my setup

        next
    }

# -------------------------------------------------------------------------

    CfgDisplay instproc destroy {} {
        next
    }

# -------------------------------------------------------------------------

    CfgDisplay instproc reset {} {
    }

# -------------------------------------------------------------------------

    CfgDisplay instproc start {} {
        my instvar config

        trace add variable [myvar dac1] write [myproc dac1_update]
        trace add variable [myvar dac2] write [myproc dac2_update]
        trace add variable [myvar polar] write [myproc polar_update]

        ${config(1)}.dac1 set 0
        ${config(1)}.dac2 set 0

        ${config(2)}.polar1 select
        ${config(2)}.polar2 select
        ${config(2)}.polar3 select
    }

# -------------------------------------------------------------------------

    CfgDisplay instproc setup {} {
        my instvar number master
        my instvar config

        set config(1) [labelframe ${master}.cfg1 -borderwidth 1 -relief sunken -text {DAC}]
        set config(2) [labelframe ${master}.cfg2 -borderwidth 1 -relief sunken -text {polarity inversion}]

        frame ${config(1)}.limits
        label ${config(1)}.limits.min -text {0.0V}
        label ${config(1)}.limits.max -text {-3.3V}

        scale ${config(1)}.dac1 -orient vertical -from 0 -to 4095 -tickinterval 500 -variable [myvar dac1]
        scale ${config(1)}.dac2 -orient vertical -from 0 -to 4095 -tickinterval 0 -variable [myvar dac2]

        checkbutton ${config(2)}.polar1 -text {channel 1} -variable [myvar polar(1)]
        checkbutton ${config(2)}.polar2 -text {channel 2} -variable [myvar polar(2)]
        checkbutton ${config(2)}.polar3 -text {channel 3} -variable [myvar polar(3)]

        grid ${config(1)} -sticky ns
        grid ${config(2)} -sticky ew -pady 7

        pack ${config(1)}.limits.min -anchor n -side top -pady 10
        pack ${config(1)}.limits.max -anchor s -side bottom -pady 9

        grid ${config(1)}.dac1 ${config(1)}.dac2 ${config(1)}.limits -sticky ns -pady 7

        grid ${config(2)}.polar1
        grid ${config(2)}.polar2
        grid ${config(2)}.polar3

        grid rowconfigure ${master} 0 -weight 1
        grid rowconfigure ${config(1)} 0 -weight 1
        grid rowconfigure ${config(2)} 0 -weight 1
    }

# -------------------------------------------------------------------------

    CfgDisplay instproc dac1_update args {
        my instvar dac1

        set value [format {%03x} $dac1]
        set command 0005012000050030000500[string range $value 0 1]000502[string index $value 2]0

        my usbCmd $command
    }

# -------------------------------------------------------------------------

    CfgDisplay instproc dac2_update args {
        my instvar dac2

        set value [format {%03x} $dac2]
        set command 0005012400050030000500[string range $value 0 1]000502[string index $value 2]0

        my usbCmd $command
    }

# -------------------------------------------------------------------------

    CfgDisplay instproc polar_update args {
        my instvar polar

        set value [format {0%x%x%x} $polar(3) $polar(2) $polar(1)]

        my usbCmd 000A${value}
    }

# -------------------------------------------------------------------------

    Class OscDisplay -superclass Display -parameter {
        {number}
        {master}
    }

# -------------------------------------------------------------------------

    OscDisplay instproc init {} {
        my instvar data xvec yvec

        set xvec [vector #auto]
        set yvec [vector #auto]
        # fill one vector for the x axis with 1025 points
        $xvec seq 0 1024

        my reset

        my setup

        next
    }

# -------------------------------------------------------------------------

    OscDisplay instproc destroy {} {
        next
    }

# -------------------------------------------------------------------------

    OscDisplay instproc reset {} {
        my instvar data xvec yvec

        set data {}

        $yvec set {}
    }

# -------------------------------------------------------------------------

    OscDisplay instproc start {} {
        my instvar config trig_mux disp_mux

        set trig_mux 2
        set disp_mux 2

        trace add variable [myvar data] write [myproc data_update]
        trace add variable [myvar auto] write [myproc auto_update]
        trace add variable [myvar thrs] write [myproc thrs_update]
        trace add variable [myvar thrs_val] write [myproc thrs_val_update]
        trace add variable [myvar disp_val] write [myproc disp_val_update]
        trace add variable [myvar trig_val] write [myproc trig_val_update]

        ${config}.auto_check select
        ${config}.thrs_check select
        ${config}.thrs_field set 1278
        ${config}.disp_uwt2 select
        ${config}.trig_uwt2 select
    }

# -------------------------------------------------------------------------

    OscDisplay instproc setup {} {
        my instvar number master
        my instvar data xvec yvec
        my instvar config auto thrs thrs_val disp_val trig_val

        # create a graph widget and show a grid
        set graph [graph ${master}.graph -height 250 -leftmargin 80]
        $graph crosshairs configure -hide no -linewidth 2 -dashes { 1 1 }
        $graph grid configure -hide no
        $graph legend configure -hide yes
        $graph axis configure x -min 0 -max 1024
        $graph axis configure y -min 0 -max 4100

        set config [frame ${master}.config]

        checkbutton ${config}.auto_check -text {auto update} -variable [myvar auto]

        frame ${config}.spc1 -width 10 -height 10

        checkbutton ${config}.thrs_check -text threshold -variable [myvar thrs]
        spinbox ${config}.thrs_field -from 1 -to 4095 \
            -increment 5 -width 10 -textvariable [myvar thrs_val] \
            -validate all -vcmd {::mca::validate 4095 %P}

        frame ${config}.spc2 -width 10 -height 10

        label ${config}.disp -text {display input}
        radiobutton ${config}.disp_data -text {raw data} -variable [myvar disp_val] -value data
        radiobutton ${config}.disp_uwt1 -text {filter 1} -variable [myvar disp_val] -value uwt1
        radiobutton ${config}.disp_uwt2 -text {filter 2} -variable [myvar disp_val] -value uwt2
        radiobutton ${config}.disp_uwt3 -text {filter 3} -variable [myvar disp_val] -value uwt3
        radiobutton ${config}.disp_base -text {baseline} -variable [myvar disp_val] -value base
#        radiobutton ${config}.disp_sum8 -text {sum of 8} -variable [myvar disp_val] -value sum8

        frame ${config}.spc3 -width 10 -height 10

        label ${config}.trig -text {trigger input}
        radiobutton ${config}.trig_data -text {raw data} -variable [myvar trig_val] -value data
        radiobutton ${config}.trig_uwt1 -text {filter 1} -variable [myvar trig_val] -value uwt1
        radiobutton ${config}.trig_uwt2 -text {filter 2} -variable [myvar trig_val] -value uwt2
        radiobutton ${config}.trig_uwt3 -text {filter 3} -variable [myvar trig_val] -value uwt3
        radiobutton ${config}.trig_base -text {baseline} -variable [myvar trig_val] -value base
#        radiobutton ${config}.trig_sum8 -text {sum of 8} -variable [myvar trig_val] -value sum8

        frame ${config}.spc4 -width 10 -height 10

        button ${config}.acquire -text Acquire \
            -bg green -activebackground green -command [myproc acquire]
        button ${config}.restart -text Restart \
            -bg yellow -activebackground yellow -command [myproc restart]
        button ${config}.register -text Register \
            -bg lightblue -activebackground lightblue -command [myproc register]

        grid ${config}.auto_check -sticky w
        grid ${config}.spc1
        grid ${config}.thrs_check -sticky w
        grid ${config}.thrs_field -sticky ew -pady 1 -padx 5
        grid ${config}.spc2
        grid ${config}.disp -sticky w -pady 1 -padx 3
        grid ${config}.disp_data -sticky w
        grid ${config}.disp_uwt1 -sticky w
        grid ${config}.disp_uwt2 -sticky w
        grid ${config}.disp_uwt3 -sticky w
        grid ${config}.disp_base -sticky w
#        grid ${config}.disp_sum8 -sticky w
        grid ${config}.spc3
        grid ${config}.trig -sticky w -pady 1 -padx 3
        grid ${config}.trig_data -sticky w
        grid ${config}.trig_uwt1 -sticky w
        grid ${config}.trig_uwt2 -sticky w
        grid ${config}.trig_uwt3 -sticky w
        grid ${config}.trig_base -sticky w
#        grid ${config}.disp_sum8 -sticky w
        grid ${config}.spc4
        grid ${config}.acquire -sticky ew -pady 3 -padx 5
        grid ${config}.restart -sticky ew -pady 3 -padx 5
        grid ${config}.register -sticky ew -pady 3 -padx 5

        grid ${graph} -row 0 -column 0 -sticky news
        grid ${config} -row 0 -column 1

        grid rowconfigure ${master} 0 -weight 1
        grid columnconfigure ${master} 0 -weight 1
        grid columnconfigure ${master} 1 -weight 0 -minsize 80

        # enable zooming
        Blt_ZoomStack $graph

        #bind .graph <Motion> {%W crosshairs configure -position @%x,%y}

        # create one element with data for the x and y axis, no dots
        $graph element create Spectrum1 -symbol none -xdata $xvec -ydata $yvec
    }

# -------------------------------------------------------------------------

    OscDisplay instproc data_update args {
        my instvar data yvec
        $yvec set $data
    }

# -------------------------------------------------------------------------

    OscDisplay instproc auto_update args {
        my instvar config auto after_handle

        if {$auto} {
            ${config}.acquire configure -state disabled
            ${config}.restart configure -state disabled
            ${config}.register configure -state disabled

            my acquire_restart_loop
        } else {
            if {[my exists after_handle]} {
                after cancel $after_handle
            }
            ${config}.acquire configure -state active
            ${config}.restart configure -state active
            ${config}.register configure -state active
        }
    }

# -------------------------------------------------------------------------

    OscDisplay instproc thrs_update args {
        my instvar config number thrs thrs_val

        set val_addr [format %04x [expr {17 + ${number}}]]

        if {$thrs} {
            ${config}.thrs_field configure -state normal
            my thrs_val_update
        } else {
            ${config}.thrs_field configure -state disabled
            my usbCmd ${val_addr}0000
        }
    }

# -------------------------------------------------------------------------

    OscDisplay instproc thrs_val_update args {
        my instvar config number thrs_val

        if {[string equal $thrs_val {}]} {
            set thrs_val 0
        }

        set val_addr [format %04x [expr {17 + ${number}}]]
        set value [format %04x $thrs_val]

        my usbCmd ${val_addr}${value}
    }

# -------------------------------------------------------------------------

    OscDisplay instproc mux {} {
        my instvar trig_mux disp_mux

        format {00%x%x} $trig_mux $disp_mux
    }

# ------------------------------------------------------------------------

    OscDisplay instproc disp_val_update args {
        my instvar number disp_val disp_mux

        set mux_addr [format %04x [expr {20 + ${number}}]]

        switch -- $disp_val {
            data {
                set disp_mux 0
                my usbCmd ${mux_addr}[my mux]
            }
            uwt1 {
                set disp_mux 1
                my usbCmd ${mux_addr}[my mux]
            }
            uwt2 {
                set disp_mux 2
                my usbCmd ${mux_addr}[my mux]
            }
            uwt3 {
                set disp_mux 3
                my usbCmd ${mux_addr}[my mux]
            }
            base {
                set disp_mux 4
                my usbCmd ${mux_addr}[my mux]
            }
        }
    }

# ------------------------------------------------------------------------

    OscDisplay instproc trig_val_update args {
        my instvar number trig_val trig_mux

        set mux_addr [format %04x [expr {20 + ${number}}]]

        switch -- $trig_val {
            data {
                set trig_mux 0
                my usbCmd ${mux_addr}[my mux]
            }
            uwt1 {
                set trig_mux 1
                my usbCmd ${mux_addr}[my mux]
            }
            uwt2 {
                set trig_mux 2
                my usbCmd ${mux_addr}[my mux]
            }
            uwt3 {
                set trig_mux 3
                my usbCmd ${mux_addr}[my mux]
            }
            base {
                set trig_mux 4
                my usbCmd ${mux_addr}[my mux]
            }
        }
    }

# -------------------------------------------------------------------------

    OscDisplay instproc save_data {data} {
        set file [tk_getSaveFile]
        if {[string equal $file {}]} {
            return
        }

        set x [catch {set fid [open $file w+]}]
        set y [catch {puts $fid $data}]
        set z [catch {close $fid}]

        if { $x || $y || $z || ![file exists $file] || ![file isfile $file] || ![file readable $file] } {
            tk_messageBox -icon error \
                -message "An error occurred while writing to \"$file\""
        } else {
            tk_messageBox -icon info \
                -message "File \"$file\" written successfully"
        }
    }

# -------------------------------------------------------------------------

    OscDisplay instproc acquire {} {
        my instvar number
        my usbCmdRead 0002000${number} 2 1024
   }

# -------------------------------------------------------------------------

    OscDisplay instproc restart {} {
        my instvar number
        my usbCmd 0001000${number}
    }

# -------------------------------------------------------------------------

    OscDisplay instproc register {} {
        my save_data [my set data]
    }

# -------------------------------------------------------------------------

    OscDisplay instproc acquire_restart_loop {} {
        my instvar number after_handle

        my acquire
        my restart

        set after_handle [after 1000 [myproc acquire_restart_loop]]
    }

# -------------------------------------------------------------------------

    Class HstDisplay -superclass Display -parameter {
        {number}
        {master}
    }

# -------------------------------------------------------------------------

    HstDisplay instproc init {} {
        my instvar data xvec yvec

        set xvec [vector #auto]
        set yvec [vector #auto]
        # fill one vector for the x axis with 4097 points
        $xvec seq 0 4096

        my reset

        my setup

        next
    }

# -------------------------------------------------------------------------

    HstDisplay instproc destroy {} {
        next
    }

# -------------------------------------------------------------------------

    HstDisplay instproc reset {} {
        my instvar data xvec yvec

        set data {}

        $yvec set {}
    }

# -------------------------------------------------------------------------

    HstDisplay instproc start {} {
        my instvar config base_mux peak_mux

        set base_mux 0
        set peak_mux 1

        trace add variable [myvar axis] write [myproc axis_update]
        trace add variable [myvar data] write [myproc data_update]
        trace add variable [myvar auto] write [myproc auto_update]
        trace add variable [myvar peak] write [myproc peak_update]
        trace add variable [myvar thrs] write [myproc thrs_update]
        trace add variable [myvar thrs_val] write [myproc thrs_val_update]
        trace add variable [myvar base] write [myproc base_update]
        trace add variable [myvar base_typ] write [myproc base_typ_update]
        trace add variable [myvar base_val] write [myproc base_val_update]

        ${config}.axis_check deselect
        ${config}.auto_check select
        ${config}.peak_check select

        ${config}.thrs_check select
        ${config}.thrs_field set 1278

        ${config}.base_const select
        ${config}.base_field set 35
        ${config}.base_check deselect
    }

# -------------------------------------------------------------------------

    HstDisplay instproc setup {} {
        my instvar number master
        my instvar data xvec yvec graph
        my instvar config auto thrs thrs_val base base_typ base_val

        # create a graph widget and show a grid
        set graph [graph ${master}.graph -height 250 -leftmargin 80]
        $graph crosshairs configure -hide no -linewidth 2 -dashes { 1 1 }
        $graph grid configure -hide no
        $graph legend configure -hide yes
        $graph axis configure x -min 0 -max 4096

        set config [frame ${master}.config]

        checkbutton ${config}.axis_check -text {log scale} -variable [myvar axis]

        frame ${config}.spc1 -width 10 -height 10

        checkbutton ${config}.auto_check -text {auto update} -variable [myvar auto]

        frame ${config}.spc2 -width 10 -height 10

        checkbutton ${config}.peak_check -text {peak detect} -variable [myvar peak]

        frame ${config}.spc3 -width 10 -height 10

        checkbutton ${config}.thrs_check -text threshold -variable [myvar thrs]
        spinbox ${config}.thrs_field -from 1 -to 4095 \
            -increment 5 -width 10 -textvariable [myvar thrs_val] \
            -validate all -vcmd {::mca::validate 4095 %P}

        frame ${config}.spc4 -width 10 -height 10

        checkbutton ${config}.base_check -text baseline -variable [myvar base]
        radiobutton ${config}.base_auto -text automatic -variable [myvar base_typ] -value auto
        radiobutton ${config}.base_const -text constant -variable [myvar base_typ] -value const
        spinbox ${config}.base_field -from 1 -to 4095 \
            -increment 5 -width 10 -textvariable [myvar base_val] \
            -validate all -vcmd {::mca::validate 4095 %P}

        frame ${config}.spc5 -width 10 -height 10

        button ${config}.acquire -text Acquire \
            -bg green -activebackground green -command [myproc acquire]
        button ${config}.restart -text Restart \
            -bg yellow -activebackground yellow -command [myproc restart]
        button ${config}.register -text Register \
            -bg lightblue -activebackground lightblue -command [myproc register]

        grid ${config}.axis_check -sticky w
        grid ${config}.spc1
        grid ${config}.auto_check -sticky w
        grid ${config}.spc2
        grid ${config}.peak_check -sticky w
        grid ${config}.spc3
        grid ${config}.thrs_check -sticky w
        grid ${config}.thrs_field -sticky ew -pady 1 -padx 5
        grid ${config}.spc4
        grid ${config}.base_check -sticky w
        grid ${config}.base_auto -sticky w
        grid ${config}.base_const -sticky w
        grid ${config}.base_field -sticky ew -pady 1 -padx 5
        grid ${config}.spc5
        grid ${config}.acquire -sticky ew -pady 3 -padx 5
        grid ${config}.restart -sticky ew -pady 3 -padx 5
        grid ${config}.register -sticky ew -pady 3 -padx 5

        grid ${graph} -row 0 -column 0 -sticky news
        grid ${config} -row 0 -column 1

        grid rowconfigure ${master} 0 -weight 1
        grid columnconfigure ${master} 0 -weight 1
        grid columnconfigure ${master} 1 -weight 0 -minsize 80

        # enable zooming
        Blt_ZoomStack $graph

        #bind .graph <Motion> {%W crosshairs configure -position @%x,%y}

        # create one element with data for the x and y axis, no dots
        $graph element create Spectrum1 -symbol none -smooth step -xdata $xvec -ydata $yvec
    }

# -------------------------------------------------------------------------

    HstDisplay instproc axis_update args {
        my instvar axis graph
        if {$axis} {
            $graph axis configure y -min 1 -max 1E10 -logscale yes
        } else {
            $graph axis configure y -min {} -max {} -logscale no
        }
    }

# -------------------------------------------------------------------------

    HstDisplay instproc data_update args {
        my instvar data yvec
        $yvec set $data
    }

# -------------------------------------------------------------------------

    HstDisplay instproc auto_update args {
        my instvar auto after_handle
        my instvar config
        if {$auto} {
            ${config}.acquire configure -state disabled
            ${config}.register configure -state disabled

            my acquire_loop
        } else {
            if {[my exists after_handle]} {
                after cancel $after_handle
            }
            ${config}.acquire configure -state active
            ${config}.register configure -state active
        }
    }

# -------------------------------------------------------------------------

    HstDisplay instproc mux {} {
        my instvar base_mux peak_mux

        format {00%x%x} $base_mux $peak_mux
    }

# -------------------------------------------------------------------------

    HstDisplay instproc peak_update args {
        my instvar number peak peak_mux

        set mux_addr [format %04x [expr {23 + ${number}}]]

        if {$peak} {
            set peak_mux 1
            my usbCmd ${mux_addr}[my mux]
        } else {
            set peak_mux 0
            my usbCmd ${mux_addr}[my mux]
        }
    }

# -------------------------------------------------------------------------

    HstDisplay instproc thrs_update args {
        my instvar config number thrs thrs_val

        set val_addr [format %04x [expr {14 + ${number}}]]

        if {$thrs} {
            ${config}.thrs_field configure -state normal
            my thrs_val_update
        } else {
            ${config}.thrs_field configure -state disabled
            my usbCmd ${val_addr}0000
        }
    }

# -------------------------------------------------------------------------

    HstDisplay instproc thrs_val_update args {
        my instvar config number thrs_val

        if {[string equal $thrs_val {}]} {
            set thrs_val 0
        }

        set val_addr [format %04x [expr {14 + ${number}}]]
        set value [format %04x $thrs_val]

        my usbCmd ${val_addr}${value}
    }

# -------------------------------------------------------------------------

    HstDisplay instproc base_update args {
        my instvar config number base base_val base_mux

        set mux_addr [format %04x [expr {23 + ${number}}]]
        set val_addr [format %04x [expr {11 + ${number}}]]

        if {$base} {
            ${config}.base_auto configure -state normal
            ${config}.base_const configure -state normal
            my base_typ_update
        } else {
            ${config}.base_auto configure -state disabled
            ${config}.base_const configure -state disabled
            ${config}.base_field configure -state disabled
            set base_mux 0
            my usbCmd ${mux_addr}[my mux]${val_addr}0000
        }
    }

# -------------------------------------------------------------------------

    HstDisplay instproc base_typ_update args {
        my instvar config number base_typ base_val base_mux

        set mux_addr [format %04x [expr {23 + ${number}}]]
        set val_addr [format %04x [expr {11 + ${number}}]]
        set value [format %04x $base_val]

        switch -- $base_typ {
            auto {
                ${config}.base_field configure -state disabled
                set base_mux 1
                my usbCmd ${mux_addr}[my mux]
            }
            const {
                ${config}.base_field configure -state normal
                set base_mux 0
                my usbCmd ${mux_addr}[my mux]${val_addr}${value}
            }
        }
    }

# -------------------------------------------------------------------------

    HstDisplay instproc base_val_update args {
        my instvar number base_val

        if {[string equal $base_val {}]} {
            set base_val 0
        }

        set val_addr [format %04x [expr {11 + ${number}}]]
        set value [format %04x $base_val]

        my usbCmd ${val_addr}${value}
    }

# -------------------------------------------------------------------------

    HstDisplay instproc acquire {} {
        my instvar number
        my usbCmdRead 0002001${number} 4 4096
   }

# -------------------------------------------------------------------------

    HstDisplay instproc restart {} {
        my instvar number
        my usbCmd 0001001${number}
    }

# -------------------------------------------------------------------------

    HstDisplay instproc save_data {data} {
        set file [tk_getSaveFile]
        if {[string equal $file {}]} {
            return
        }

        set x [catch {set fid [open $file w+]}]
        set y [catch {puts $fid $data}]
        set z [catch {close $fid}]

        if { $x || $y || $z || ![file exists $file] || ![file isfile $file] || ![file readable $file] } {
            tk_messageBox -icon error \
                -message "An error occurred while writing to \"$file\""
        } else {
            tk_messageBox -icon info \
                -message "File \"$file\" written successfully"
        }
    }

# -------------------------------------------------------------------------

    HstDisplay instproc register {} {
        my save_data [my set data]
    }

# -------------------------------------------------------------------------

    HstDisplay instproc acquire_loop {} {
        my instvar number after_handle

        my acquire

        set after_handle [after 1000 [myproc acquire_loop]]
    }

# -------------------------------------------------------------------------

    Class EptDisplay -superclass Display -parameter {
        {number}
        {master}
    }

# -------------------------------------------------------------------------

    EptDisplay instproc init {} {
        my instvar data xvec yvec

        set xvec [vector #auto]

        for {set i 0} {$i < 11} {incr i} {
          set yvec($i) [vector #auto]
        }

        # fill one vector for the x axis
        $xvec seq 0 10000

        my reset

        my setup

        next
    }

# -------------------------------------------------------------------------

    EptDisplay instproc destroy {} {
        next
    }

# -------------------------------------------------------------------------

    EptDisplay instproc reset {} {
        my instvar data xvec yvec
        my instvar number_val directory

        set data {}

        set directory $::env(HOMEPATH)
        set number_val 10

        for {set i 0} {$i < 11} {incr i} {
          $yvec($i) set {}
        }
    }

# -------------------------------------------------------------------------

    EptDisplay instproc start {} {
        my instvar config

        trace add variable [myvar recs_val] write [myproc recs_val_update]
        trace add variable [myvar data] write [myproc data_update]
        trace add variable [myvar last] write [myproc data_update]

    }

# -------------------------------------------------------------------------

    EptDisplay instproc setup {} {
        my instvar master
        my instvar data xvec yvec graph
        my instvar config number_val

        # create a graph widget and show a grid
        set display [frame ${master}.display]

        set graph(0) [graph ${display}.graph0 -height 200 -leftmargin 80]
        $graph(0) crosshairs configure -hide no -linewidth 2 -dashes { 1 1 }
        $graph(0) grid configure -hide no
        $graph(0) legend configure -hide yes
        $graph(0) axis configure x -min 0 -max 10000
        $graph(0) axis configure y -min 0 -max 4100

        set graph(1) [graph ${display}.graph1 -height 200 -leftmargin 80]
        $graph(1) crosshairs configure -hide no -linewidth 2 -dashes { 1 1 }
        $graph(1) grid configure -hide no
        $graph(1) legend configure -hide yes
        $graph(1) axis configure x -min 0 -max 10000 -hide yes
        $graph(1) axis configure y -min 0 -max 4100

        set graph(2) [graph ${display}.graph2 -height 200 -leftmargin 80]
        $graph(2) crosshairs configure -hide no -linewidth 2 -dashes { 1 1 }
        $graph(2) grid configure -hide no
        $graph(2) legend configure -hide yes
        $graph(2) axis configure x -min 0 -max 10000 -hide yes
        $graph(2) axis configure y -min 0 -max 4100

        set graph(3) [graph ${display}.graph3 -height 100 -leftmargin 80]
        $graph(3) crosshairs configure -hide no -linewidth 2 -dashes { 1 1 }
        $graph(3) grid configure -hide no
        $graph(3) legend configure -hide yes
        $graph(3) axis configure x -min 0 -max 10000 -hide yes
        $graph(3) axis configure y -min 0 -max 25

        scale ${master}.last -orient horizontal -from 1 -to 35 -tickinterval 0 -showvalue no -variable [myvar last]

        set config [frame ${master}.config -width 120]

        label ${config}.recs -text {number of records}
        spinbox ${config}.recs_field -from 5 -to 100 \
            -increment 5 -width 10 -textvariable [myvar recs_val] \
            -validate all -vcmd {::mca::validate 100 %P}

        frame ${config}.spc1 -width 10 -height 10

        button ${config}.sequence -text {Start Recording}  \
            -bg red -activebackground red -command [myproc sequence]

        frame ${config}.spc2 -width 10 -height 10

        label ${config}.stat -text {}

        frame ${config}.spc3 -width 10 -height 20

        button ${config}.acquire -text Acquire \
            -bg green -activebackground green -command [myproc acquire]
        button ${config}.register -text Register \
            -bg lightblue -activebackground lightblue -command [myproc register]


        grid ${config}.recs -sticky w -pady 1 -padx 3
        grid ${config}.recs_field -sticky ew -pady 1 -padx 5
        grid ${config}.spc1
        grid ${config}.sequence -sticky ew -pady 3 -padx 5
        grid ${config}.spc2
        grid ${config}.stat -sticky w -pady 1 -padx 3
        grid ${config}.spc3
        grid ${config}.acquire -sticky ew -pady 3 -padx 5
        grid ${config}.register -sticky ew -pady 3 -padx 5

        grid ${graph(3)} -row 0 -column 0 -sticky news
        grid ${graph(2)} -row 1 -column 0 -sticky news
        grid ${graph(1)} -row 2 -column 0 -sticky news
        grid ${graph(0)} -row 3 -column 0 -sticky news

        grid ${display} -row 0 -column 0 -sticky news
        grid ${config} -row 0 -column 1

        grid ${master}.last -row 1 -column 0 -columnspan 2 -sticky ew

        grid rowconfigure ${master} 0 -weight 1
        grid columnconfigure ${master} 0 -weight 1
        grid columnconfigure ${master} 1 -weight 0 -minsize 120

        grid columnconfigure ${display} 0 -weight 1
        grid rowconfigure ${display} 0 -weight 1
        grid rowconfigure ${display} 1 -weight 1
        grid rowconfigure ${display} 2 -weight 1
        grid rowconfigure ${display} 3 -weight 1

        # enable zooming
        Blt_ZoomStack $graph(0)
        Blt_ZoomStack $graph(1)
        Blt_ZoomStack $graph(2)
        Blt_ZoomStack $graph(3)

        #bind .graph <Motion> {%W crosshairs configure -position @%x,%y}

        # create one element with data for the x and y axis, no dots
        $graph(0) element create Spectrum0 -symbol none -xdata $xvec -ydata $yvec(0)
        $graph(1) element create Spectrum1 -symbol none -xdata $xvec -ydata $yvec(1)
        $graph(2) element create Spectrum2 -symbol none -xdata $xvec -ydata $yvec(2)
        for {set i 3} {$i < 11} {incr i} {
          $graph(3) element create Spectrum$i -symbol none -xdata $xvec -ydata $yvec($i)
        }
    }

# -------------------------------------------------------------------------

    EptDisplay instproc recs_val_update args {
        my instvar recs_val
        if {[string equal $recs_val {}]} {
            set number_val 0
        }
    }

# -------------------------------------------------------------------------

    EptDisplay instproc data_update args {
        my instvar data xvec yvec graph last

        set first [expr {$last - 1}]
        
        $xvec seq ${first}0000 ${last}0000
        for {set i 0} {$i < 4} {incr i} {
            $graph($i) axis configure x -min ${first}0000 -max ${last}0000
        }

        for {set i 0} {$i < 11} {incr i} {
            $yvec($i) set [lrange [lindex $data $i] ${first}0000 ${last}0000]
        }
    }


# -------------------------------------------------------------------------

    EptDisplay instproc save_data {data} {
        set file [tk_getSaveFile]
        if {[string equal $file {}]} {
            return
        }

        set x [catch {set fid [open $file w+]}]
        set y [catch {puts $fid $data}]
        set z [catch {close $fid}]

        if { $x || $y || $z || ![file exists $file] || ![file isfile $file] || ![file readable $file] } {
            tk_messageBox -icon error \
                -message "An error occurred while writing to \"$file\""
        } else {
            tk_messageBox -icon info \
                -message "File \"$file\" written successfully"
        }
    }

# -------------------------------------------------------------------------

    EptDisplay instproc acquire {} {
        my usbCmdReadEpt 00060000 2097152
   }

# -------------------------------------------------------------------------

    EptDisplay instproc register {} {
        my save_data [my set data]
    }

# -------------------------------------------------------------------------

    EptDisplay instproc sequence {} {
        my instvar config recs_val directory counter

        set directory [tk_chooseDirectory -initialdir $directory -title {Choose a directory}]

        if {[string equal $directory {}]} {
           return
        }

        ${config}.recs_field configure -state disabled
        ${config}.sequence configure -state disabled
        ${config}.acquire configure -state disabled
        ${config}.register configure -state disabled
        
        set counter 1
        
        if {$counter <= $recs_val} {
            ${config}.stat configure -text "record $counter of $recs_val"
            set after_handle [after 100 [myproc acquire_register_loop]]
        }
    }

# -------------------------------------------------------------------------

    EptDisplay instproc acquire_register_loop {} {
        my instvar after_handle
        my instvar config data recs_val directory counter

        my acquire

        set file [file join $directory ept_$counter.dat.gz]

        set x [catch {set fid [open $file w+]}]
        fconfigure $fid -translation binary -encoding binary

        set y [catch {
            puts -nonewline $fid [binary format "H*iH*" "1f8b0800" [clock seconds] "0003"]
            puts -nonewline $fid [zlib deflate $data]
            puts -nonewline $fid [binary format i [zlib crc32 $data]]
            puts -nonewline $fid [binary format i [string length $data]]
        }]

        set z [catch {close $fid}]

        my incr counter

        if { $x || $y || $z || ![file exists $file] || ![file isfile $file] || ![file readable $file] } {
            tk_messageBox -icon error -message "An error occurred while writing to \"$file\""
        } elseif {$counter <= $recs_val} {
            ${config}.stat configure -text "record $counter of $recs_val"
            set after_handle [after 100 [myproc acquire_register_loop]]
            return
        }
        
        ${config}.stat configure -text {}
        ${config}.recs_field configure -state normal
        ${config}.sequence configure -state active
        ${config}.acquire configure -state active
        ${config}.register configure -state active

    }

# -------------------------------------------------------------------------

    Class MemDisplay -superclass Display -parameter {
        {number}
        {master}
    }

# -------------------------------------------------------------------------

    MemDisplay instproc init {} {
        my instvar data xvec yvec

        set xvec [vector #auto]
        set yvec [vector #auto]

        # fill one vector for the x axis
        $xvec seq 0 10000

        my reset

        my setup

        next
    }

# -------------------------------------------------------------------------

    MemDisplay instproc destroy {} {
        next
    }

# -------------------------------------------------------------------------

    MemDisplay instproc reset {} {
        my instvar data xvec yvec

        set data {}

        $yvec set {}
    }

# -------------------------------------------------------------------------

    MemDisplay instproc start {} {
        my instvar config

        trace add variable [myvar data] write [myproc data_update]
        trace add variable [myvar last] write [myproc data_update]
    }

# -------------------------------------------------------------------------

    MemDisplay instproc setup {} {
        my instvar master
        my instvar data xvec yvec
        my instvar config auto thrs thrs_val disp_val trig_val

        # create a graph widget and show a grid
        set graph [graph ${master}.graph -height 250 -leftmargin 80]
        $graph crosshairs configure -hide no -linewidth 2 -dashes { 1 1 }
        $graph grid configure -hide no
        $graph legend configure -hide yes
        $graph axis configure x -min 0 -max 10000
#        $graph axis configure y -min 0 -max 4100

        scale ${master}.last -orient horizontal -from 1 -to 105 -tickinterval 0 -variable [myvar last]

        set config [frame ${master}.config]

        button ${config}.acquire -text Acquire \
            -bg green -activebackground green -command [myproc acquire]
        button ${config}.register -text Register \
            -bg lightblue -activebackground lightblue -command [myproc register]

        grid ${config}.acquire -sticky ew -pady 3 -padx 5
        grid ${config}.register -sticky ew -pady 3 -padx 5

        grid ${graph} -row 0 -column 0 -sticky news
        grid ${config} -row 0 -column 1

        grid ${master}.last -row 1 -column 0 -columnspan 2 -sticky ew

        grid rowconfigure ${master} 0 -weight 1
        grid columnconfigure ${master} 0 -weight 1
        grid columnconfigure ${master} 1 -weight 0 -minsize 80

        # enable zooming
        Blt_ZoomStack $graph

        #bind .graph <Motion> {%W crosshairs configure -position @%x,%y}

        # create one element with data for the x and y axis, no dots
        $graph element create Spectrum -symbol none -xdata $xvec -ydata $yvec
    }

# -------------------------------------------------------------------------

    MemDisplay instproc data_update args {
        my instvar data yvec last

        set first [expr {$last - 1}]
        $yvec set [lrange $data ${first}0000 ${last}0000]
    }

# -------------------------------------------------------------------------

    MemDisplay instproc save_data {data} {
        set file [tk_getSaveFile]
        if {[string equal $file {}]} {
            return
        }

        set x [catch {set fid [open $file w+]}]
        set y [catch {puts $fid $data}]
        set z [catch {close $fid}]

        if { $x || $y || $z || ![file exists $file] || ![file isfile $file] || ![file readable $file] } {
            tk_messageBox -icon error \
                -message "An error occurred while writing to \"$file\""
        } else {
            tk_messageBox -icon info \
                -message "File \"$file\" written successfully"
        }
    }

# -------------------------------------------------------------------------

    MemDisplay instproc acquire {} {
        my usbCmdRead 00040000 1 524288
#        my usbCmdRead 00060000 1 1048576
   }

# -------------------------------------------------------------------------

    MemDisplay instproc register {} {
        my save_data [my set data]
    }

# -------------------------------------------------------------------------

    namespace export MemDisplay
    namespace export EptDisplay
    namespace export HstDisplay
    namespace export OscDisplay
    namespace export CfgDisplay
}

set config [frame .config]
set notebook [::blt::tabnotebook .notebook -borderwidth 1 -selectforeground black -side bottom]

grid ${config} -row 0 -column 0 -sticky ns -padx 3
grid ${notebook} -row 0 -column 1  -sticky news -pady 5

grid rowconfigure . 0 -weight 1
grid columnconfigure . 0 -weight 0  -minsize 50
grid columnconfigure . 1 -weight 1

foreach i {0 1 2} {
    set channel [expr $i + 1]

    set window [frame ${notebook}.hst_$i]
    $notebook insert end -text "Histogram $channel" -window $window -fill both
    ::mca::HstDisplay hst_$i -number $i -master $window

    set window [frame ${notebook}.osc_$i]
    $notebook insert end -text "Pulse shape $channel" -window $window -fill both
    ::mca::OscDisplay osc_$i -number $i -master $window
}

#set window [frame ${notebook}.cfg]
#$notebook insert end -text "Configuration" -window $window -fill both
::mca::CfgDisplay cfg -master $config

set window [frame ${notebook}.ept]
$notebook insert end -text "EPT" -window $window -fill both
::mca::EptDisplay ept -master $window

#set window [frame ${notebook}.mem]
#$notebook insert end -text "Memory test" -window $window -fill both
#::mca::MemDisplay mem -master $window

set usb_handle {}

while {[catch {usb::connect 0x09FB 0x6001 1 1 0} usb_handle]} {
    set answer [tk_messageBox -icon error -type retrycancel \
        -message {Cannot access USB device} -detail $usb_handle]
    if {[string equal $answer cancel]} break
}

# cfg reset
cfg reset

cfg start

foreach i {0 1 2} {
    hst_$i start
    osc_$i start

    hst_$i restart
    osc_$i restart
}

ept start
#mem start