module Paella
	(
		input	wire			CLK_50MHz,
		output	wire			LED,

		inout	wire	[3:0]	TRG,
		inout	wire			I2C_SDA,
		inout	wire			I2C_SCL,
		inout	wire	[4:0]	CON_A,
		input	wire	[15:0]	CON_B,
		input	wire	[12:0]	CON_C,
		input	wire	[1:0]	CON_BCLK,
		input	wire	[1:0]	CON_CCLK,

		input	wire			ADC_DCO,
		input	wire			ADC_FCO,
		input	wire	[2:0]	ADC_D,

		output	wire			USB_SLRD, 
		output	wire			USB_SLWR,
		input	wire			USB_IFCLK,
		input	wire			USB_FLAGA, // EMPTY flag for EP6
		input	wire			USB_FLAGB, // FULL flag for EP8
		input	wire			USB_FLAGC,
		inout	wire			USB_PA0,
		inout	wire			USB_PA1,
		output	wire			USB_PA2,
		inout	wire			USB_PA3,
		output	wire			USB_PA4,
		output	wire			USB_PA5,
		output	wire			USB_PA6,
		inout	wire			USB_PA7,
		inout	wire	[7:0]	USB_PB,

		output	wire			RAM_CLK,
		output	wire			RAM_CE1,
		output	wire			RAM_WE,
		output	wire	[19:0]	RAM_ADDR,
		inout	wire			RAM_DQAP,
		inout	wire	[7:0]	RAM_DQA,
		inout	wire			RAM_DQBP,
		inout	wire	[7:0]	RAM_DQB
	);

	localparam	N		=	3;

	//	Turn output ports off
/*
	assign	RAM_CLK		=	1'b0;
	assign	RAM_CE1		=	1'b0;
	assign	RAM_WE		=	1'b0;
	assign	RAM_ADDR	=	20'h00000;
*/
	assign	RAM_CLK = sys_clk;
	assign	RAM_CE1 = 1'b0;

	//	Turn inout ports to tri-state
	assign	TRG			=	4'bz;
	assign	CON_A		=	5'bz;
	assign	USB_PA0		=	1'bz;
	assign	USB_PA1		=	1'bz;
	assign	USB_PA3		=	1'bz;
	assign	USB_PA7		=	1'bz;
//	assign	RAM_DQAP	=	1'bz;
//	assign	RAM_DQA		=	8'bz;
//	assign	RAM_DQBP	=	1'bz;
//	assign	RAM_DQB		=	8'bz;

	assign	USB_PA2		=	~usb_rden;
	assign	USB_PA4		=	usb_addr[0];
	assign	USB_PA5		=	usb_addr[1];
	assign	USB_PA6		=	~usb_pktend;

	wire			usb_wrreq, usb_rdreq, usb_rden, usb_pktend;
	wire			usb_aclr;
	wire			usb_tx_wrreq, usb_rx_rdreq;
	wire			usb_tx_full, usb_rx_empty;
	wire	[7:0]	usb_tx_data, usb_rx_data;
	wire	[1:0]	usb_addr;

	assign	USB_SLRD = ~usb_rdreq;
	assign	USB_SLWR = ~usb_wrreq;

	usb_fifo usb_unit
	(
		.usb_clk(USB_IFCLK),
		.usb_data(USB_PB),
		.usb_full(~USB_FLAGB),
		.usb_empty(~USB_FLAGA),
		.usb_wrreq(usb_wrreq),
		.usb_rdreq(usb_rdreq),
		.usb_rden(usb_rden),
		.usb_pktend(usb_pktend),
		.usb_addr(usb_addr),

		.clk(sys_clk),
		.aclr(usb_aclr),

		.tx_full(usb_tx_full),
		.tx_wrreq(usb_tx_wrreq),
		.tx_data(usb_tx_data),

		.rx_empty(usb_rx_empty),
		.rx_rdreq(usb_rx_rdreq),
		.rx_q(usb_rx_data)
	);
		
	reg 			bln_reset [N-1:0];
	wire	[11:0]	baseline [N-1:0];
	wire	[11:0]	bln_baseline [N-1:0];

	reg 			ana_reset [N-1:0];
	wire			ana_peak_ready [N-1:0];
	wire			ana_peak_debug [N-1:0];

	reg				osc_reset [N-1:0];
	reg 	[9:0]	osc_addr [N-1:0];
	wire	[9:0]	osc_start_addr [N-1:0];
	wire	[15:0]	osc_q [N-1:0];
	wire			osc_trig [N-1:0];

	wire	[3:0]	osc_mux_sel [N-1:0];
	wire	[11:0]	osc_mux_data [N-1:0];

	wire 			trg_reset [N-1:0];
	wire	[3:0]	trg_mux_sel [N-1:0];
	wire	[11:0]	trg_mux_data [N-1:0];
	wire	[11:0]	trg_thrs [N-1:0];

	reg 			hst_reset [N-1:0];
	reg 	[11:0]	hst_addr [N-1:0];
	wire			hst_data_ready [N-1:0];
	wire	[11:0]	hst_data [N-1:0];
	wire	[31:0]	hst_q [N-1:0];


	wire	[3:0]	hst_mux_sel [N-1:0];
	wire	[12:0]	hst_mux_data [N-1:0];

	wire	[3:0]	bln_mux_sel [N-1:0];
	wire	[11:0]	bln_mux_data [N-1:0];

	wire 			mux_reset, mux_type;
	wire 	[1:0]	mux_chan, mux_byte;
	wire 	[15:0]	mux_addr;
	
	reg		[7:0]	mux_q;
	reg 	[1:0]	mux_max_byte;
	reg 	[15:0]	mux_min_addr, mux_max_addr;

	wire			adc_clk [N-1:0];
	wire 	[11:0]	adc_data [N-1:0];

	wire			data_ready [N-1:0];
    wire	[11:0]	data [N-1:0];
    wire	[11:0]	int_data [N-1:0];

    wire			cmp_data_ready;
    wire	[11:0]	cmp_data;

    wire			ept_data_ready;
    wire	[47:0]	ept_data;
    
	wire	[11:0]	nowhere;

	wire			sys_clk;


/*
    assign	osc_thrs[0] = 16'd40;
    assign	osc_thrs[1] = 16'd60;
    assign	osc_thrs[2] = 16'd40;
    assign	osc_thrs[3] = 16'd1650;
*/
	wire 	[31:0]	uwt_d1 [N-1:0];
	wire 	[31:0]	uwt_a1 [N-1:0];
	wire 	[31:0]	uwt_peak1 [N-1:0];
	wire 	[31:0]	uwt_d2 [N-1:0];
	wire 	[31:0]	uwt_a2 [N-1:0];
	wire 	[31:0]	uwt_peak2 [N-1:0];
	wire 	[31:0]	uwt_d3 [N-1:0];
	wire 	[31:0]	uwt_a3 [N-1:0];
	wire 	[31:0]	uwt_peak3 [N-1:0];

	wire 	[1:0]	uwt_flag1 [N-1:0];
	wire 	[1:0]	uwt_flag2 [N-1:0];
	wire 	[1:0]	uwt_flag3 [N-1:0];

	assign	adc_clk[0] = ADC_FCO;
	assign	adc_clk[1] = ADC_FCO;
	assign	adc_clk[2] = ADC_FCO;

/*    
    assign	adc_clk[2] = CON_CCLK[0];
    assign	adc_data[2] = CON_C[11:0];
*/
/*
	adc_para adc_para_unit (
		.lvds_dco(ADC_DCO),
		.lvds_fco(ADC_FCO),
		.para_data_ready(CON_CCLK[0]),
 		.para_data(CON_C[11:0]),
		.adc_data(adc_data[2]));
*/
/* 
	wire			adc_pll_clk;

	adc_pll adc_pll_unit(
		.inclk0(ADC_FCO),
		.c0(adc_pll_clk));
*/

	sys_pll sys_pll_unit(
		.inclk0(CLK_50MHz),
		.c0(sys_clk));

	test test_unit(
		.clk(ADC_FCO),
		.data(adc_data[2]));
//		.data(nowhere);


	adc_lvds #(
		.size(3),
		.width(12)) adc_lvds_unit (
		.lvds_dco(ADC_DCO),
//		.lvds_dco(adc_pll_clk),
		.lvds_fco(ADC_FCO),
		.lvds_d(ADC_D[2:0]),
//		.adc_data({	adc_data[2],
		.adc_data({	nowhere,
					adc_data[1],
					adc_data[0] }));


	reg		[15:0]	cfg_memory [31:0];
	wire	[15:0]	cfg_src_data;
	wire	[15:0]	cfg_src_addr, cfg_dst_data, cfg_dst_addr;

	wire			cfg_polarity [N-1:0];
	wire	[11:0]	cfg_baseline [N-1:0];
	wire	[11:0]	cfg_hst_threshold [N-1:0];
	wire	[11:0]	cfg_trg_threshold [N-1:0];

	wire 			cfg_reset;

	integer j;

	always @(posedge sys_clk)
	begin
		if (cfg_reset)
		begin
			for(j = 0; j <= 31; j = j + 1)
			begin
				cfg_memory[j] <= 16'd0;
			end
		end
		else
		begin
			cfg_memory[cfg_dst_addr[4:0]] <= cfg_dst_data;
		end
	end

	assign ept_data_ready = cmp_data_ready & data_ready[2] & data_ready[1] & data_ready[0];
	assign ept_data = {cmp_data, int_data[2], int_data[1], int_data[0]};

	adc_fifo cmp_fifo_unit (
		.adc_clk(ADC_FCO),
		.adc_data(CON_B[11:0]),
		.clk(sys_clk),
		.data_ready(cmp_data_ready),
		.data(cmp_data));

	genvar i;

	generate
		for (i = 0; i < N; i = i + 1)
		begin : MCA_CHAIN

			assign cfg_polarity[i] = cfg_memory[10][4*i];
			assign cfg_baseline[i] = cfg_memory[11+i][11:0];
			assign cfg_hst_threshold[i] = cfg_memory[14+i][11:0];
			assign cfg_trg_threshold[i] = cfg_memory[17+i][11:0];

			assign osc_mux_sel[i] = cfg_memory[20+i][3:0];
			assign trg_mux_sel[i] = cfg_memory[20+i][7:4];

			assign hst_mux_sel[i] = cfg_memory[23+i][3:0];
			assign bln_mux_sel[i] = cfg_memory[23+i][7:4];

			adc_fifo adc_fifo_unit (
				.adc_clk(adc_clk[i]),
				.adc_data(adc_data[i]),
				.clk(sys_clk),
				.data_ready(data_ready[i]),
				.data(int_data[i]));

			assign data[i] = (cfg_polarity[i]) ? (int_data[i] ^ 12'hfff) : (int_data[i]);

			uwt_bior31 #(.L(1)) uwt_1_unit (
				.clk(sys_clk),
				.data_ready(data_ready[i]),
				.x({20'h00000, data[i]}),
				.d(uwt_d1[i]),
				.a(uwt_a1[i]),
				.peak(uwt_peak1[i]),
				.flag(uwt_flag1[i]));
		
			uwt_bior31 #(.L(2)) uwt_2_unit (
				.clk(sys_clk),
				.data_ready(data_ready[i]),
				.x(uwt_a1[i]),
				.d(uwt_d2[i]),
				.a(uwt_a2[i]),
				.peak(uwt_peak2[i]),
				.flag(uwt_flag2[i]));
		
			uwt_bior31 #(.L(3)) uwt_3_unit (
				.clk(sys_clk),
				.data_ready(data_ready[i]),
				.x(uwt_a2[i]),
				.d(uwt_d3[i]),
				.a(uwt_a3[i]),
				.peak(uwt_peak3[i]),
				.flag(uwt_flag3[i]));

			lpm_mux #(
				.lpm_size(7),
				.lpm_type("LPM_MUX"),
				.lpm_width(12),
				.lpm_widths(3)) osc_mux_unit (
				.sel(osc_mux_sel[i][2:0]),
				.data({	{ana_peak_debug[i], 11'd0},
						hst_data[i],
//						uwt_d3[i][11:0],
						bln_baseline[i],
						uwt_a3[i][20:9],
						uwt_a2[i][17:6],
						uwt_a1[i][14:3],
						data[i] }),
				.result(osc_mux_data[i]));

			lpm_mux #(
				.lpm_size(7),
				.lpm_type("LPM_MUX"),
				.lpm_width(12),
				.lpm_widths(3)) trg_mux_unit (
				.sel(trg_mux_sel[i][2:0]),
				.data({	{ana_peak_ready[i], 11'd0},
						hst_data[i],
//						uwt_d3[i][11:0],
						bln_baseline[i],
						uwt_a3[i][20:9],
						uwt_a2[i][17:6],
						uwt_a1[i][14:3],
						data[i] }),
				.result(trg_mux_data[i]));

			lpm_mux #(
				.lpm_size(2),
				.lpm_type("LPM_MUX"),
				.lpm_width(13),
				.lpm_widths(1)) hst_mux_unit (
				.sel(hst_mux_sel[i][0]),
				.data({	{uwt_peak3[i][11:0], ana_peak_ready[i]},
						{data[i], data_ready[i]} }),
				.result(hst_mux_data[i]));
	
			lpm_mux #(
				.lpm_size(2),
				.lpm_type("LPM_MUX"),
				.lpm_width(12),
				.lpm_widths(1)) bln_mux_unit (
				.sel(bln_mux_sel[i][0]),
				.data({bln_baseline[i], cfg_baseline[i]}),
				.result(bln_mux_data[i]));

			baseline baseline_unit (
				.clk(sys_clk),
				.reset(bln_reset[i]),
				.data_ready(data_ready[i]),
				.uwt_flag(uwt_flag3[i]),
				.uwt_data(uwt_peak3[i]),
				.baseline(bln_baseline[i]));

			analyser analyser_unit (
				.clk(sys_clk),
				.reset(ana_reset[i]),
				.data_ready(data_ready[i]),
				.uwt_flag(uwt_flag3[i]),
				.peak_ready(ana_peak_ready[i]),
				.peak_debug(ana_peak_debug[i]));

			suppression suppression_unit (
				.clk(sys_clk),
				.data(hst_mux_data[i][12:1]),
				.baseline(bln_mux_data[i]),
				.result(hst_data[i]));

			assign hst_data_ready[i] = (hst_mux_data[i][0]) & (hst_data[i] >= cfg_hst_threshold[i]);

			histogram #(.W(32)) histogram_unit (
				.clk(sys_clk),
				.reset(hst_reset[i]),
				.data_ready(hst_data_ready[i]),
				.data(hst_data[i]),
				.address(hst_addr[i]),
				.q(hst_q[i]));

			trigger trigger_unit (
				.clk(sys_clk),
				.reset(trg_reset[i]),
				.data_ready(data_ready[i]),
				.data(trg_mux_data[i]),
				.threshold(cfg_trg_threshold[i]),
				.trigger(osc_trig[i]));

			
			oscilloscope oscilloscope_unit (
				.clk(sys_clk),
				.reset(osc_reset[i]),
				.data_ready(data_ready[i]),
				.data(osc_mux_data[i]),
				.trigger(osc_trig[i]),
				.address(osc_addr[i]),
				.start_address(osc_start_addr[i]),
				.q(osc_q[i]));
		end
	endgenerate

	always @*
	begin
		for (j = 0; j < N; j = j + 1)
		begin
			osc_reset[j] = 1'b0;
			osc_addr[j] = 10'b0;
			hst_reset[j] = 1'b0;
			hst_addr[j] = 12'b0;
		end

		case(mux_type)
//		case({mux_type, mux_chan})
			1'b0:
//			3'b000, 3'b001, 3'b010, 3'b011:
			begin
				osc_reset[mux_chan] = mux_reset;
				osc_addr[mux_chan] = mux_addr[9:0];
				mux_max_byte = 2'd1;	
				mux_min_addr = {6'd0, osc_start_addr[mux_chan]};
				mux_max_addr = 16'd1023;
			end

			1'b1:
//			3'b100, 3'b101, 3'b110, 3'b011:
			begin
				hst_reset[mux_chan] = mux_reset;
				hst_addr[mux_chan] = mux_addr[11:0];
				mux_max_byte = 2'd3;	
				mux_min_addr = 16'd0;
				mux_max_addr = 16'd4095;
			end
		endcase
	end
	
	always @*
	begin
		case ({mux_type, mux_byte})
			3'b000: mux_q = osc_q[mux_chan][7:0];
			3'b001: mux_q = osc_q[mux_chan][15:8];

			3'b100: mux_q = hst_q[mux_chan][7:0];
			3'b101: mux_q = hst_q[mux_chan][15:8];
			3'b110: mux_q = hst_q[mux_chan][23:16];
			3'b111: mux_q = hst_q[mux_chan][31:24];

			default: mux_q = 8'd0;
		endcase     
	end

	wire			i2c_aclr;
	wire			i2c_wrreq;
	wire			i2c_full;
	wire 	[15:0]	i2c_data;

	i2c_fifo i2c_unit(
		.clk(sys_clk),
		.aclr(i2c_aclr),
		.wrreq(i2c_wrreq),
		.data(i2c_data),
		.full(i2c_full),
/*
		normal connection
		.i2c_sda(I2C_SDA),
		.i2c_scl(I2C_SCL),

		following is a cross wire connection for EPT
*/
		.i2c_sda(I2C_SCL),
		.i2c_scl(I2C_SDA));

	control control_unit (
		.clk(sys_clk),
		.cfg_reset(cfg_reset),
		.cfg_src_data(cfg_memory[cfg_src_addr[4:0]]),
		.cfg_src_addr(cfg_src_addr),
		.cfg_dst_data(cfg_dst_data),
		.cfg_dst_addr(cfg_dst_addr),
		.rx_empty(usb_rx_empty),
		.tx_full(usb_tx_full),
		.rx_data(usb_rx_data),
		.mux_max_byte(mux_max_byte),
		.mux_min_addr(mux_min_addr),
		.mux_max_addr(mux_max_addr),
		.mux_q(mux_q),
		.mux_reset(mux_reset),
		.mux_type(mux_type),
		.mux_chan(mux_chan),
		.mux_byte(mux_byte),
		.mux_addr(mux_addr),
		.rx_rdreq(usb_rx_rdreq),
		.tx_wrreq(usb_tx_wrreq),
		.tx_data(usb_tx_data),
		.ram_we(RAM_WE),
		.ram_addr(RAM_ADDR),
		.ram_data({RAM_DQA, RAM_DQAP, RAM_DQB, RAM_DQBP}),
		.ept_data_ready(ept_data_ready),
		.ept_data(ept_data),
		.i2c_wrreq(i2c_wrreq),
		.i2c_data(i2c_data),
		.i2c_full(i2c_full),
		.led(LED));

/*
	altserial_flash_loader #(
		.enable_shared_access("OFF"),
		.enhanced_mode(1),
		.intended_device_family("Cyclone III")) sfl_unit (
		.noe(1'b0),
		.asmi_access_granted(),
		.asmi_access_request(),
		.data0out(),
		.dclkin(),
		.scein(),
		.sdoin());
*/

endmodule
