module usb_fifo
	(
		input	wire			usb_clk,
		inout	wire	[7:0]	usb_data,
		input	wire			usb_full, usb_empty,
		output	wire			usb_wrreq, usb_rdreq, usb_rden, usb_pktend,
		output	wire	[1:0]	usb_addr,
		
		input	wire			clk, aclr,
		input	wire			tx_wrreq, rx_rdreq,
		input	wire	[7:0]	tx_data,
		output	wire			tx_full, rx_empty,
		output	wire	[7:0]	rx_data
	);

	// bidirectional data bus
	wire			usb_wren;
	wire	[7:0]	usb_datain = usb_data;
	wire	[7:0]	usb_dataout;

	assign	usb_data = usb_wren ? usb_dataout : 8'bz;

	wire			rx_full, tx_empty;
	reg		[8:0]	byte_counter;
	reg		[4:0]	idle_counter;

	fifo32x8 fifo_tx_unit (
		.aclr(aclr),
		.data(tx_data),
		.rdclk(usb_clk),
		.rdreq(usb_wrreq),
		.wrclk(clk),
		.wrreq(tx_wrreq),
		.q(usb_dataout),
		.rdempty(tx_empty),
		.wrfull(tx_full));

	fifo32x8 fifo_rx_unit (
		.aclr(aclr),
		.data(usb_datain),
		.rdclk(clk),
		.rdreq(rx_rdreq),
		.wrclk(usb_clk),
		.wrreq(usb_rdreq),
		.q(rx_data),
		.rdempty(rx_empty),
		.wrfull(rx_full));
		
	always @ (posedge usb_clk)
	begin
		if (usb_pktend)
		begin
			byte_counter <= 9'd0;
			idle_counter <= 5'd0;
		end
		else if (usb_wrreq)
		begin
			byte_counter <= byte_counter + 9'd1;
			idle_counter <= 5'd0;
		end
		else if ((|byte_counter) & (tx_empty))
		begin
			byte_counter <= byte_counter;
			idle_counter <= idle_counter + 5'd1;
		end
	end

	assign	usb_pktend = (&idle_counter);
//	assign	usb_pktend = 1'b0;
	assign	usb_rdreq = (~usb_empty) & (~rx_full);
	assign	usb_wrreq = (~usb_rdreq) & (~usb_full) & (~tx_empty);
	assign	usb_rden = usb_rdreq;
	assign	usb_wren = usb_wrreq;
	assign	usb_addr = usb_empty ? 2'b11 : 2'b10; 

endmodule
