module classifier
	#(
		parameter	width	=	12 // bit width of the input data (unsigned)
	)
	(
		input	wire					clock, frame, reset,
		input	wire	[22*width-1:0]	cfg_data,
		input	wire	[6*width-1:0]	inp_data, // {D3, D2, D1, S2, S1_S, S1_F}
		input	wire	[5:0]			inp_flag,
		output	wire	[7:0]			out_data,
		output	wire					out_flag
	);

	reg					out_flag_reg [2:0], out_flag_next [2:0];
	reg		[10:0]		out_data_reg [2:0], out_data_next [2:0];
	reg		[5:0]		inp_flag_reg, inp_flag_next;
	reg		[width-1:0]	inp_data_reg [5:0], inp_data_next [5:0];
	reg		[15:0]		int_pipe_reg [23:0], int_pipe_next [23:0];
	reg		[2:0]		int_data_reg [3:0], int_data_next [3:0];
	reg		[4:0]		int_temp_reg [1:0], int_temp_next [1:0];

	wire	[width-1:0]	inp_data_wire [5:0];
	wire	[3:0]		int_pipe_wire [5:0];
	wire	[19:0]		int_comp_wire;

	integer i;
	genvar j;

	generate
		for (j = 0; j < 6; j = j + 1)
		begin : CLASSIFIER_INPUT_DATA
			assign inp_data_wire[j] = inp_data[j*width+width-1:j*width];
		end
	endgenerate

	generate
		assign int_comp_wire[0] = inp_flag_reg[0] & (inp_data_reg[0] < cfg_data[width-1:0]);
		assign int_comp_wire[1] = inp_flag_reg[0] & (inp_data_reg[0] > cfg_data[2*width-1:1*width]) & (inp_data_reg[0] < cfg_data[3*width-1:2*width]);

		assign int_comp_wire[2] = inp_flag_reg[1] & (inp_data_reg[1] < cfg_data[4*width-1:3*width]);
		assign int_comp_wire[3] = inp_flag_reg[1] & (inp_data_reg[1] > cfg_data[5*width-1:4*width]) & (inp_data_reg[1] < cfg_data[6*width-1:5*width]);

		for (j = 0; j < 4; j = j + 1)
		begin : CLASSIFIER_COMPARTORS
			assign int_comp_wire[j*4+0+4] = (inp_data_reg[j+2] > cfg_data[(j*4+0+6)*width+width-1:(j*4+0+6)*width]);
			assign int_comp_wire[j*4+1+4] = (inp_data_reg[j+2] > cfg_data[(j*4+1+6)*width+width-1:(j*4+1+6)*width]);
			assign int_comp_wire[j*4+2+4] = (inp_data_reg[j+2] > cfg_data[(j*4+2+6)*width+width-1:(j*4+2+6)*width]);
			assign int_comp_wire[j*4+3+4] = (inp_data_reg[j+2] > cfg_data[(j*4+3+6)*width+width-1:(j*4+3+6)*width]);
		end
	endgenerate

	generate
		for (j = 0; j < 4; j = j + 1)
		begin : CLASSIFIER_PIPELINE
		    assign int_pipe_wire[0][j] = (|int_pipe_reg[j]);
		    assign int_pipe_wire[1][j] = (|int_pipe_reg[j+4]);
			assign int_pipe_wire[j+2][0] = (|int_pipe_reg[j*4+0+8]);
			assign int_pipe_wire[j+2][1] = (|int_pipe_reg[j*4+1+8]);
			assign int_pipe_wire[j+2][2] = (|int_pipe_reg[j*4+2+8]);
			assign int_pipe_wire[j+2][3] = (|int_pipe_reg[j*4+3+8]);
		end
	endgenerate

	always @(posedge clock)
	begin
		if (reset)
		begin
			inp_flag_reg <= {(6){1'b0}};
			for (i = 0; i < 3; i = i + 1)
			begin
				out_data_reg[i] <= {(11){1'b0}};
				out_flag_reg[i] <= 1'b0;
			end
			for (i = 0; i < 6; i = i + 1)
			begin
				inp_data_reg[i] <= {(width){1'b0}};
			end
			for (i = 0; i < 24; i = i + 1)
			begin
				int_pipe_reg[i] <= {(16){1'b0}};
			end
			for (i = 0; i < 4; i = i + 1)
			begin
				int_data_reg[i] <= {(3){1'b0}};
			end
			for (i = 0; i < 2; i = i + 1)
			begin
				int_temp_reg[i] <= {(5){1'b0}};
			end
		end
		else
		begin
			inp_flag_reg <= inp_flag_next;
			for (i = 0; i < 3; i = i + 1)
			begin
				out_data_reg[i] <= out_data_next[i];
				out_flag_reg[i] <= out_flag_next[i];
			end
			for (i = 0; i < 6; i = i + 1)
			begin
				inp_data_reg[i] <= inp_data_next[i];
			end
			for (i = 0; i < 24; i = i + 1)
			begin
				int_pipe_reg[i] <= int_pipe_next[i];
			end
			for (i = 0; i < 4; i = i + 1)
			begin
				int_data_reg[i] <= int_data_next[i];
			end
			for (i = 0; i < 2; i = i + 1)
			begin
				int_temp_reg[i] <= int_temp_next[i];
			end
		end
	end

	always @*
	begin
		inp_flag_next = inp_flag_reg;
		for (i = 0; i < 3; i = i + 1)
		begin
			out_data_next[i] = out_data_reg[i];
			out_flag_next[i] = out_flag_reg[i];
		end
		for (i = 0; i < 6; i = i + 1)
		begin
			inp_data_next[i] = inp_data_reg[i];
		end
		for (i = 0; i < 24; i = i + 1)
		begin
			int_pipe_next[i] = int_pipe_reg[i];
		end
		for (i = 0; i < 4; i = i + 1)
		begin
			int_data_next[i] = int_data_reg[i];
		end
		for (i = 0; i < 2; i = i + 1)
		begin
			int_temp_next[i] = int_temp_reg[i];
		end

		if (frame)
		begin
			inp_flag_next = inp_flag;
			for (i = 0; i < 6; i = i + 1)
			begin
				inp_data_next[i] = inp_data_wire[i];
			end

			if (out_flag_reg[2])
			begin
				for (i = 0; i < 3; i = i + 1)
				begin
					out_data_next[i] = {(11){1'b0}};
					out_flag_next[i] = 1'b0;
				end
				for (i = 0; i < 24; i = i + 1)
				begin
					int_pipe_next[i] = {(16){1'b0}};
				end
				int_temp_next[0] = {(5){1'b0}};
				int_temp_next[1] = {(5){1'b0}};
			end
			else
			begin
				out_data_next[0] = {(11){1'b0}};		
				out_data_next[1] = out_data_reg[0];
				out_data_next[2] = {3'd0, out_data_reg[1][10:5], 2'd0} + {6'd0, out_data_reg[1][4:0]};

				out_flag_next[0] = 1'b1;				
				out_flag_next[1] = out_flag_reg[0];
				out_flag_next[2] = out_flag_reg[1] & (out_data_reg[1] > out_data_reg[0]);

				for (i = 0; i < 4; i = i + 1)
				begin
					int_pipe_next[i] = {int_pipe_reg[i][14:0], int_comp_wire[i]};
				end
				for (i = 4; i < 8; i = i + 1)
				begin
					int_pipe_next[i] = {int_pipe_reg[i][14:0], inp_flag_reg[i-2]};
				end
				for (i = 8; i < 24; i = i + 1)
				begin
					int_pipe_next[i] = {int_pipe_reg[i][14:0], int_comp_wire[i-4]};
				end

				for (i = 0; i < 4; i = i + 1)
				begin
					case (int_pipe_wire[i+2][3:0])
						4'b0000: int_data_next[i] = 3'b000;
						4'b0001: int_data_next[i] = 3'b001;
						4'b0011: int_data_next[i] = 3'b010;
						4'b0111: int_data_next[i] = 3'b011;
						4'b1111: int_data_next[i] = 3'b100;
						default: int_data_next[i] = 3'b000;
					endcase
				end

				int_temp_next[0] = {int_pipe_wire[1], ^int_pipe_wire[0]};
				int_temp_next[1] = {1'b0, int_pipe_wire[0]};

				case (int_temp_reg[0][4:0])
					5'b00011: out_data_next[0][4:0] = int_data_next[0];
					5'b00111: out_data_next[0][4:0] = int_data_next[1] + 4'd5;
					5'b01111: out_data_next[0][4:0] = int_data_next[2] + 4'd10;
					5'b11111: out_data_next[0][4:0] = int_data_next[3] + 4'd15;
					default: out_flag_next[0] = 1'b0;
				endcase

				case (int_temp_reg[1][3:0])
					// S1_F, electron
					4'b0001: out_data_next[0][10:5] = 6'b100000;

					// S1_F, proton
					4'b0010: out_data_next[0][10:5] = 6'b100101;

					// S1_S, electron
					4'b0100: out_data_next[0][10:5] = 6'b101010;

					// S1_S, proton
					4'b1000: out_data_next[0][10:5] = 6'b101111;

					default: out_flag_next[0] = 1'b0;
				endcase
			end
		end
	end

//	assign out_data = {1'd0, int_pipe_wire[1+2][2:0], int_data_reg[1]};
//	assign out_data = {1'd0, int_comp_wire[0], int_temp_reg[1][3:0]};
//	assign out_data = {1'd0, int_temp_reg[0][4:0]};
	assign out_data = out_data_reg[2][7:0];
	assign out_flag = out_flag_reg[2];

endmodule
