
/*
  Copyright (c) 2007, Pavel Demin

  All rights reserved.

  Redistribution and use in source and binary forms,
  with or without modification, are permitted
  provided that the following conditions are met:

      * Redistributions of source code must retain
        the above copyright notice, this list of conditions
        and the following disclaimer.
      * Redistributions in binary form must reproduce
        the above copyright notice, this list of conditions
        and the following disclaimer in the documentation
        and/or other materials provided with the distribution.
      * Neither the name of the SRMlite nor the names of its
        contributors may be used to endorse or promote products
        derived from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <string.h>
#include <stdlib.h>
#include <memory.h>
#include <stdio.h>
#include <math.h>
#include <tcl.h>

#define SQRT_2 1.4142135623730951

static double bior_3_1_hi[4] = {-0.25, -0.75, 0.75, 0.25};
static double bior_3_1_lo[4] = {0.125, 0.375, 0.375, 0.125};

/*
static double bior_3_1_hi[4] = {-1.0, -3.0, 3.0, 1.0};
static double bior_3_1_lo[4] = {1.0, 3.0, 3.0, 1.0};
*/

static void uwt_forward(double *input, double *d, double *a, int n, int level,
                        double *hi, double *lo, int l)
{
    int i, j, k;
    double entry;

    for(i = 0; i < n; ++i)
    {
        k = i + (((l - 1) << (level - 1)) + 1)/2;
        entry = (k >= 0 && k < n) ? input[k] : 0.0;
        d[i] = hi[0] * entry;
        a[i] = lo[0] * entry;
/*
        printf("%d %d -> %g * %g -> %g\n", i, 0, hi[0], entry, d[i]);
*/
        for(j = 1; j < l; ++j)
        {
          k -= (1 << (level - 1));
          entry = (k >= 0 && k < n) ? input[k] : 0.0;
          d[i] += hi[j] * entry;
          a[i] += lo[j] * entry;
/*
          printf("%d %d -> + %g * %g -> %g\n", i, j, hi[j], entry, d[i]);
*/
        }
    }
}

/* ----------------------------------------------------------------- */

static int UwtObjCmdProc(ClientData clientData, Tcl_Interp *interp,
                         int objc, Tcl_Obj *CONST objv[])
{
  int shift = 0;
  int size = 0;
  int i = 0;
  int level = 0;
  double value = 0.0;
  double *tmp, *input, *d, *a;
  Tcl_Obj *result = NULL;
  Tcl_Obj *lista = NULL;
  Tcl_Obj *listd = NULL;

  if(objc != 3)
  {
    Tcl_WrongNumArgs(interp, 1, objv, "level data");
		return TCL_ERROR;
  }

  if(TCL_OK != Tcl_GetIntFromObj(interp, objv[1], &level))
  {
    return TCL_ERROR;
  }

  if(TCL_OK != Tcl_ListObjLength(interp, objv[2], &size))
  {
    return TCL_ERROR;
  }

  input = (double *) malloc(size*sizeof(double));
  d = (double *) malloc(size*sizeof(double));
  a = (double *) malloc(size*sizeof(double));

  for(i = 0; i < size; ++i)
  {
    Tcl_ListObjIndex(interp, objv[2], i, &result);
    if(TCL_OK != Tcl_GetDoubleFromObj(interp, result, &value))
    {
      free(input);
      free(d);
      free(a);
      return TCL_ERROR;
    }
    input[i] = value;
/*
    printf("%d -> %g\n", i, input[i]);
*/
  }

  for(i = 1; i <= level; ++i)
  {
    uwt_forward(input, d, a, size, i, bior_3_1_hi, bior_3_1_lo, 4);
    tmp = input;
    input = a;
    a = tmp;
  }

  result = Tcl_GetObjResult(interp);
  lista = Tcl_NewObj();
  listd = Tcl_NewObj();
  for(i = 0; i < size; ++i)
  {
    Tcl_ListObjAppendElement(interp, listd, Tcl_NewDoubleObj(d[i]));
    Tcl_ListObjAppendElement(interp, lista, Tcl_NewDoubleObj(input[i]));
/*
    printf("%d -> %g\n", i, d[i]);
*/	}

  Tcl_ListObjAppendElement(interp, result, listd);
  Tcl_ListObjAppendElement(interp, result, lista);

  free(input);
  free(d);
  free(a);

  return TCL_OK;
}

/* ----------------------------------------------------------------- */

static void sum8(double *input, double *output, int n)
{
    int i, j, k;
    double entry;

    for(i = 0; i < n; ++i)
    {
        k = i + 4;
        entry = (k >= 0 && k < n) ? input[k] : 0.0;
        output[i] = entry;
        for(j = 1; j < 8; ++j)
        {
          --k;
          entry = (k >= 0 && k < n) ? input[k] : 0.0;
          output[i] += entry;
        }
        
        output[i] = output[i]/8.0;

    }
}

/* ----------------------------------------------------------------- */

static int Sum8ObjCmdProc(ClientData clientData, Tcl_Interp *interp,
                          int objc, Tcl_Obj *CONST objv[])
{
  int size = 0;
  int i = 0;
  int level = 0;
  double value = 0.0;
  double *input, *output;
  Tcl_Obj *result = NULL;

  if(objc != 2)
  {
    Tcl_WrongNumArgs(interp, 1, objv, "data");
		return TCL_ERROR;
  }

  if(TCL_OK != Tcl_ListObjLength(interp, objv[1], &size))
  {
    return TCL_ERROR;
  }

  input = (double *) malloc(size*sizeof(double));
  output = (double *) malloc(size*sizeof(double));

  for(i = 0; i < size; ++i)
  {
    Tcl_ListObjIndex(interp, objv[1], i, &result);
    if(TCL_OK != Tcl_GetDoubleFromObj(interp, result, &value))
    {
      free(input);
      free(output);
      return TCL_ERROR;
    }
    input[i] = value;
/*
    printf("%d -> %g\n", i, input[i]);
*/
  }

  sum8(input, output, size);

  result = Tcl_GetObjResult(interp);
  for(i = 0; i < size; ++i)
  {
    Tcl_ListObjAppendElement(interp, result, Tcl_NewDoubleObj(output[i]));
/*
    printf("%d -> %g\n", i, d[i]);
*/
	}

  free(input);
  free(output);

  return TCL_OK;
}

/* ----------------------------------------------------------------- */

int Swt_Init(Tcl_Interp *interp)
{
    Tcl_CreateObjCommand(interp, "uwt", UwtObjCmdProc, 0, 0);
    Tcl_CreateObjCommand(interp, "sum8", Sum8ObjCmdProc, 0, 0);

    return Tcl_PkgProvide(interp, "swt", "0.1");
}
