module Paella
	(
		input	wire			CLK_50MHz,
		output	wire			LED,

		inout	wire	[3:0]	TRG,
		inout	wire			I2C_SDA,
		inout	wire			I2C_SCL,
		inout	wire	[4:0]	CON_A,
		input	wire	[15:0]	CON_B,
		input	wire	[12:0]	CON_C,
		input	wire	[1:0]	CON_BCLK,
		input	wire	[1:0]	CON_CCLK,

		input	wire			ADC_DCO,
		input	wire			ADC_FCO,
		input	wire	[2:0]	ADC_D,

		output	wire			USB_SLRD, 
		output	wire			USB_SLWR,
		input	wire			USB_IFCLK,
		input	wire			USB_FLAGA, // EMPTY flag for EP6
		input	wire			USB_FLAGB, // FULL flag for EP8
		input	wire			USB_FLAGC,
		inout	wire			USB_PA0,
		inout	wire			USB_PA1,
		output	wire			USB_PA2,
		inout	wire			USB_PA3,
		output	wire			USB_PA4,
		output	wire			USB_PA5,
		output	wire			USB_PA6,
		inout	wire			USB_PA7,
		inout	wire	[7:0]	USB_PB,

		output	wire			RAM_CLK,
		output	wire			RAM_CE1,
		output	wire			RAM_WE,
		output	wire	[19:0]	RAM_ADDR,
		inout	wire			RAM_DQAP,
		inout	wire	[7:0]	RAM_DQA,
		inout	wire			RAM_DQBP,
		inout	wire	[7:0]	RAM_DQB
	);

	localparam	N		=	3;

	//	Turn output ports off
/*
	assign	RAM_CLK		=	1'b0;
	assign	RAM_CE1		=	1'b0;
	assign	RAM_WE		=	1'b0;
	assign	RAM_ADDR	=	20'h00000;
*/
	assign	RAM_CLK = sys_clock;
	assign	RAM_CE1 = 1'b0;

	//	Turn inout ports to tri-state
	assign	TRG			=	4'bz;
	assign	CON_A		=	5'bz;
	assign	USB_PA0		=	1'bz;
	assign	USB_PA1		=	1'bz;
	assign	USB_PA3		=	1'bz;
	assign	USB_PA7		=	1'bz;
//	assign	RAM_DQAP	=	1'bz;
//	assign	RAM_DQA		=	8'bz;
//	assign	RAM_DQBP	=	1'bz;
//	assign	RAM_DQB		=	8'bz;

	assign	USB_PA2		=	~usb_rden;
	assign	USB_PA4		=	usb_addr[0];
	assign	USB_PA5		=	usb_addr[1];
	assign	USB_PA6		=	~usb_pktend;

	wire			usb_wrreq, usb_rdreq, usb_rden, usb_pktend;
	wire			usb_tx_wrreq, usb_rx_rdreq;
	wire			usb_tx_full, usb_rx_empty;
	wire	[7:0]	usb_tx_data, usb_rx_data;
	wire	[1:0]	usb_addr;

	assign	USB_SLRD = ~usb_rdreq;
	assign	USB_SLWR = ~usb_wrreq;

	usb_fifo usb_unit
	(
		.usb_clock(USB_IFCLK),
		.usb_data(USB_PB),
		.usb_full(~USB_FLAGB),
		.usb_empty(~USB_FLAGA),
		.usb_wrreq(usb_wrreq),
		.usb_rdreq(usb_rdreq),
		.usb_rden(usb_rden),
		.usb_pktend(usb_pktend),
		.usb_addr(usb_addr),

		.clock(sys_clock),

		.tx_full(usb_tx_full),
		.tx_wrreq(usb_tx_wrreq),
		.tx_data(usb_tx_data),

		.rx_empty(usb_rx_empty),
		.rx_rdreq(usb_rx_rdreq),
		.rx_q(usb_rx_data)
	);
		
	wire	[11:0]	osc_mux_data [4:0];

	wire	[11:0]	trg_mux_data;
	wire			trg_flag;

	wire	[2:0]	coi_data;
	wire			coi_flag;

	wire	[7*12-1:0]	int_mux_data [N-1:0];

	wire			ana_dead [N-1:0];
	wire			ana_good [N-1:0];
	wire	[11:0]	ana_data [N-1:0];
	wire	[11:0]	ana_base [N-1:0];

	wire			amp_flag [N-1:0];
	wire	[11:0]	amp_data [N-1:0];

	wire			cnt_good [N-1:0];
	wire	[15:0]	cnt_bits_wire;

	wire			sys_clock, sys_frame;

    wire	[11:0]	adc_data [N-1:0];
    wire	[11:0]	sys_data [N-1:0];
	wire	[11:0]	tst_data;

    wire	[11:0]	cmp_data;
    wire	[11:0]	del_data;

	wire 	[15:0]	uwt_data1 [N-1:0];
	wire 	[18:0]	uwt_data2 [N-1:0];
	wire 	[1:0]	uwt_flag1 [N-1:0];
	wire 	[1:0]	uwt_flag2 [N-1:0];

	wire	[27:0]	cic_data1 [N-1:0];
	wire	[27:0]	cic_data2 [N-1:0];
	wire	[27:0]	cic_data3 [N-1:0];

	wire	[11:0]	dec_data [N-1:0];

	wire	[1:0]	ext_flag [N-1:0];

	wire			i2c_reset;

	sys_pll sys_pll_unit(
		.inclk0(CLK_50MHz),
		.c0(sys_clock));

	test test_unit(
		.clock(ADC_FCO),
		.data(tst_data));

	adc_lvds #(
		.size(3),
		.width(12)) adc_lvds_unit (
		.clock(sys_clock),
		.lvds_dco(ADC_DCO),
		.lvds_fco(ADC_FCO),
		.lvds_d(ADC_D),
		.test(tst_data),
		.trig({CON_B[9:0], TRG[1:0]}),
		.adc_frame(sys_frame),
		.adc_data({cmp_data, adc_data[2], adc_data[1], adc_data[0]}));

	wire	[15:0]	cfg_bits [31:0];
	wire	[511:0]	int_cfg_bits;

	wire	[39:0]	cfg_mux_selector;

	wire 			cfg_reset;

	wire 	[11:0]	bus_ssel;
	wire			bus_wren;
	wire	[31:0]	bus_addr;
	wire	[15:0]	bus_mosi;
	wire 	[15:0]	bus_miso [10:0];
	wire 	[11:0]	bus_busy;

	wire 	[15:0]	mrg_bus_miso;
	wire 			mrg_bus_busy;

	wire 	[11*16-1:0]	int_bus_miso;

	genvar j;

	generate
		for (j = 0; j < 32; j = j + 1)
		begin : CONFIGURATION_OUTPUT
			assign cfg_bits[j] = int_cfg_bits[j*16+15:j*16];
		end
	endgenerate

	configuration configuration_unit (
		.clock(sys_clock),
		.reset(cfg_reset),
		.bus_ssel(bus_ssel[0]),
		.bus_wren(bus_wren),
		.bus_addr(bus_addr[4:0]),
		.bus_mosi(bus_mosi),
		.bus_miso(bus_miso[0]),
		.bus_busy(bus_busy[0]),
		.cfg_bits(int_cfg_bits));

	generate
		for (j = 0; j < 3; j = j + 1)
		begin : MUX_DATA
			assign int_mux_data[j] = {
//				{4'd0, amp_flag[j], 7'd0},
//				dec_data[j][37:26],
//				dec_data[j][36:25],
//				dec_data[j][35:24],
				dec_data[j][11:0],
				dec_data[j][11:0],
				amp_data[j][11:0],
				{ext_flag[j][1], 11'd0},
				{ext_flag[j][0], 11'd0},
				cic_data3[j][26:15],
				sys_data[j]};
		end
	endgenerate

	assign cfg_mux_selector = {cfg_bits[4][7:0], cfg_bits[3], cfg_bits[2]};

	lpm_mux #(
		.lpm_size(7*3),
		.lpm_type("LPM_MUX"),
		.lpm_width(12),
		.lpm_widths(5)) trg_mux_unit (
		.sel(cfg_bits[4][12:8]),
		.data({int_mux_data[2], int_mux_data[1], int_mux_data[0]}),
		.result(trg_mux_data));

	generate
		for (j = 0; j < 5; j = j + 1)
		begin : OSC_CHAIN
		
			lpm_mux #(
				.lpm_size(7*3),
				.lpm_type("LPM_MUX"),
				.lpm_width(12),
				.lpm_widths(5)) osc_mux_unit (
				.sel(cfg_mux_selector[j*8+4:j*8]),
				.data({int_mux_data[2], int_mux_data[1], int_mux_data[0]}),
				.result(osc_mux_data[j]));
		end
	endgenerate

	trigger trigger_unit (
		.clock(sys_clock),
		.frame(sys_frame),
		.reset(cfg_bits[0][0]),
		.cfg_data(cfg_bits[5][11:0]),
		.trg_data(trg_mux_data),
		.trg_flag(trg_flag));
	
	oscilloscope oscilloscope_unit (
		.clock(sys_clock),
		.frame(sys_frame),
		.reset(cfg_bits[0][1]),
		.cfg_data(cfg_bits[5][12]),
		.trg_flag(trg_flag),
		.osc_data({cmp_data[3:0], osc_mux_data[4], osc_mux_data[3], osc_mux_data[2], osc_mux_data[1], osc_mux_data[0]}),
		.ram_wren(RAM_WE),
		.ram_addr(RAM_ADDR),
		.ram_data({RAM_DQA, RAM_DQAP, RAM_DQB, RAM_DQBP}),
		.bus_ssel(bus_ssel[1]),
		.bus_wren(bus_wren),
		.bus_addr(bus_addr[19:0]),
		.bus_mosi(bus_mosi),
		.bus_miso(bus_miso[1]),
		.bus_busy(bus_busy[1]));

	cic_filter #(.size(3), .width(12)) cic_filter_unit (
		.clock(sys_clock),
		.frame(sys_frame),
		.reset(1'b0),
		.inp_data({sys_data[2], sys_data[1], sys_data[0]}),
		.out_data2({cic_data2[2], cic_data2[1], cic_data2[0]}),
		.out_data3({cic_data3[2], cic_data3[1], cic_data3[0]}),
		.out_data({cic_data1[2], cic_data1[1], cic_data1[0]}));
	
	deconv #(.size(1), .shift(25), .width(27), .widthr(12)) deconv_unit (
		.clock(sys_clock),
		.frame(sys_frame),
		.reset(1'b0),
		.del_data({6'd20, 6'd20, 6'd20}),
		.amp_data({8'd1, 8'd1, 8'd1}),
		.tau_data({16'd980, 16'd980, 16'd980}),
//		.del_data({cfg_bits[16][5:0], cfg_bits[15][13:8], cfg_bits[15][5:0]}),
//		.amp_data({cfg_bits[18][7:0], cfg_bits[17][15:8], cfg_bits[17][7:0]}),
//		.tau_data({cfg_bits[21], cfg_bits[20], cfg_bits[19]}),
		.inp_data({cic_data3[2][26:0], cic_data3[1][26:0], cic_data3[0][26:0]}),
		.out_data({dec_data[2], dec_data[1], dec_data[0]}));


	generate
		for (j = 0; j < 3; j = j + 1)
		begin : MCA_CHAIN

			assign sys_data[j] = (cfg_bits[1][4*j]) ? (adc_data[j] ^ 12'hfff) : (adc_data[j]);
/*
			uwt_bior31 #(.level(1), .width(13)) uwt_1_unit (
				.clock(sys_clock),
				.frame(sys_frame),
				.reset(1'b0),
				.inp_data(dec_data[j][12:0]),
				.out_data(uwt_data1[j]),
				.out_flag(uwt_flag1[j]));
		
			uwt_bior31 #(.level(1), .width(16)) uwt_2_unit (
				.clock(sys_clock),
				.frame(sys_frame),
				.reset(1'b0),
				.inp_data(uwt_data1[j]),
				.out_data(uwt_data2[j]),
				.out_flag(uwt_flag2[j]));
*/		
			extrema #(.width(12)) extrema_unit (
				.clock(sys_clock),
				.frame(sys_frame),
				.reset(1'b0),
//				.inp_data(cic_data3[j][26:15]),
				.inp_data(dec_data[j]),
				.out_flag(ext_flag[j]));

			analyser analyser_unit (
				.clock(sys_clock),
				.frame(sys_frame),
				.reset(cfg_bits[0][2+j]),
				.cfg_data({cfg_bits[7+2*j][12:0], cfg_bits[6+2*j][11:0]}),
				.uwt_flag(uwt_flag2[j]),
				.uwt_data(uwt_data2[j][18:7]),
				.ana_dead(ana_dead[j]),
				.ana_good(ana_good[j]),
				.ana_data(ana_data[j]),
				.ana_base(ana_base[j]));

			amplitude #(.width(12)) amplitude_unit (
				.clock(sys_clock),
				.frame(sys_frame),
				.reset(cfg_bits[0][2+j]),
//				.cfg_data(cfg_bits[12][11:0]),
				.cfg_data(12'd5),
//				.inp_data(cic_data3[j][22:11]),
				.inp_data(dec_data[j]),
				.out_flag(amp_flag[j]),
				.out_data(amp_data[j]));
		end
	endgenerate

	histogram32 histogram32_unit (
		.clock(sys_clock),
		.frame(sys_frame),
		.reset(cfg_bits[0][5]),
		.hst_good((ana_good[0]) & (cnt_good[0]) & (cfg_bits[13][1])),
		.hst_data(ana_data[0]),
/*
		.hst_good((amp_flag[j]) & (cnt_good[j]) & (cfg_bits[13][1])),
		.hst_data(amp_data[j]),
*/
		.bus_ssel(bus_ssel[2]),
		.bus_wren(bus_wren),
		.bus_addr(bus_addr[12:0]),
		.bus_mosi(bus_mosi),
		.bus_miso(bus_miso[2]),
		.bus_busy(bus_busy[2]));

	counter hst_counter_unit (
		.clock(sys_clock),
		.frame((sys_frame) & (~ana_dead[0])),
//		.frame(sys_frame),
		.reset(cfg_bits[0][8]),
		.setup(cfg_bits[13][0]),
		.count(cfg_bits[13][1]),
		.bus_ssel(bus_ssel[5]),
		.bus_wren(bus_wren),
		.bus_addr(bus_addr[1:0]),
		.bus_mosi(bus_mosi),
		.bus_miso(bus_miso[5]),
		.bus_busy(bus_busy[5]),
		.cnt_good(cnt_good[0]));


	i2c_fifo i2c_unit(
		.clock(sys_clock),
		.reset(i2c_reset),
/*
		normal connection
		.i2c_sda(I2C_SDA),
		.i2c_scl(I2C_SCL),

		following is a cross wire connection for EPT
*/
		.i2c_sda(I2C_SCL),
		.i2c_scl(I2C_SDA),
		
		.bus_ssel(bus_ssel[11]),
		.bus_wren(bus_wren),
		.bus_mosi(bus_mosi),
		.bus_busy(bus_busy[11]));

	generate
		for (j = 0; j < 11; j = j + 1)
		begin : BUS_OUTPUT
			assign int_bus_miso[j*16+15:j*16] = bus_miso[j];
		end
	endgenerate

	lpm_mux #(
		.lpm_size(11),
		.lpm_type("LPM_MUX"),
		.lpm_width(16),
		.lpm_widths(4)) bus_miso_mux_unit (
		.sel(bus_addr[31:28]),
		.data(int_bus_miso),
		.result(mrg_bus_miso));

	lpm_mux #(
		.lpm_size(12),
		.lpm_type("LPM_MUX"),
		.lpm_width(1),
		.lpm_widths(4)) bus_busy_mux_unit (
		.sel(bus_addr[31:28]),
		.data(bus_busy),
		.result(mrg_bus_busy));

	lpm_decode #(
		.lpm_decodes(12),
		.lpm_type("LPM_DECODE"),
		.lpm_width(4)) lpm_decode_unit (
		.data(bus_addr[31:28]),
		.eq(bus_ssel));


	control control_unit (
		.clock(sys_clock),
		.rx_empty(usb_rx_empty),
		.tx_full(usb_tx_full),
		.rx_data(usb_rx_data),
		.rx_rdreq(usb_rx_rdreq),
		.tx_wrreq(usb_tx_wrreq),
		.tx_data(usb_tx_data),
		.bus_wren(bus_wren),
		.bus_addr(bus_addr),
		.bus_mosi(bus_mosi),
		.bus_miso(mrg_bus_miso),
		.bus_busy(mrg_bus_busy),
		.led(LED));

/*
	altserial_flash_loader #(
		.enable_shared_access("OFF"),
		.enhanced_mode(1),
		.intended_device_family("Cyclone III")) sfl_unit (
		.noe(1'b0),
		.asmi_access_granted(),
		.asmi_access_request(),
		.data0out(),
		.dclkin(),
		.scein(),
		.sdoin());
*/

endmodule
