module oscilloscope
	(
		input	wire			clk, reset,
		input	wire			data_ready, trigger,
		input	wire	[15:0]  data,
		input	wire	[9:0]	address,
		output	wire	[9:0]	start_address,
		output	wire	[15:0]  q
	);
	
	// signal declaration
	reg		[3:0]	state_reg, state_next;
	reg				wren_reg, wren_next;
	reg		[9:0]	addr_reg, addr_next;
	reg		[15:0]	data_reg, data_next;

	reg				trig_reg, trig_next;
	reg		[9:0]	trig_addr_reg, trig_addr_next;
	reg		[9:0]	counter_reg, counter_next;

	wire	[15:0]	q_wire;

	altsyncram #(
		.address_reg_b("CLOCK0"),
		.clock_enable_input_a("BYPASS"),
		.clock_enable_input_b("BYPASS"),
		.clock_enable_output_a("BYPASS"),
		.clock_enable_output_b("BYPASS"),
		.intended_device_family("Cyclone III"),
		.lpm_type("altsyncram"),
		.numwords_a(1024),
		.numwords_b(1024),
		.operation_mode("DUAL_PORT"),
		.outdata_aclr_b("NONE"),
		.outdata_reg_b("CLOCK0"),
		.power_up_uninitialized("FALSE"),
		.read_during_write_mode_mixed_ports("OLD_DATA"),
		.widthad_a(10),
		.widthad_b(10),
		.width_a(16),
		.width_b(16),
		.width_byteena_a(1)) osc_ram_unit(
		.wren_a(wren_reg),
		.clock0(clk),
		.address_a(addr_reg),
		.address_b(address),
		.data_a(data_reg),
		.q_b(q_wire),
		.aclr0(1'b0),
		.aclr1(1'b0),
		.addressstall_a(1'b0),
		.addressstall_b(1'b0),
		.byteena_a(1'b1),
		.byteena_b(1'b1),
		.clock1(1'b1),
		.clocken0(1'b1),
		.clocken1(1'b1),
		.clocken2(1'b1),
		.clocken3(1'b1),
		.data_b({16{1'b1}}),
		.eccstatus(),
		.q_a(),
		.rden_a(1'b1),
		.rden_b(1'b1),
		.wren_b(1'b0));

	// body
	always @(posedge clk)
	begin
		if (reset)
        begin
			state_reg <= 4'b1;
			wren_reg <= 1'b1;
			addr_reg <= 10'd0;
			data_reg <= 16'd0;
			trig_reg <= 1'b0;
			trig_addr_reg <= 10'd0;
			counter_reg <= 10'd0;
		end
		else
		begin
			state_reg <= state_next;
			wren_reg <= wren_next;
			addr_reg <= addr_next;
			data_reg <= data_next;
			trig_reg <= trig_next;
			trig_addr_reg <= trig_addr_next;
			counter_reg <= counter_next;
		end
	end

	always @*
	begin
		state_next = state_reg;
		wren_next = wren_reg;
		addr_next = addr_reg;
		data_next = data_reg;
		trig_next = trig_reg;
		trig_addr_next = trig_addr_reg;
		counter_next = counter_reg;

		case (state_reg)
			0:
			begin
				// nothing to do 
				state_next = 4'b0;
				wren_next = 1'b0;
				addr_next = 10'd0;
				data_next = 16'd0;
				counter_next = 10'd0;
			end
			
			1:
			begin
				// write zeros
				if (&addr_reg)
				begin
					wren_next = 1'b0;
					state_next = 4'd2;
				end
				else
				begin
					addr_next = addr_reg + 10'd1;
				end
			end
	
			2:
			begin
				if (data_ready)
				begin
					wren_next = 1'b1;
					data_next = data;
					state_next = 4'd3;
				end
			end

			3:
			begin
				// stop write
				wren_next = 1'b0;
				addr_next = addr_reg + 10'd1;

				if (&counter_reg)
				begin
					state_next = 4'd0;
				end
				else
				begin
					state_next = 4'd2;

					if ((~trig_reg) & (trigger) 
						& (counter_reg == 10'd512))
					begin
						// trigger
						trig_next = 1'b1;
						trig_addr_next = addr_reg;
					end
					
					if (trig_reg | (counter_reg < 10'd512))
					begin
						counter_next = counter_reg + 10'd1;
					end
				end
			end

			default:
			begin
				state_next = 4'b0;
				wren_next = 1'b0;
				addr_next = 10'd0;
				data_next = 16'd0;
				counter_next = 10'd0;
			end
		endcase
	end

	// output logic
	assign	q = q_wire;
	assign	start_address = trig_reg ? (trig_addr_reg ^ 10'h200) + 10'd1: addr_reg + 10'd1;

endmodule
