module histogram
	(
		input	wire			clk, reset,
		input	wire			data_ready,
		input	wire	[11:0]  data, address,
		output	wire	[31:0]  q
	);
	
	// signal declaration
	reg		[3:0]	state_reg, state_next;
	reg				flag_reg, flag_next;
	reg				wren_reg, wren_next;
	reg		[11:0]	addr_reg, addr_next;
	reg		[31:0]	data_reg, data_next;

	wire	[31:0]	q_a_wire, q_b_wire;
	
	wire	[11:0]	addr_wire;
	wire	[31:0]	data_wire;
	
	assign	addr_wire = (flag_reg) ? data : addr_reg;
	assign	data_wire = (flag_reg) ? (q_a_wire + 32'd1) : data_reg;

	altsyncram #(
		.address_reg_b("CLOCK0"),
		.clock_enable_input_a("BYPASS"),
		.clock_enable_input_b("BYPASS"),
		.clock_enable_output_a("BYPASS"),
		.clock_enable_output_b("BYPASS"),
		.indata_reg_b("CLOCK0"),
		.intended_device_family("Cyclone III"),
		.lpm_type("altsyncram"),
		.numwords_a(4096),
		.numwords_b(4096),
		.operation_mode("BIDIR_DUAL_PORT"),
		.outdata_aclr_a("NONE"),
		.outdata_aclr_b("NONE"),
		.outdata_reg_a("UNREGISTERED"),
		.outdata_reg_b("UNREGISTERED"),
		.power_up_uninitialized("FALSE"),
		.read_during_write_mode_mixed_ports("OLD_DATA"),
		.widthad_a(12),
		.widthad_b(12),
		.width_a(32),
		.width_b(32),
		.width_byteena_a(1),
		.width_byteena_b(1),
		.wrcontrol_wraddress_reg_b("CLOCK0")) hst_ram_unit(
		.wren_a(wren_reg),
		.clock0(clk),
		.wren_b(1'b0),
		.address_a(addr_wire),
		.address_b(address),
		.data_a(data_wire),
		.data_b(),
		.q_a(q_a_wire),
		.q_b(q_b_wire),
		.aclr0(1'b0),
		.aclr1(1'b0),
		.addressstall_a(1'b0),
		.addressstall_b(1'b0),
		.byteena_a(1'b1),
		.byteena_b(1'b1),
		.clock1(1'b1),
		.clocken0(1'b1),
		.clocken1(1'b1),
		.clocken2(1'b1),
		.clocken3(1'b1),
		.eccstatus(),
		.rden_a(1'b1),
		.rden_b(1'b1));

	// body
	always @(posedge clk)
	begin
		if (reset)
        begin
			flag_reg <= 1'b0;
			wren_reg <= 1'b1;
			addr_reg <= 12'd0;
			data_reg <= 32'd0;
			state_reg <= 4'b1;
		end
		else
		begin
			flag_reg <= flag_next;
			wren_reg <= wren_next;
			addr_reg <= addr_next;
			data_reg <= data_next;
			state_reg <= state_next;
		end
	end

	always @*
	begin
		flag_next = flag_reg;
		wren_next = wren_reg;
		addr_next = addr_reg;
		data_next = data_reg;
		state_next = state_reg;
		case (state_reg)
			0:
			begin
				// nothing to do
				flag_next = 1'b0;
				wren_next = 1'b0;
				addr_next = 12'd0;
				data_next = 32'd0;
				state_next = 4'd0;
			end
						
			1:
			begin
				// write zeros
				if (&addr_reg)
				begin
					flag_next = 1'b1;
					wren_next = 1'b0;
					state_next = 4'd2;
				end
				else
				begin
					addr_next = addr_reg + 12'd1;
				end
			end	

			2:
			begin
				if (data_ready)
				begin
					if (&q_a_wire)
					begin
						flag_next = 1'b0;
						state_next = 4'd0;
					end
					else
					begin
						wren_next = 1'b1;
						state_next = 4'd3;
					end
				end
			end

			3:
			begin
				wren_next = 1'b0;
				state_next = 4'd2;
			end

			default:
			begin
				flag_next = 1'b0;
				wren_next = 1'b0;
				addr_next = 12'd0;
				data_next = 32'd0;
				state_next = 4'd0;
			end
		endcase
	end

	// output logic
	assign	q = q_b_wire;
endmodule
