module control
	(
		input	wire			clk,
		input	wire			rx_empty, tx_full,
		input	wire	[7:0]	rx_data,
		input	wire	[1:0]	mux_max_byte,
		input	wire	[15:0]	mux_min_addr, mux_max_addr,
		input	wire	[7:0]	mux_q,

		output	wire			mux_reset,
		output	wire			mux_type,
		output	wire	[1:0]	mux_chan,
		output	wire	[1:0]	mux_byte,
		output	wire	[15:0]	mux_addr,
		output	wire			rx_rdreq,
		output	wire			tx_wrreq,
		output	wire	[7:0]	tx_data,
		output	wire			led
	);

	reg		[23:0]	rx_counter;
	reg		[10:0]	tst_counter;	
	reg 	[15:0]	int_addr, int_max_addr;

	reg				int_rdreq, int_wrreq;
	reg				int_type, int_reset;
	reg		[1:0]	int_chan, int_byte, int_max_byte;
	reg		[7:0]	int_data;
	reg				int_led;


	reg		[3:0]	state;

	always @(posedge clk)
	begin
		if (~rx_empty)
		begin
			int_led <= 1'b0;
			rx_counter <= 24'd0;
		end
		else
		begin
			if (&rx_counter)
			begin
				int_led <= 1'b1;
			end
			else
			begin
				rx_counter <= rx_counter + 24'd1;
			end
		end

		case(state)
			1:
			begin
				int_rdreq <= 1'b1;
				int_wrreq <= 1'b0;
				int_type <= 1'b0;
				int_chan <= 2'd0;
				int_byte <= 2'd0;	
				int_reset <= 1'b0;
				state <= 4'd2;
			end

			2: 
			begin
				if (~rx_empty)
				begin
					case (rx_data)
						8'h40, 8'h41, 8'h42, 8'h43, 8'h50, 8'h51, 8'h52, 8'h53:
						begin
							int_rdreq <= 1'b0;
							int_type <= rx_data[4];
							int_chan <= rx_data[1:0];
							int_reset <= 1'b1;
							state <= 4'd1;
						end

						8'h60, 8'h61, 8'h62, 8'h63, 8'h70, 8'h71, 8'h72, 8'h73:
						begin
							int_rdreq <= 1'b0;
							int_type <= rx_data[4];
							int_chan <= rx_data[1:0];
							state <= 4'd3;
						end

						8'h30:
						begin
							int_rdreq <= 1'b0;
							state <= 4'd1;
						end

						8'h31:
						begin
							int_rdreq <= 1'b0;
							tst_counter <= 11'd0;	
							state <= 4'd6;
						end
					endcase
				end
			end
			// mux transfer
			3:
			begin
				int_addr <= mux_min_addr;
				int_max_addr <= mux_min_addr + mux_max_addr;
				int_max_byte <= mux_max_byte;
				int_byte <= 2'd0;	
				state <= 4'd4;
			end
	
			4:
			begin
				int_wrreq <= 1'b0;
				state <= 4'd5;
			end

			5:
			begin
				if (~tx_full)
				begin
					int_data <= mux_q;
					int_wrreq <= 1'b1;
					if ((int_byte == int_max_byte) && (int_addr == int_max_addr))
					begin
						state <= 4'd1;
					end
					else
					begin
						state <= 4'd4;
						if (int_byte == int_max_byte)
						begin
							int_addr <= int_addr + 16'd1;
							int_byte <= 2'd0;
						end
						else
						begin
							int_byte <= int_byte + 2'd1;
						end
					end
				end
			end

			// tst transfer
			6:
			begin
				int_data <= tst_counter;
				int_wrreq <= 1'b1;
				tst_counter <= tst_counter + 11'd1;
				state <= 4'd8;
			end
			7:
			begin
				if (~tx_full)
				begin
					int_data <= tst_counter;
					if (tst_counter == 11'd0)
					begin
						state <= 4'd9;
					end
					else
					begin
						tst_counter <= tst_counter + 11'd1;
					end
				end
			end
			8:
			begin
				if (~tx_full)
				begin
					int_wrreq <= 1'b0;
					state <= 4'd1;
				end
			end
						
			default:
			begin
				state <= 4'd1;
			end
		endcase
	end
	
	assign	mux_reset = int_reset;
	assign	mux_type = int_type;
	assign	mux_chan = int_chan;
	assign	mux_byte = int_byte;
	assign	mux_addr = int_addr;
	assign	rx_rdreq = int_rdreq & (~rx_empty);
	assign	tx_wrreq = int_wrreq & (~tx_full);
	assign	tx_data = int_data;
	assign	led = int_led;

endmodule
