module usb_fifo
	(
		input	wire			usb_clk,
		inout	wire	[7:0]	usb_data,
		input	wire			usb_full, usb_empty,
		output	wire			usb_wrreq, usb_rdreq, usb_rden, usb_pktend,
		output	wire	[1:0]	usb_addr,
		
		input	wire			clk, aclr,
		input	wire			tx_wrreq, rx_rdreq,
		input	wire	[7:0]	tx_data,
		output	wire			tx_full, rx_empty,
		output	wire	[7:0]	rx_data,
		output	wire			led
	);

	// bidirectional data bus
	reg				int_addr, int_wren, int_rden, int_wrreq, int_rdreq;
	wire	[7:0]	int_datain = usb_data;
	wire	[7:0]	int_dataout;

	assign	usb_data = int_wren ? int_dataout : 8'bz;

	wire			rx_full, tx_empty;
	wire			rx_ready, tx_ready;
	reg		[8:0]	byte_counter;
	reg		[4:0]	idle_counter;

	assign	led = ~usb_empty;

	fifo32x8 fifo_tx_unit (
		.aclr(aclr),
		.data(tx_data),
		.rdclk(usb_clk),
		.rdreq(int_wrreq),
		.wrclk(clk),
		.wrreq(tx_wrreq),
		.q(int_dataout),
		.rdempty(tx_empty),
		.wrfull(tx_full));

	fifo32x8 fifo_rx_unit (
		.aclr(aclr),
		.data(int_datain),
		.rdclk(clk),
		.rdreq(rx_rdreq),
		.wrclk(usb_clk),
		.wrreq(int_rdreq),
		.q(rx_data),
		.rdempty(rx_empty),
		.wrfull(rx_full));

	assign	rx_ready = (~usb_empty) & (~rx_full);
	assign	tx_ready = (~usb_full) & (~tx_empty);
		
	always @ (posedge usb_clk)
	begin
		casez ({rx_ready, tx_ready, int_addr, int_rden})
			4'b00??: // idle
			begin
				int_addr <= 1'b0;
				int_rden <= 1'b0;
				int_wren <= 1'b0;
				int_rdreq <= 1'b0;
				int_wrreq <= 1'b0;
			end
			4'b1?1?: // set read addr
			begin
				int_addr <= 1'b0;
				int_rden <= 1'b0;
				int_wren <= 1'b0;
				int_rdreq <= 1'b0;
				int_wrreq <= 1'b0;
			end
			4'b1?00: // enable reads
			begin
				int_addr <= 1'b0;
				int_rden <= 1'b1;
				int_wren <= 1'b0;
				int_rdreq <= 1'b0;
				int_wrreq <= 1'b0;
			end
			4'b1?01: // read
			begin
				int_addr <= 1'b0;
				int_rden <= 1'b1;
				int_wren <= 1'b0;
				int_rdreq <= 1'b1;
				int_wrreq <= 1'b0;
			end
			4'b0101: // disable reads
			begin
				int_addr <= 1'b0;
				int_rden <= 1'b0;
				int_wren <= 1'b0;
				int_rdreq <= 1'b0;
				int_wrreq <= 1'b0;
			end
			4'b0100: // set write addr
			begin
				int_addr <= 1'b1;
				int_rden <= 1'b0;
				int_wren <= 1'b1;
				int_rdreq <= 1'b0;
				int_wrreq <= 1'b0;
			end
			4'b011?: // write
			begin
				int_addr <= 1'b1;
				int_rden <= 1'b0;
				int_wren <= 1'b1;
				int_rdreq <= 1'b0;
				int_wrreq <= 1'b1;
			end
		endcase
/*
		if (usb_pktend)
		begin
			byte_counter <= 9'd0;
			idle_counter <= 5'd0;
		end
		else if (usb_wrreq)
		begin
			byte_counter <= byte_counter + 9'd1;
			idle_counter <= 5'd0;
		end
		else if ((|byte_counter) & (tx_empty))
		begin
			byte_counter <= byte_counter;
			idle_counter <= idle_counter + 5'd1;
		end
*/
	end

	assign	usb_addr = {1'b1, int_addr}; 
	assign	usb_rden = int_rden;
	assign	usb_rdreq = int_rdreq;
	assign	usb_wrreq = int_wrreq;
//	assign	usb_pktend = (&idle_counter);
	assign	usb_pktend = 1'b0;

endmodule
