module usb_fifo
	(
		input	wire			usb_clk,
		inout	wire	[7:0]	usb_data,
		input	wire			usb_full, usb_empty,
		output	wire			usb_wrreq, usb_rdreq, usb_rden, usb_pktend,
		output	wire	[1:0]	usb_addr,
		
		input	wire			clk, aclr,
		input	wire			tx_wrreq, rx_rdreq,
		input	wire	[7:0]	tx_data,
		output	wire			tx_full, rx_empty,
		output	wire	[7:0]	rx_data
	);
	
	localparam EPRD_ADDR	=	2'b10;  // 6
	localparam EPWR_ADDR	=	2'b11;  // 8

	// bidirectional data bus
	wire			usb_wren;
	wire	[7:0]	usb_datain = usb_data;
	wire	[7:0]	usb_dataout;

	assign	usb_data = usb_wren ? usb_dataout : 8'bz;

	wire			tx_rdreq, tx_empty;
	wire			rx_wrreq, rx_full;

	fifo32x8 fifo_tx_unit (
		.aclr(aclr),
		.data(tx_data),
		.rdclk(usb_clk),
		.rdreq(tx_rdreq),
		.wrclk(clk),
		.wrreq(tx_wrreq),
		.q(usb_dataout_bis),
		.rdempty(tx_empty),
		.wrfull(tx_full));

	fifo32x8 fifo_rx_unit (
		.aclr(aclr),
		.data(usb_datain),
		.rdclk(clk),
		.rdreq(rx_rdreq),
		.wrclk(usb_clk),
		.wrreq(rx_wrreq),
		.q(rx_data),
		.rdempty(rx_empty),
		.wrfull(rx_full));

	reg		[31:0]	counter;	
	
	reg		[2:0]	state;
	reg				tx;
	reg		[7:0]	dout;

	always @(posedge usb_clk)
	begin
		case(state)
			0:
			begin
				tx <= 1'b0;
				counter <= 32'd0;
				state <= 3'd1;
			end
			1:
			begin
				if((~usb_full) & (counter < 32'd512))
				begin
					counter <= counter + 32'd1;
					state <= 3'd2;
					dout <= 1;
					tx <= 1'b1;
				end
				else
				begin
					tx <= 1'b0;
				end
			end
				
			2:
			begin
				if((~usb_full) & (counter < 32'd512))
				begin
					counter <= counter + 32'd1;
					state <= 3'd1;
					dout <= 0;
					tx <= 1'b1;
				end
				else
				begin
					tx <= 1'b0;
				end
			end
						
			default: state <= 3'd0;
		endcase
	end

	assign	usb_addr = 2'b11; // FIFO8
	assign	usb_rdreq = 1'b0; // always TX for now
	assign	usb_dataout = dout;
	assign	usb_wrreq = tx;
	assign	usb_pktend = 1'b0;
	assign	usb_rden = 1'b0;
	assign	usb_wren = tx;

endmodule
