package require XOTcl

package require BLT
package require swt
package require usb

package require zlib

wm minsize . 1000 700

namespace eval ::mca {
    namespace import ::xotcl::*

    namespace import ::blt::vector
    namespace import ::blt::graph
    namespace import ::blt::tabnotebook

# -------------------------------------------------------------------------

    variable oscCodes
    array set oscCodes {
        1 {Channel 1}
        2 {Channel 2}
        3 {Channel 3}
        4 {Channel 4}
        5 {Channel 5}
        6 {Trigger}
    }

# -------------------------------------------------------------------------

    variable adcCodes
    array set adcCodes {
        1 {S1_F}
        2 {S1_S}
        3 {S2}
        4 {D1}
        5 {D2}
        6 {D3}
    }

# -------------------------------------------------------------------------

    variable cfgCodes
    array set cfgCodes {
        {1_0} 0
        {1_1} 6
        {1_2} 7
        {1_3} 8
        {2_0} 1
        {2_1} 9
        {2_2} 10
        {2_3} 11
        {3_0} 2
        {3_1} 12
        {3_2} 13
        {3_3} 14
        {3_4} 15
        {4_0} 3
        {4_1} 16
        {4_2} 17
        {4_3} 18
        {4_4} 19
        {5_0} 4
        {5_1} 20
        {5_2} 21
        {5_3} 22
        {5_4} 23
        {6_0} 5
        {6_1} 24
        {6_2} 25
        {6_3} 26
        {6_4} 27
    }
# -------------------------------------------------------------------------

    variable cfgThrs
    array set cfgThrs {
        0     8
        6   150
        7   200
        8  3000
        1     8
        9   150
        10  200
        11 3000
        2    10
        12   10
        13   31
        14   52
        15   79
        3    10
        16   60
        17  135
        18  171
        19  233
        4    10
        20  249
        21  348
        22  495
        23  693
        5    10
        24  505
        25  606
        26  707
        27  808
    }


# -------------------------------------------------------------------------

    variable inpCodes
    array set inpCodes {
        0 {rs}
        1 {cs}
        2 {av}
        3 {af}
        4 {bn}
        5 {bf}
    }

# -------------------------------------------------------------------------

    proc validate {max size value} {
        if {![regexp {^[0-9]*$} $value]} {
            return 0
        } elseif {[regexp {^0[0-9]+$} $value]} {
            return 0
        } elseif {$value > $max} {
            return 0
        } elseif {[string length $value] > $size} {
            return 0
        } else {
            return 1
        }
    }

# -------------------------------------------------------------------------

    proc doublevalidate {max value} {
        if {![regexp {^[0-9]{0,2}\.?[0-9]{0,3}$} $value]} {
            return 0
        } elseif {[regexp {^0[0-9]+$} $value]} {
            return 0
        } elseif {$value > $max} {
            return 0
        } else {
            return 1
        }
    }

# -------------------------------------------------------------------------

    proc legendLabel {master row key title} {
        label ${master}.${key}_label -anchor w -text ${title}
        label ${master}.${key}_value -width 10 -anchor e -text {}

        grid ${master}.${key}_label -row ${row} -column 1 -sticky w
        grid ${master}.${key}_value -row ${row} -column 2 -sticky ew
    }

# -------------------------------------------------------------------------

    proc legendButton {master row key title var bg {fg black}} {
        checkbutton ${master}.${key}_check -variable $var
        label ${master}.${key}_label -anchor w -text ${title} -bg ${bg} -fg $fg
        label ${master}.${key}_value -width 10 -anchor e -text {} -bg ${bg} -fg $fg

        grid ${master}.${key}_check -row ${row} -column 0 -sticky w
        grid ${master}.${key}_label -row ${row} -column 1 -sticky w
        grid ${master}.${key}_value -row ${row} -column 2 -sticky ew
    }

# -------------------------------------------------------------------------

    Class UsbController

# -------------------------------------------------------------------------

    UsbController instproc init {} {

        my set ignore false

        next
    }

# -------------------------------------------------------------------------

    UsbController instproc usbConnect {} {
        my instvar handle ignore

        puts usbConnect

        if {[my exists handle]} {
            $handle disconnect
            unset handle
        }
        if {!$ignore} {
            while {[catch {usb::connect 0x09FB 0x6001 1 1 0} result]} {
                set answer [tk_messageBox -icon error -type abortretryignore \
                    -message {Cannot access USB device} -detail $result]
                if {[string equal $answer abort]} exit
                if {[string equal $answer ignore]} {
                    set ignore true
                    return
                }
            }

            set handle $result

        }
    }

# -------------------------------------------------------------------------

    UsbController instproc usbHandle {} {
        my instvar handle ignore

        if {[my exists handle]} {
            return $handle
        } elseif {!$ignore} {
            my usbConnect
        }
    }

# -------------------------------------------------------------------------

    UsbController instproc usbCmd {command} {
        set code [catch {[my usbHandle] writeRaw [usb::convert $command]} result]
        switch -- $code {
            1 {
#                puts $result
                my usbConnect
            }
        }

    }

# -------------------------------------------------------------------------

    UsbController instproc usbCmdReadRaw {command size data} {
        my usbCmd $command

        set code [catch {[my usbHandle] readRaw $size} result]
        switch -- $code {
            0 {
                set $data $result
            }
            1 {
#                puts $result
                my usbConnect
            }
            5 {
#                puts Busy
            }
        }
    }

# -------------------------------------------------------------------------

    UsbController instproc usbCmdReadRaw {command size data} {
        my usbCmd $command

        set code [catch {[my usbHandle] readRaw $size} result]
        switch -- $code {
            0 {
                set $data $result
            }
            1 {
#                puts $result
                my usbConnect
            }
            5 {
#                puts Busy
            }
        }
    }

# -------------------------------------------------------------------------

    UsbController instproc usbCmdReadHex {command width size data} {
        my usbCmd $command

        set code [catch {[my usbHandle] readHex $width $size} result]
        switch -- $code {
            0 {
                set $data $result
            }
            1 {
#                puts $result
                my usbConnect
            }
            5 {
#                puts Busy
            }
        }
    }

# -------------------------------------------------------------------------

    Class CfgDisplay -parameter {
        {master}
        {controller}
    }

# -------------------------------------------------------------------------

    CfgDisplay instproc init {} {

        my setup

        next
    }

# -------------------------------------------------------------------------

    CfgDisplay instproc destroy {} {
        next
    }

# -------------------------------------------------------------------------

    CfgDisplay instproc start {} {
        variable adcCodes
        variable cfgThrs

        foreach {ch id} [array get adcCodes] {
            my set delay($ch) 32
            my set decay($ch) 1000
        }

        foreach {i value} [array get cfgThrs] {
            my set thrs($i) $value
        }

        trace add variable [myvar decay] write [myproc decay_update]
        trace add variable [myvar delay] write [myproc delay_update]
        trace add variable [myvar thrs] write [myproc thrs_update]

        my delay_update
        my thrs_update
    }

# -------------------------------------------------------------------------

    CfgDisplay instproc setup {} {
        variable adcCodes
        variable cfgCodes
        my instvar number master
        my instvar config

        set thrs [frame ${master}.thrs]
        set bins [frame ${master}.bins]
        set clip [frame ${master}.clip]

        set config(thrs) [labelframe ${thrs}.frame -borderwidth 1 -relief sunken -text {Thresholds}]

        set column 0
        foreach {input} [list "ADC" "e thrs min" "e thrs max" "p thrs min" "p thrs max"] {
            label ${config(thrs)}.label_${column} -text "${input}"
            grid ${config(thrs)}.label_${column} -row 0 -column ${column} -sticky ew -padx 5 -pady 7
            incr column
        }

        foreach {ch id} [array get adcCodes {[1-2]}] {
            label ${config(thrs)}.chan_${ch} -text "${id} "
            grid ${config(thrs)}.chan_${ch} -row ${ch} -column 0 -sticky ew -padx 5 -pady 7
            foreach {num} [list 0 1 2 3] {
                set column [expr {$num + 1}]
                set value $cfgCodes(${ch}_${num})
                spinbox ${config(thrs)}.thrs_${value} -from 0 -to 4095 \
                  -increment 10 -width 10 -textvariable [myvar thrs($value)] \
                  -validate all -vcmd {::mca::validate 4095 4 %P}
                grid ${config(thrs)}.thrs_${value} -row ${ch} -column ${column} -sticky w -padx 5 -pady 7
            }
        }

        grid $config(thrs) -row 0 -column 0 -sticky news -padx 10

        set config(bins) [labelframe ${bins}.frame -borderwidth 1 -relief sunken -text {Bins}]

        set column 0
        foreach {input} [list "ADC" "thrs 0" "thrs 1" "thrs 2" "thrs 3" "thrs 4"] {
            label ${config(bins)}.label_${column} -text "${input}"
            grid ${config(bins)}.label_${column} -row 0 -column ${column} -sticky ew -padx 5 -pady 7
            incr column
        }

        foreach {ch id} [array get adcCodes {[3-6]}] {
            label ${config(bins)}.chan_${ch} -text "${id} "
            grid ${config(bins)}.chan_${ch} -row ${ch} -column 0 -sticky ew -padx 5 -pady 7
            foreach {num} [list 0 1 2 3 4] {
                set column [expr {$num + 1}]
                set value $cfgCodes(${ch}_${num})
                spinbox ${config(bins)}.bins_${value} -from 0 -to 4095 \
                  -increment 10 -width 10 -textvariable [myvar thrs($value)] \
                  -validate all -vcmd {::mca::validate 4095 4 %P}
                grid ${config(bins)}.bins_${value} -row ${ch} -column ${column} -sticky w -padx 5 -pady 7
            }
        }

        grid $config(bins) -row 0 -column 0 -sticky news -padx 10

        set config(clip) [labelframe ${clip}.frame -borderwidth 1 -relief sunken -text {Signal clipping}]

        set column 0
        foreach {input} [list "ADC" "delay" "decay"] {
            label ${config(clip)}.label_${column} -text "${input}"
            grid ${config(clip)}.label_${column} -row 0 -column ${column} -sticky ew -padx 5 -pady 7
            incr column
        }

        foreach {ch id} [array get adcCodes] {
            label ${config(clip)}.chan_${ch} -text "${id} "
            grid ${config(clip)}.chan_${ch} -row ${ch} -column 0 -sticky ew -padx 5 -pady 7
            spinbox ${config(clip)}.delay_${ch} -from 0 -to 62 \
                -increment 2 -width 10 -textvariable [myvar delay($ch)] \
                -validate all -vcmd {::mca::validate 63 5 %P}
            grid ${config(clip)}.delay_${ch} -row ${ch} -column 1 -sticky w -padx 5 -pady 7
            spinbox ${config(clip)}.decay_${ch} -from 0 -to 65535 \
                -increment 10 -width 10 -textvariable [myvar decay($ch)] \
                -validate all -vcmd {::mca::validate 65535 5 %P}
            grid ${config(clip)}.decay_${ch} -row ${ch} -column 2 -sticky w -padx 5 -pady 7
        }

        grid $config(clip) -row 0 -column 0 -sticky news -padx 10

        grid ${thrs} -row 0 -column 2 -sticky news
        grid ${bins} -row 0 -column 1 -sticky news
        grid ${clip} -row 0 -column 0 -sticky news

        grid columnconfigure ${master} 0 -weight 1
        grid columnconfigure ${master} 1 -weight 1
        grid columnconfigure ${master} 2 -weight 1
        grid rowconfigure ${master} 0 -weight 1

        grid rowconfigure ${thrs} 0 -weight 0
        grid rowconfigure ${bins} 0 -weight 0
        grid rowconfigure ${clip} 0 -weight 0
    }

# -------------------------------------------------------------------------

    CfgDisplay instproc decay_update args {
        my instvar controller decay delay

        set command {}
        for {set i 1} {$i <= 6} {incr i} {
            set a $delay($i).0
            set b $decay($i).0
            set value [expr int(exp(-${a}/${b})*1024*20)]
            append command [format {000200%02x0004%04x} [expr {38 + 2 * (${i} - 1)}] $value]
        }

        $controller usbCmd $command
    }

# -------------------------------------------------------------------------

    CfgDisplay instproc delay_update args {
        my instvar controller delay

        set command {}
        for {set i 1} {$i <= 6} {incr i} {
            append command [format {000200%02x0004%04x} [expr {39 + 2 * (${i} - 1)}] $delay($i)]
        }

        $controller usbCmd $command

        my decay_update
    }

# -------------------------------------------------------------------------

    CfgDisplay instproc thrs_update args {
        my instvar controller thrs

        set command {}
        for {set i 0} {$i <= 27} {incr i} {
            append command [format {000200%02x0004%04x} [expr {10 + ${i}}] $thrs($i)]
        }

        $controller usbCmd $command
    }

# -------------------------------------------------------------------------

    Class MuxDisplay -parameter {
        {master}
        {controller}
    }

# -------------------------------------------------------------------------

    MuxDisplay instproc init {} {

        my setup

        next
    }

# -------------------------------------------------------------------------

    MuxDisplay instproc destroy {} {
        next
    }

# -------------------------------------------------------------------------

    MuxDisplay instproc start {} {
        variable adcCodes
        my instvar config chan_val

        set chan_val(1) 0
        set chan_val(2) 0
        set chan_val(3) 0
        set chan_val(4) 0
        set chan_val(5) 0
        set chan_val(6) 0

        trace add variable [myvar chan_val] write [myproc chan_val_update]
        trace add variable [myvar polar] write [myproc polar_update]

        $config(1).chan_1_1 select
        $config(2).chan_1_2 select
        $config(3).chan_1_3 select
        $config(4).chan_1_4 select
        $config(5).chan_1_5 select
        $config(6).chan_2_4 select

        foreach {ch dummy} [array get adcCodes] {
          $config(inv).polar_${ch} deselect
        }
    }

# -------------------------------------------------------------------------

    MuxDisplay instproc setup {} {
        variable oscCodes
        variable adcCodes
        variable inpCodes
        my instvar master
        my instvar config

        set size [array size inpCodes]
        set oscList [array get oscCodes]
        set adcList [array get adcCodes]
        set inpList [array get inpCodes]

        set mux [frame ${master}.mux]
        set key [frame ${master}.key]
        set inv [frame ${master}.inv]

        foreach {osc title} $oscList {
            set config($osc) [labelframe ${mux}.$osc -borderwidth 1 -relief sunken -text $title]
            foreach {code input} $inpList {
                set column [expr {$code + 1}]
                label $config($osc).input_${input} -text " ${input}"
                grid $config($osc).input_${input} -row 0 -column ${column} -sticky w
            }
            foreach {ch id} $adcList {
                label $config($osc).chan_${ch} -text "${id} "
                grid $config($osc).chan_${ch} -row ${ch} -column 0 -sticky ew
                foreach {code input} $inpList {
                    set column [expr {$code + 1}]
                    set value [expr {$size * ($ch - 1) + $code}]
                    radiobutton $config($osc).chan_${code}_${ch} -variable [myvar chan_val($osc)] -value ${value}
                    grid $config($osc).chan_${code}_${ch} -row ${ch} -column ${column} -sticky w
                }
            }
            set column [expr {($osc - 1) % 3}]
            set row [expr {($osc - 1) / 3}]
            grid $config($osc) -row ${row} -column ${column} -sticky news -padx 10
        }

        set config(key) [labelframe ${key}.frame -borderwidth 1 -relief sunken -text {legend}]

        label $config(key).rs -text "rs - raw signal"
        grid $config(key).rs -row 0 -column 0 -sticky news

        label $config(key).cs -text "cs - filtered and clipped signal"
        grid $config(key).cs -row 0 -column 1 -sticky news

        label $config(key).av -text "av - amplitude value"
        grid $config(key).av -row 0 -column 2 -sticky news

        label $config(key).af -text "af - amplitude flag"
        grid $config(key).af -row 0 -column 3 -sticky news

        label $config(key).bn -text "bn - bin number"
        grid $config(key).bn -row 0 -column 4 -sticky news

        label $config(key).bf -text "bf - bin flag"
        grid $config(key).bf -row 0 -column 5 -sticky news

        grid $config(key) -row 0 -column 0 -sticky news -padx 10


        set config(inv) [labelframe ${inv}.frame -borderwidth 1 -relief sunken -text {polarity inversion}]
        label $config(inv).chan_label -text "channel "
        grid $config(inv).chan_label -row 0 -column 0 -sticky e
        label $config(inv).polar_label -text "polarity"
        grid $config(inv).polar_label -row 1 -column 0 -sticky e
        foreach {ch dummy} $adcList {
            label $config(inv).chan_${ch} -text "${ch} "
            grid $config(inv).chan_${ch} -row 0 -column ${ch} -sticky ew
            checkbutton $config(inv).polar_${ch} -variable [myvar polar($ch)]
            grid $config(inv).polar_${ch} -row 1 -column ${ch} -sticky w
        }
        grid $config(inv) -row 0 -column 0 -sticky news -padx 10

        grid ${key} -row 0 -column 0 -sticky news
        grid ${mux} -row 1 -column 0 -sticky news
        grid ${inv} -row 2 -column 0 -sticky news

        grid columnconfigure ${master} 0 -weight 1
        grid rowconfigure ${master} 0 -weight 1
        grid rowconfigure ${master} 1 -weight 1
        grid rowconfigure ${master} 2 -weight 1

        grid columnconfigure ${inv} 0 -weight 1

        grid columnconfigure ${key} 0 -weight 1
        grid columnconfigure $config(key) 0 -weight 1
        grid columnconfigure $config(key) 1 -weight 1
        grid columnconfigure $config(key) 2 -weight 1
        grid columnconfigure $config(key) 3 -weight 1
        grid columnconfigure $config(key) 4 -weight 1
        grid columnconfigure $config(key) 5 -weight 1


        grid columnconfigure ${mux} 0 -weight 1
        grid columnconfigure ${mux} 1 -weight 1
        grid columnconfigure ${mux} 2 -weight 1
    }

# ------------------------------------------------------------------------

    MuxDisplay instproc chan_val_update args {
        my instvar controller chan_val

        set byte1 [format {%02x%02x} $chan_val(2) $chan_val(1)]
        set byte2 [format {%02x%02x} $chan_val(4) $chan_val(3)]
        set byte3 [format {%02x%02x} $chan_val(6) $chan_val(5)]

        $controller usbCmd 000200020004${byte1}000200030004${byte2}000200040004${byte3}
    }

# -------------------------------------------------------------------------

    MuxDisplay instproc polar_update args {
        my instvar controller polar

        set value {0b}
        for {set i 6} {$i >= 1} {incr i -1} {
            append value $polar($i)
        }

        set value [format {%04x} $value]

        $controller usbCmd 000200010004${value}
    }

# -------------------------------------------------------------------------

    Class HstDisplay -parameter {
        {number}
        {master}
        {controller}
    }

# -------------------------------------------------------------------------

    HstDisplay instproc init {} {

        my set data {}

        vector create [myvar xvec](80)
        vector create [myvar yvec](80)

        # fill one vector for the x axis with 80 points
        [myvar xvec] seq -0.5 79.5

        my setup

        next
    }

# -------------------------------------------------------------------------

    HstDisplay instproc destroy {} {
        next
    }

# -------------------------------------------------------------------------

    HstDisplay instproc start {} {
        my instvar config

        trace add variable [myvar data] write [myproc data_update]

        trace add variable [myvar axis] write [myproc axis_update]

        ${config}.axis_check select

        my set yvec_bak 0.0
        my set yvec_old 0.0
    }

# -------------------------------------------------------------------------

    HstDisplay instproc setup {} {
        my instvar number master
        my instvar xvec yvec graph
        my instvar config thrs thrs_val base base_typ base_val
        my instvar cntr_h cntr_m cntr_s

        # create a graph widget and show a grid
        set graph [graph ${master}.graph -height 250 -leftmargin 80]
        $graph crosshairs configure -hide no -linewidth 1 -color darkblue -dashes {2 2}
        $graph grid configure -hide no
        $graph legend configure -hide yes
        $graph axis configure x -min 0 -max 80

        set config [frame ${master}.config -width 170]

        checkbutton ${config}.axis_check -text {log scale} -variable [myvar axis]

        frame ${config}.spc1 -width 170 -height 30

        frame ${config}.chan_frame -borderwidth 0 -width 170
        legendLabel ${config}.chan_frame 0 axisy {Bin entries}
        legendLabel ${config}.chan_frame 1 axisx {Bin number}

        frame ${config}.spc2 -width 170 -height 30

        button ${config}.start -text Start \
            -bg yellow -activebackground yellow -command [myproc cntr_start]
        button ${config}.reset -text Reset \
            -bg red -activebackground red -command [myproc cntr_reset]

        frame ${config}.spc3 -width 170 -height 30

        button ${config}.register -text Register \
            -bg lightblue -activebackground lightblue -command [myproc register]

        grid ${config}.axis_check -sticky w
        grid ${config}.spc1
        grid ${config}.chan_frame -sticky ew -padx 5
        grid ${config}.spc2
        grid ${config}.start -sticky ew -pady 3 -padx 5
        grid ${config}.reset -sticky ew -pady 3 -padx 5
        grid ${config}.spc3
        grid ${config}.register -sticky ew -pady 3 -padx 5

        grid ${graph} -row 0 -column 0 -sticky news
        grid ${config} -row 0 -column 1

        grid rowconfigure ${master} 0 -weight 1
        grid columnconfigure ${master} 0 -weight 1
        grid columnconfigure ${master} 1 -weight 0 -minsize 80

        grid columnconfigure ${config}.chan_frame 1 -weight 1

        # enable zooming
        Blt_ZoomStack $graph

        my crosshairs $graph

        #bind .graph <Motion> {%W crosshairs configure -position @%x,%y}

        # create one element with data for the x and y axis, no dots
        $graph element create Spectrum1 -color blue -linewidth 2 -symbol none -smooth step -xdata [myvar xvec] -ydata [myvar yvec]
    }

# -------------------------------------------------------------------------

    HstDisplay instproc coor_update {W x y} {
        my instvar config graph

        $W crosshairs configure -position @${x},${y}

        set index [$W axis invtransform x $x]
        set index [::tcl::mathfunc::round $index]
        catch {
            ${config}.chan_frame.axisy_value configure -text [[myvar yvec] index $index]
            ${config}.chan_frame.axisx_value configure -text ${index}.0
        }
    }
# -------------------------------------------------------------------------

    HstDisplay instproc crosshairs {graph} {
        set method [myproc coor_update]
        bind $graph <Motion> [list [self] coor_update %W %x %y]
        bind $graph <Leave> {
            %W crosshairs off
        }
        bind $graph <Enter> {
            %W crosshairs on
        }
    }

# -------------------------------------------------------------------------

    HstDisplay instproc axis_update args {
        my instvar axis graph
        if {$axis} {
            $graph axis configure y -min 1 -max 1E10 -logscale yes
        } else {
            $graph axis configure y -min {} -max {} -logscale no
        }
    }

# -------------------------------------------------------------------------

    HstDisplay instproc cntr_reset {} {
        my instvar controller number

        my cntr_stop

        set value [format %04x [expr {1 << (5 + ${number})}]]
        $controller usbCmd 000200000004${value}0002000000040000

        my acquire
    }

# -------------------------------------------------------------------------

    HstDisplay instproc cntr_start {} {
        my instvar controller config number auto

        set val_addr [format %02x [expr {6 + ${number}}]]

        ${config}.start configure -text Stop -command [myproc cntr_stop]
#        ${config}.reset configure -state disabled

        $controller usbCmd 000200${val_addr}00040001

        set auto 1

        after 100 [myproc acquire_loop]
    }

# -------------------------------------------------------------------------

    HstDisplay instproc cntr_stop {} {
        my instvar controller config number auto

        set val_addr [format %02x [expr {6 + ${number}}]]

        ${config}.start configure -text Start -command [myproc cntr_start]

        $controller usbCmd 000200${val_addr}00040000

        set auto 0

        my acquire
    }

# -------------------------------------------------------------------------

    HstDisplay instproc data_update args {
        my instvar data
        usb::convertBlt $data 4 [myvar yvec]
    }

# -------------------------------------------------------------------------

    HstDisplay instproc acquire_loop {} {
        my instvar cntr_val auto

        my acquire

        if {$auto} {
            after 1000 [myproc acquire_loop]
        }
    }

# -------------------------------------------------------------------------

    HstDisplay instproc acquire {} {
        my instvar controller config number

        set size 80

        set prefix [format {%x} [expr {$number + 2}]]

        set value [format {%08x} [expr {$size * 2}]]

        set command 0001${prefix}000000200000001[string range $value 0 3]0003[string range $value 4 7]00050000

        $controller usbCmdReadRaw $command [expr {$size * 4}] [myvar data]
   }

# -------------------------------------------------------------------------

    HstDisplay instproc save_data {data} {
        my instvar number

        set types {
            {{Data Files}       {.dat}        }
            {{All Files}        *             }
        }

        set stamp [clock format [clock seconds] -format %Y%m%d_%H%M%S]
        set fname spectrum_[expr {$number + 1}]_${stamp}.dat

        set fname [tk_getSaveFile -filetypes $types -initialfile $fname]
        if {[string equal $fname {}]} {
            return
        }

        set x [catch {
            set fid [open $fname w+]
            puts $fid $data
            close $fid
        }]

        if { $x || ![file exists $fname] || ![file isfile $fname] || ![file readable $fname] } {
            tk_messageBox -icon error \
                -message "An error occurred while writing to \"$fname\""
        } else {
            tk_messageBox -icon info \
                -message "File \"$fname\" written successfully"
        }
    }

# -------------------------------------------------------------------------

    HstDisplay instproc register {} {
        my save_data [join [[myvar yvec] range 0 79] \n]
    }

# -------------------------------------------------------------------------

    Class OscDisplay -parameter {
        {master}
        {controller}
    }

# -------------------------------------------------------------------------

    OscDisplay instproc init {} {
        my instvar sequence data xvec yvec

        set data {}

        set sequence 0

#        set xvec [vector create #auto(262144)]
#        set xvec [vector create #auto(10000)]
        set xvec [vector create #auto(60000)]

        for {set i 1} {$i <= 9} {incr i} {
#          dict set yvec $i [vector create #auto(262144)]
#          dict set yvec $i [vector create #auto(10000)]
          dict set yvec $i [vector create #auto(60000)]
        }

        # fill one vector for the x axis
#        $xvec seq 0 262143
#        $xvec seq 0 10000
        $xvec seq 0 60000

        my setup

        next
    }

# -------------------------------------------------------------------------

    OscDisplay instproc destroy {} {
        next
    }

# -------------------------------------------------------------------------

    OscDisplay instproc start {} {
        my instvar config
        my instvar recs_val directory

        set directory $::env(HOME)
        set recs_val 100

        trace add variable [myvar chan] write [myproc chan_update]

        trace add variable [myvar data] write [myproc data_update]

        trace add variable [myvar auto] write [myproc auto_update]

        trace add variable [myvar thrs] write [myproc thrs_update 0]
        trace add variable [myvar thrs_val] write [myproc thrs_update 0]

        trace add variable [myvar recs_val] write [myproc recs_val_update]

        trace add variable [myvar last] write [myproc last_update]

        for {set i 1} {$i <= 6} {incr i} {
            ${config}.chan_frame.chan${i}_check select
            ${config}.chan_frame.chan${i}_value configure -text 0.0
        }
        ${config}.chan_frame.axisx_value configure -text 0.0

        ${config}.thrs_check select
        ${config}.thrs_field set 60
    }

# -------------------------------------------------------------------------

    OscDisplay instproc setup {} {
        my instvar master
        my instvar xvec yvec graph
        my instvar config

        # create a graph widget and show a grid
        set graph [graph ${master}.graph -height 250 -leftmargin 80]
        $graph crosshairs configure -hide no -linewidth 1 -color darkblue -dashes {2 2}
        $graph grid configure -hide no
        $graph legend configure -hide yes
        $graph axis configure x -min 0 -max 60000
        $graph axis configure y -min 0 -max 4100

#        scale ${master}.last -orient horizontal -from 1 -to 27 -tickinterval 0 -showvalue no -variable [myvar last]

        set config [frame ${master}.config -width 170]

        frame ${config}.chan_frame -width 170
        legendButton ${config}.chan_frame 0 chan1 {Channel 1} [myvar chan(1)] turquoise2
        legendButton ${config}.chan_frame 1 chan2 {Channel 2} [myvar chan(2)] SpringGreen2
        legendButton ${config}.chan_frame 2 chan3 {Channel 3} [myvar chan(3)] orchid2
        legendButton ${config}.chan_frame 3 chan4 {Channel 4} [myvar chan(4)] orange2
        legendButton ${config}.chan_frame 4 chan5 {Channel 5} [myvar chan(5)] blue1 white
        legendButton ${config}.chan_frame 5 chan6 {Channel 6} [myvar chan(6)] gray65 white
        legendLabel  ${config}.chan_frame 6 axisx {Time axis}

        frame ${config}.spc1 -width 170 -height 30

        checkbutton ${config}.auto_check -text {auto update} -variable [myvar auto]

        frame ${config}.spc2 -width 170 -height 30

        checkbutton ${config}.thrs_check -text threshold -variable [myvar thrs]
        spinbox ${config}.thrs_field -from 1 -to 4095 \
            -increment 5 -width 10 -textvariable [myvar thrs_val] \
            -validate all -vcmd {::mca::validate 4095 4 %P}

        frame ${config}.spc3 -width 170 -height 30

        button ${config}.acquire -text Acquire \
            -bg green -activebackground green -command [myproc acquire_start]
        button ${config}.register -text Register \
            -bg lightblue -activebackground lightblue -command [myproc register]

        frame ${config}.spc4 -width 170 -height 30

        label ${config}.recs -text {number of records}
        spinbox ${config}.recs_field -from 0 -to 10000 \
            -increment 10 -width 10 -textvariable [myvar recs_val] \
            -validate all -vcmd {::mca::validate 10000 5 %P}

        frame ${config}.spc5 -width 170 -height 10

        button ${config}.sequence -text {Start Recording} -command [myproc sequence_start] \
            -bg yellow -activebackground yellow

        frame ${config}.spc6 -width 170 -height 30

        button ${config}.recover -text {Read file} \
            -bg lightblue -activebackground lightblue -command [myproc recover]

        grid ${config}.chan_frame -sticky ew
        grid ${config}.spc1
        grid ${config}.auto_check -sticky w
        grid ${config}.spc2
        grid ${config}.thrs_check -sticky w
        grid ${config}.thrs_field -sticky ew -pady 1 -padx 5
        grid ${config}.spc3
        grid ${config}.acquire -sticky ew -pady 3 -padx 5
        grid ${config}.register -sticky ew -pady 3 -padx 5
        grid ${config}.spc4
        grid ${config}.recs -sticky w -pady 1 -padx 3
        grid ${config}.recs_field -sticky ew -pady 1 -padx 5
        grid ${config}.spc5
        grid ${config}.sequence -sticky ew -pady 3 -padx 5
        grid ${config}.spc6
        grid ${config}.recover -sticky ew -pady 3 -padx 5

        grid ${graph} -row 0 -column 0 -sticky news
        grid ${config} -row 0 -column 1

#        grid ${master}.last -row 1 -column 0 -columnspan 2 -sticky ew

        grid rowconfigure ${master} 0 -weight 1
        grid columnconfigure ${master} 0 -weight 1
        grid columnconfigure ${master} 1 -weight 0 -minsize 120

        grid columnconfigure ${config}.chan_frame 2 -weight 1

        # enable zooming
        Blt_ZoomStack $graph

        my crosshairs $graph

        # create one element with data for the x and y axis, no dots
        $graph pen create pen1 -color turquoise3 -linewidth 2 -symbol none
        $graph pen create pen2 -color SpringGreen3 -linewidth 2 -symbol none
        $graph pen create pen3 -color orchid3 -linewidth 2 -symbol none
        $graph pen create pen4 -color orange3 -linewidth 2 -symbol none
        $graph pen create pen5 -color blue2 -linewidth 2 -symbol none
        $graph pen create pen6 -color gray55 -linewidth 2 -symbol none

        $graph element create Spectrum1 -pen pen1 -xdata $xvec -ydata [dict get $yvec 1]
        $graph element create Spectrum2 -pen pen2 -xdata $xvec -ydata [dict get $yvec 2]
        $graph element create Spectrum3 -pen pen3 -xdata $xvec -ydata [dict get $yvec 3]
        $graph element create Spectrum4 -pen pen4 -xdata $xvec -ydata [dict get $yvec 4]
        $graph element create Spectrum5 -pen pen5 -xdata $xvec -ydata [dict get $yvec 5]
        $graph element create Spectrum6 -pen pen6 -xdata $xvec -ydata [dict get $yvec 6]
    }

# -------------------------------------------------------------------------

    OscDisplay instproc coor_update {W x y} {
        my instvar xvec yvec graph
        my instvar config

        $W crosshairs configure -position @${x},${y}

        set index [$W axis invtransform x $x]
        set index [::tcl::mathfunc::round $index]
        catch {
            ${config}.chan_frame.chan1_value configure -text [[dict get $yvec 1] index $index]
            ${config}.chan_frame.chan2_value configure -text [[dict get $yvec 2] index $index]
            ${config}.chan_frame.chan3_value configure -text [[dict get $yvec 3] index $index]
            ${config}.chan_frame.chan4_value configure -text [[dict get $yvec 4] index $index]
            ${config}.chan_frame.chan5_value configure -text [[dict get $yvec 5] index $index]
            ${config}.chan_frame.chan6_value configure -text [[dict get $yvec 6] index $index]
            ${config}.chan_frame.axisx_value configure -text ${index}.0
        }
    }
# -------------------------------------------------------------------------

    OscDisplay instproc crosshairs {graph} {
        set method [myproc coor_update]
        bind $graph <Motion> [list [self] coor_update %W %x %y]
        bind $graph <Leave> {
            %W crosshairs off
        }
        bind $graph <Enter> {
            %W crosshairs on
        }
    }

# -------------------------------------------------------------------------

    OscDisplay instproc chan_update {name key op} {
        my instvar config graph chan

        if {$chan(${key})} {
            $graph pen configure pen${key} -linewidth 2
        } else {
            $graph pen configure pen${key} -linewidth 0
        }
    }

# -------------------------------------------------------------------------

    OscDisplay instproc recs_val_update args {
        my instvar recs_val
        if {[string equal $recs_val {}]} {
            set recs_val 0
        }
    }

# -------------------------------------------------------------------------

    OscDisplay instproc last_update args {
        my instvar graph last

        set first [expr {$last - 1}]

        $graph axis configure x -min ${first}0000 -max ${last}0000
    }

# -------------------------------------------------------------------------

    OscDisplay instproc thrs_update {reset args} {
        my instvar controller config thrs thrs_val

        if {[string equal $thrs_val {}]} {
            set thrs_val 0
        }

        if {$thrs} {
            ${config}.thrs_field configure -state normal
            set value [format {%03x} $thrs_val]
        } else {
            ${config}.thrs_field configure -state disabled
            set value 000
        }

        set command {}
        if {$reset} {
            append command 0002000500041${value}
        }
        append command 0002000500040${value}

        $controller usbCmd $command
    }

# -------------------------------------------------------------------------

    OscDisplay instproc data_update args {
        my instvar data yvec
        my instvar graph chan waiting sequence auto

        usb::convertOsc $data $yvec

        foreach {key value} [array get chan] {
            $graph pen configure pen${key} -dashes 0
        }

        set waiting 0

        if {$sequence} {
            my sequence_register
        } elseif {$auto} {
            after 1000 [myproc acquire_start]
        }
    }

# -------------------------------------------------------------------------

    OscDisplay instproc acquire_start {} {
        my instvar graph chan controller waiting

        foreach {key value} [array get chan] {
            $graph pen configure pen${key} -dashes dot
        }

        # restart
        my thrs_update 1

        set waiting 1

        after 200 [myproc acquire_loop]
    }

# -------------------------------------------------------------------------

    OscDisplay instproc acquire_loop {} {
        my instvar controller waiting

#        set size 262144
#        set size 10000
        set size 60000

        set value [format {%08x} [expr {$size * 4}]]

        set command 00011000000200000001[string range $value 0 3]0003[string range $value 4 7]00050000

        $controller usbCmdReadRaw $command [expr {$size * 8}] [myvar data]

        if {$waiting} {
            after 200 [myproc acquire_loop]
        }
    }

# -------------------------------------------------------------------------

    OscDisplay instproc auto_update args {
        my instvar config auto

        if {$auto} {
            ${config}.recs_field configure -state disabled
            ${config}.sequence configure -state disabled
            ${config}.acquire configure -state disabled
            ${config}.register configure -state disabled
            ${config}.recover configure -state disabled

            my acquire_start
        } else {
            ${config}.recs_field configure -state normal
            ${config}.sequence configure -state active
            ${config}.acquire configure -state active
            ${config}.register configure -state active
            ${config}.recover configure -state active
        }
    }

# -------------------------------------------------------------------------

    OscDisplay instproc save_data {fname} {
        my instvar data

        set fid [open $fname w+]
        fconfigure $fid -translation binary -encoding binary

#        puts -nonewline $fid [binary format "H*iH*" "1f8b0800" [clock seconds] "0003"]
#        puts -nonewline $fid [zlib deflate $data]
        puts -nonewline $fid $data
#        puts -nonewline $fid [binary format i [zlib crc32 $data]]
#        puts -nonewline $fid [binary format i [string length $data]]

        close $fid
    }

# -------------------------------------------------------------------------

    OscDisplay instproc open_data {} {
        set types {
            {{Data Files}       {.dat}     }
            {{All Files}        *             }
        }

        set fname [tk_getOpenFile -filetypes $types]
        if {[string equal $fname {}]} {
            return
        }

        set x [catch {
            set fid [open $fname r+]
            fconfigure $fid -translation binary -encoding binary
#            set size [file size $fname]
#            seek $fid 10
#            my set data [zlib inflate [read $fid [expr {$size - 18}]]]
            my set data [read $fid]
            close $fid
        }]

        if { $x || ![file exists $fname] || ![file isfile $fname] || ![file readable $fname] } {
            tk_messageBox -icon error \
                -message "An error occurred while reading \"$fname\""
        } else {
            tk_messageBox -icon info \
                -message "File \"$fname\" read successfully"
        }
    }

# -------------------------------------------------------------------------

    OscDisplay instproc register {} {
        set types {
            {{Data Files}       {.dat}     }
            {{All Files}        *             }
        }

        set stamp [clock format [clock seconds] -format {%Y%m%d_%H%M%S}]
        set fname oscillogram_${stamp}.dat

        set fname [tk_getSaveFile -filetypes $types -initialfile $fname]
        if {[string equal $fname {}]} {
            return
        }

        if {[catch {my save_data $fname} result]} {
            tk_messageBox -icon error \
                -message "An error occurred while writing to \"$fname\""
        } else {
            tk_messageBox -icon info \
                -message "File \"$fname\" written successfully"
        }
    }

# -------------------------------------------------------------------------

    OscDisplay instproc recover {} {
        my open_data
    }

# -------------------------------------------------------------------------

    OscDisplay instproc sequence_start {} {
        my instvar config recs_val recs_bak directory counter sequence

        set counter 1
        if {$counter > $recs_val} {
            return
        }

        set directory [tk_chooseDirectory -initialdir $directory -title {Choose a directory}]

        if {[string equal $directory {}]} {
           return
        }

        ${config}.recs_field configure -state disabled
        ${config}.sequence configure -text {Stop Recording} -command [myproc sequence_stop]
        ${config}.acquire configure -state disabled
        ${config}.register configure -state disabled
        ${config}.recover configure -state disabled

        set recs_bak $recs_val

        set sequence 1

        my acquire_start
    }

# -------------------------------------------------------------------------

    OscDisplay instproc sequence_register {} {
        my instvar config recs_val recs_bak directory counter

        set fname [file join $directory oscillogram_$counter.dat]

        my incr counter

        if {[catch {my save_data $fname} result]} {
            tk_messageBox -icon error \
                -message "An error occurred while writing to \"$fname\""
        } elseif {$counter <= $recs_bak} {
            set recs_val [expr {$recs_bak - $counter}]
            my acquire_start
            return
        }

        my sequence_stop
    }

# -------------------------------------------------------------------------

    OscDisplay instproc sequence_stop {} {
        my instvar config recs_val recs_bak sequence

        set sequence 0

        set recs_val $recs_bak

        ${config}.recs_field configure -state normal
        ${config}.sequence configure -text {Start Recording} -command [myproc sequence_start]
        ${config}.acquire configure -state active
        ${config}.register configure -state active
        ${config}.recover configure -state active
    }

# -------------------------------------------------------------------------

    namespace export MuxDisplay
    namespace export HstDisplay
    namespace export CfgDisplay
    namespace export OscDisplay
}

set notebook [::blt::tabnotebook .notebook -borderwidth 1 -selectforeground black -side bottom]

grid ${notebook} -row 0 -column 0  -sticky news -pady 5

grid rowconfigure . 0 -weight 1
grid columnconfigure . 0 -weight 1

::mca::UsbController usb

set window [frame ${notebook}.mux]
$notebook insert end -text "Interconnect" -window $window -fill both
::mca::MuxDisplay mux -master $window -controller usb

set window [frame ${notebook}.cfg]
$notebook insert end -text "Configuration" -window $window -fill both
::mca::CfgDisplay cfg -master $window -controller usb

set window [frame ${notebook}.hst]
$notebook insert end -text "Histogram" -window $window -fill both
::mca::HstDisplay hst -number 0 -master $window -controller usb

set window [frame ${notebook}.ept]
$notebook insert end -text "Oscilloscope" -window $window -fill both
::mca::OscDisplay osc -master $window -controller usb

update

cfg start

mux start

hst start

osc start
