// mrst class file
// Comments to the C++ version to: andersen@hep.phy.cam.ac.uk
// Usage:
// Initialising the class:
// c_mrst parton("filename.dat")
// with "filename.dat" being the data file to interpolate
//
// A call to the method 
// c_mrst::mrstparton(int parton,double x, double q)
// will return the value of the parton with number 'parton' at (x,q^2)
// This method is fastest, if only one parton needs evaluated
// 
// A call to the method 
// c_mrst::mrst(x,q)
// updates the parton content to the values at (x,q^2)
// The parton contents can then be accessed in
// c_mrst::cont.upv etc.
// This method is faster, if all the partons need to be evaluated

// mrst class header file
// Comments regarding the C++ version to: andersen@hep.phy.cam.ac.uk
// Interpolates on the grid for the MRST parton distributions
// Usage:
// Initialising the class:
// c_mrst parton("filename.dat")
// with "filename.dat" being the data file to interpolate
//
// A call to the method 
// c_mrst::mrstparton(int parton,double x, double q)
// will return the value of the parton with number 'parton' at (x,q^2)
// This method is fastest, if only one parton needs evaluated
// 
// A call to the method 
// c_mrst::mrst(x,q)
// updates the parton content to the values at (x,q^2)
// The parton contents can then be accessed in
// c_mrst::cont.upv etc.
// This method is faster, if all the partons need to be evaluated

#include <stdio.h>
#include <iostream>
#include <fstream>
#include <stdlib.h>
#include <string>
#include <math.h>
#include "mathlink.h"

using namespace std;

static const int np=8;
static const int nx=49;
static const int nq=37;
static const int nqc0=2; // Charm not introduced before 2nd bin in q^2
static const int nqb0=11; // Bottom not introduced before 11th bin in q^2
//static const int ntenth=23;
static const double xmin=1E-5;
static const double xmax=1.0;
static const double qsqmin=1.25;
static const double qsqmax=1E7;
static const double mc2=2.045;
static const double mb2=18.5;

struct s_partoncontent {
  double upv,dnv,usea,dsea,str,chm,bot,glu;
};

class c_mrst {
 private:
  static double xx[nx+1];
  static double qq[nq+1];
  double c[np+1][nx][nq][5][5]; //coefficients used for interpolation
 public:
  double * table[8];
  struct s_partoncontent cont;
  void mrst(double x,double q);
  double mrstparton(int parton,double x, double q);
  // The constructor (initialises the functions)
  c_mrst(string filename);
};

int locate(double xx[],int n,double x)
  // returns an integer j such that x lies inbetween xx[j] and xx[j+1]. unit
  // offset of increasing ordered array xx assumed. n is the length of the
  // array (xx[n] highest element)
{
  int ju,jm,jl(0),j;
  ju=n+1;

  while (ju-jl>1) {
    jm=(ju+jl)/2; // compute a mid point.
    if ( x>= xx[jm])
      jl=jm;
    else ju=jm;
  }
  if (x==xx[1]) j=1;
  else if (x==xx[n]) j=n-1;
  else j=jl;

  return j;
}


double polderivative(double x1,double x2,double x3,double y1, double y2, double y3)
  // returns the estimate of the derivative at x2 obtained by a polynomial 
  // interpolation using the three points (x_i,y_i)
{
  return (x3*x3*(y1-y2)-2.0*x2*(x3*(y1-y2)+x1*(y2-y3))+x2*x2*(y1-y3)+x1*x1*(y2-y3))/((x1-x2)*(x1-x3)*(x2-x3));
}

double c_mrst::xx[nx+1];

double c_mrst::qq[nq+1];

c_mrst::c_mrst(string filename)
  // The constructor
  // This will initialise the mrst functions automatically
{
  ifstream data_file;
  
  int i,n,m,k,l,j; // counters
  double dx,dq,dtemp;
  
  // Variables used for initialising c_ij array:
  double f[np+1][nx+1][nq+1];
  double f1[np+1][nx+1][nq+1];//derivative wrt.x
  double f2[np+1][nx+1][nq+1];//derivative wrt.qq
  double f12[np+1][nx+1][nq+1];//cross derivative

  int wt[16][16]={1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
		  0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,
		  -3,0,0,3,0,0,0,0,-2,0,0,-1,0,0,0,0,
		  2,0,0,-2,0,0,0,0,1,0,0,1,0,0,0,0,
		  0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,
		  0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,
		  0,0,0,0,-3,0,0,3,0,0,0,0,-2,0,0,-1,
		  0,0,0,0,2,0,0,-2,0,0,0,0,1,0,0,1,
		  -3,3,0,0,-2,-1,0,0,0,0,0,0,0,0,0,0,
		  0,0,0,0,0,0,0,0,-3,3,0,0,-2,-1,0,0,
		  9,-9,9,-9,6,3,-3,-6,6,-6,-3,3,4,2,1,2,
		  -6,6,-6,6,-4,-2,2,4,-3,3,3,-3,-2,-1,-1,-2,
		  2,-2,0,0,1,1,0,0,0,0,0,0,0,0,0,0,
		  0,0,0,0,0,0,0,0,2,-2,0,0,1,1,0,0,
		  -6,6,-6,6,-3,-3,3,3,-4,4,2,-2,-2,-2,-1,-1,
		  4,-4,4,-4,2,2,-2,-2,2,-2,-2,2,1,1,1,1};
  double xxd,d1d2,cl[16],x[16],d1,d2,y[5],y1[5],y2[5],y12[5];
  //////////

  // Initialising the x array common to all members of the class
  // Unfortunately, ANSI-C++ does not allow a initialisation of a 
  // member array. So here we go...

  xx[0]=0;
  xx[1]=1E-5;
  xx[2]=2E-5;
  xx[3]=4E-5;
  xx[4]=6E-5;
  xx[5]=8E-5;
  xx[6]=1E-4;
  xx[7]=2E-4;
  xx[8]=4E-4;
  xx[9]=6E-4;
  xx[10]=8E-4;
  xx[11]=1E-3;
  xx[12]=2E-3;
  xx[13]=4E-3;
  xx[14]=6E-3;
  xx[15]=8E-3;
  xx[16]=1E-2;
  xx[17]=1.4E-2;
  xx[18]=2E-2;
  xx[19]=3E-2;
  xx[20]=4E-2;
  xx[21]=6E-2;
  xx[22]=8E-2;
  xx[23]=.1E0;
  xx[24]=.125E0;
  xx[25]=.15E0;
  xx[26]=.175E0;
  xx[27]=.2E0;
  xx[28]=.225E0;
  xx[29]=.25E0;
  xx[30]=.275E0;
  xx[31]=.3E0;
  xx[32]=.325E0;
  xx[33]=.35E0;
  xx[34]=.375E0;
  xx[35]=.4E0;
  xx[36]=.425E0;
  xx[37]=.45E0;
  xx[38]=.475E0;
  xx[39]=.5E0;
  xx[40]=.525E0;
  xx[41]=.55E0;
  xx[42]=.575E0;
  xx[43]=.6E0;
  xx[44]=.65E0;
  xx[45]=.7E0;
  xx[46]=.75E0;
  xx[47]=.8E0;
  xx[48]=.9E0;
  xx[49]=1E0;
  
  // ditto for qq array
  
  qq[0]=0;
  qq[1]=1.25;
  qq[2]=1.5E0;
  qq[3]=2E0;
  qq[4]=2.5E0;
  qq[5]=3.2E0;
  qq[6]=4E0;
  qq[7]=5E0;
  qq[8]=6.4E0;
  qq[9]=8E0;
  qq[10]=1E1;
  qq[11]=1.2E1;
  qq[12]=1.8E1;
  qq[13]=2.6E1;
  qq[14]=4E1;
  qq[15]=6.4E1;
  qq[16]=1E2;
  qq[17]=1.6E2;
  qq[18]=2.4E2;
  qq[19]=4E2;
  qq[20]=6.4E2;
  qq[21]=1E3;
  qq[22]=1.8E3;
  qq[23]=3.2E3;
  qq[24]=5.6E3;
  qq[25]=1E4;
  qq[26]=1.8E4;
  qq[27]=3.2E4;
  qq[28]=5.6E4;
  qq[29]=1E5;
  qq[30]=1.8E5;
  qq[31]=3.2E5;
  qq[32]=5.6E5;
  qq[33]=1E6;
  qq[34]=1.8E6;
  qq[35]=3.2E6;
  qq[36]=5.6E6;
  qq[37]=1E7;
  
  // The name of the file to open is stored in 'filename'

  data_file.open(filename.c_str());
  
  if (data_file.bad()) {
    cerr << "Error opening " << filename << "\n";
    exit (-1);
  }
  for (n=1;n<=nx-1;n++) 
    for (m=1;m<=nq;m++) {
      // notation: 1=uval 2=val; 3=glue 4=usea 5=chm 6=str 7=btm 8=dsea
      data_file >> f[1][n][m];
      data_file >> f[2][n][m];
      data_file >> f[3][n][m];
      data_file >> f[4][n][m];
      data_file >> f[5][n][m];
      data_file >> f[7][n][m];
      data_file >> f[6][n][m];
      data_file >> f[8][n][m];
      
      if (data_file.eof()) {
	cerr << "Error reading " << filename << "\n";
	exit (-1);
      }
    }      
  data_file >> dtemp;
  if (!data_file.eof()) {
    cerr << "Error reading " << filename << "\n";
    exit (-1);
  }
  
  
  // close the datafile
  data_file.close();
  
  // zero remaining elements
  
  for (i=1;i<=np;i++)
    for (m=1;m<=nq;m++)
      f[i][nx][m]=0.0;
  // NEW BIT STARTS HERE
  
  // Set up the new array in log x and log qsq
  for (i=1;i<=nx;i++) {
    xx[i]=log10(xx[i]);
    cout <<xx[i]<<endl;
  }
  for (m=1;m<=nq;m++)
    qq[m]=log10(qq[m]);

  // Now calculate the derivatives used for bicubic interpolation
  for (i=1;i<=np;i++) {
    // Start by calculating the first x derivatives
    // along the first x value
    dx=xx[2]-xx[1];
    for (m=1;m<=nq;m++)
      f1[i][1][m]=(f[i][2][m]-f[i][1][m])/dx;
    // The along the rest (up to the last)
    for (k=2;k<nx;k++) {
      for (m=1;m<=nq;m++) {
	f1[i][k][m]=polderivative(xx[k-1],xx[k],xx[k+1],f[i][k-1][m],f[i][k][m],f[i][k+1][m]);
      }
    }
    // Then for the last column
    dx=xx[nx]-xx[nx-1];
    for (m=1;m<=nq;m++)
      f1[i][nx][m]=(f[i][nx][m]-f[i][nx-1][m])/dx;
    

    if ((i!=5)&&(i!=7)) {
      // then calculate the qq derivatives
      // Along the first qq value
      dq=qq[2]-qq[1];
      for (k=1;k<=nx;k++)
	f2[i][k][1]=(f[i][k][2]-f[i][k][1])/dq;
      // The rest up to the last qq value
      for (m=2;m<nq;m++) {
	for (k=1;k<=nx;k++)
	  f2[i][k][m]=polderivative(qq[m-1],qq[m],qq[m+1],f[i][k][m-1],f[i][k][m],f[i][k][m+1]);
      }
      // then for the last row
      dq=qq[nq]-qq[nq-1];
      for (k=1;k<=nx;k++)
	f2[i][k][nq]=(f[i][k][nq]-f[i][k][nq-1])/dq;
      
      // Now, calculate the cross derivatives.
      // Calculate these as x-derivatives of the y-derivatives
      // ?? Could be improved by taking the average between dxdy and dydx ??
      
      // Start by calculating the first x derivatives
      // along the first x value
      dx=xx[2]-xx[1];
      for (m=1;m<=nq;m++)
	f12[i][1][m]=(f2[i][2][m]-f2[i][1][m])/dx;
      // The along the rest (up to the last)
      for (k=2;k<nx;k++) {
	for (m=1;m<=nq;m++)
	  f12[i][k][m]=polderivative(xx[k-1],xx[k],xx[k+1],f2[i][k-1][m],f2[i][k][m],f2[i][k+1][m]);
      }
      // Then for the last column
      dx=xx[nx]-xx[nx-1];
      for (m=1;m<=nq;m++)
	f12[i][nx][m]=(f2[i][nx][m]-f2[i][nx-1][m])/dx;
    }
    
    if (i==5) {
      // zero all elements below the charm threshold
      for (m=1;m<=nqc0;m++) 
	for (k=1;k<=nx;k++)
	  f2[i][k][m]=0.0; 

      // then calculate the qq derivatives
      // Along the first qq value above the threshold
      dq=qq[nqc0+1]-qq[nqc0];
      for (k=1;k<=nx;k++)
	f2[i][k][m]=(f[i][k][m+1]-f[i][k][m])/dq;

      // The rest up to the last qq value
      for (m=nqc0+1;m<nq;m++) {
	for (k=1;k<=nx;k++)
	  f2[i][k][m]=polderivative(qq[m-1],qq[m],qq[m+1],f[i][k][m-1],f[i][k][m],f[i][k][m+1]);
      }
      // then for the last row
      dq=qq[nq]-qq[nq-1];
      for (k=1;k<=nx;k++)
	f2[i][k][nq]=(f[i][k][nq]-f[i][k][nq-1])/dq;
      
      // Now, calculate the cross derivatives.
      // Calculate these as x-derivatives of the y-derivatives
      // ?? Could be improved by taking the average between dxdy and dydx ??

      dx=xx[2]-xx[1];
      for (m=1;m<=nq;m++)
	f12[i][1][m]=(f2[i][2][m]-f2[i][1][m])/dx;
      // The along the rest (up to the last)
      for (k=2;k<nx;k++) {
	for (m=1;m<=nq;m++)
	  f12[i][k][m]=polderivative(xx[k-1],xx[k],xx[k+1],f2[i][k-1][m],f2[i][k][m],f2[i][k+1][m]);
      }
      // Then for the last column
      dx=xx[nx]-xx[nx-1];
      for (m=1;m<=nq;m++)
	f12[i][nx][m]=(f2[i][nx][m]-f2[i][nx-1][m])/dx;
    }

    if (i==7) {
      // zero all elements below the bottom threshold
      for (m=1;m<=nqb0;m++) 
	for (k=1;k<=nx;k++)
	  f2[i][k][m]=0.0; 

      // then calculate the qq derivatives
      // Along the first qq value above the threshold
      dq=qq[nqb0+1]-qq[nqb0];
      for (k=1;k<=nx;k++)
	f2[i][k][m]=(f[i][k][m+1]-f[i][k][m])/dq;

      // The rest up to the last qq value
      for (m=nqb0+1;m<nq;m++) {
	for (k=1;k<=nx;k++)
	  f2[i][k][m]=polderivative(qq[m-1],qq[m],qq[m+1],f[i][k][m-1],f[i][k][m],f[i][k][m+1]);
      }
      // then for the last row
      dq=qq[nq]-qq[nq-1];
      for (k=1;k<=nx;k++)
	f2[i][k][nq]=(f[i][k][nq]-f[i][k][nq-1])/dq;
      
      // Now, calculate the cross derivatives.
      // Calculate these as x-derivatives of the y-derivatives
      // ?? Could be improved by taking the average between dxdy and dydx ??

      dx=xx[2]-xx[1];
      for (m=1;m<=nq;m++)
	f12[i][1][m]=(f2[i][2][m]-f2[i][1][m])/dx;
      // The along the rest (up to the last)
      for (k=2;k<nx;k++) {
	for (m=1;m<=nq;m++)
	  f12[i][k][m]=polderivative(xx[k-1],xx[k],xx[k+1],f2[i][k-1][m],f2[i][k][m],f2[i][k+1][m]);
      }
      // Then for the last column
      dx=xx[nx]-xx[nx-1];
      for (m=1;m<=nq;m++)
	f12[i][nx][m]=(f2[i][nx][m]-f2[i][nx-1][m])/dx;
    }
	
	
    // Now calculate the coefficients c_ij
    for (n=1;n<=nx-1;n++) {
      for (m=1;m<=nq-1;m++) {
	d1=xx[n+1]-xx[n];
	d2=qq[m+1]-qq[m];
	d1d2=d1*d2;
	
	y[1]=f[i][n][m];
	y[2]=f[i][n+1][m];
	y[3]=f[i][n+1][m+1];
	y[4]=f[i][n][m+1];
	
	y1[1]=f1[i][n][m];
	y1[2]=f1[i][n+1][m];
	y1[3]=f1[i][n+1][m+1];
	y1[4]=f1[i][n][m+1];
	
	y2[1]=f2[i][n][m];
	y2[2]=f2[i][n+1][m];
	y2[3]=f2[i][n+1][m+1];
	y2[4]=f2[i][n][m+1];
	
	y12[1]=f12[i][n][m];
	y12[2]=f12[i][n+1][m];
	y12[3]=f12[i][n+1][m+1];
	y12[4]=f12[i][n][m+1];
	
	for (k=1;k<=4;k++) {
	  x[k-1]=y[k];
	  x[k+3]=y1[k]*d1;
	  x[k+7]=y2[k]*d2;
	  x[k+11]=y12[k]*d1d2;
	}
	
	
	for (l=0;l<=15;l++) {
	  xxd=0.0;
	  for (k=0;k<=15;k++) xxd+= wt[l][k]*x[k];
	  cl[l]=xxd;
	}
	
	l=0;
	for (k=1;k<=4;k++) 
	  for (j=1;j<=4;j++) c[i][n][m][k][j]=cl[l++];
      } //m
    } //n
    // NEW BIT ENDS HERE
  } // i


  table[0]=&cont.upv;
  table[1]=&cont.dnv;
  table[2]=&cont.usea;
  table[3]=&cont.dsea;
  table[4]=&cont.str;
  table[5]=&cont.chm;
  table[6]=&cont.bot;
  table[7]=&cont.glu;

}

void c_mrst::mrst(double x,double q)
  // Updates the parton content 
{
  double qsq;
  double xxx,g[np+1];
  int i,n,m,l;
  double t,u;
  
  qsq=q*q;
  

  if (x<xmin) {
    x=xmin;
    cerr << "   WARNING:  x   VALUE IS OUT OF RANGE (TOO LOW)\n";
  }
  else if (x>xmax) {
    x=xmax;
    cerr << "   WARNING:  x   VALUE IS OUT OF RANGE (TOO HIGH)\n";
  }

  if (qsq<qsqmin) { 
    qsq=qsqmin;
    cerr << "   WARNING:  Q^2 VALUE IS OUT OF RANGE (TOO LOW)\n";
  }
  else if (qsq>qsqmax) {
    qsq=qsqmax;
    cerr << "   WARNING:  Q^2 VALUE IS OUT OF RANGE (TOO HIGH)\n";
  }
  
  xxx=x;
  
  // interpolation in logx, log qsq:
  xxx=log10(xxx);
  qsq=log10(qsq);


  // NEW BIT STARTS HERE
  n=locate(xx,nx,xxx);
  m=locate(qq,nq,qsq);

  for (i=1;i<=np;i++) {

    t=(xxx-xx[n])/(xx[n+1]-xx[n]);
    u=(qsq-qq[m])/(qq[m+1]-qq[m]);

    g[i]=0.0;
    for (l=4;l>=1;l--) {
      g[i]=t*g[i]+((c[i][n][m][l][4]*u+c[i][n][m][l][3])*u+c[i][n][m][l][2])*u+c[i][n][m][l][1];
    }

  }
  // NEW BIT ENDS HERE
  cont.upv=g[1];
  cont.dnv=g[2];
  cont.usea=g[4];
  cont.dsea=g[8];
  cont.str=g[6];
  cont.chm=g[5];
  cont.glu=g[3];
  cont.bot=g[7];
}

double c_mrst::mrstparton(int parton,double x, double q)
  // Returns the PDf value for parton of type 'parton' at x,q
{
  double qsq;
  double xxx,g;
  int n,m,l;
  double t,u;
  
  qsq=q*q;
  

  if (x<xmin) {
    x=xmin;
    cerr << "   WARNING:  x   VALUE IS OUT OF RANGE (TOO LOW)\n";
  }
  else if (x>xmax) {
    x=xmax;
    cerr << "   WARNING:  x   VALUE IS OUT OF RANGE (TOO HIGH)\n";
  }

  if (qsq<qsqmin) { 
    qsq=qsqmin;
    cerr << "   WARNING:  Q^2 VALUE IS OUT OF RANGE (TOO LOW)\n";
  }
  else if (qsq>qsqmax) {
    qsq=qsqmax;
    cerr << "   WARNING:  Q^2 VALUE IS OUT OF RANGE (TOO HIGH)\n";
  }
  
  xxx=x;
  
  // interpolation in logx, log qsq:
  xxx=log10(xxx);
  qsq=log10(qsq);


  // NEW BIT STARTS HERE
  n=locate(xx,nx,xxx);
  m=locate(qq,nq,qsq);
  
  //  for (i=1;i<=np;i++) {
  
  t=(xxx-xx[n])/(xx[n+1]-xx[n]);
  u=(qsq-qq[m])/(qq[m+1]-qq[m]);
  
  g=0.0;
  for (l=4;l>=1;l--) {
    g=t*g+((c[parton][n][m][l][4]*u+c[parton][n][m][l][3])*u+c[parton][n][m][l][2])*u+c[parton][n][m][l][1];
  }
  
  return g;
}

//////////////////////////////////////////////////////////////////////
// Now the functions that can be called from Mathematica            //
//////////////////////////////////////////////////////////////////////

// define a global c_mrst class member:
//class c_mrst parton("lo2002.dat");
  class c_mrst parton("mrst2002nlo.dat");
//  class c_event event;

double f(int n, double x, double q)
  // Returns the x*pdf for parton n evaluated at Bjorken x and scale Q
{
  return parton.mrstparton(n,x,q);
}

int main(int argc, char *argv[]) {
  return MLMain(argc, argv);
  //  cout << f(1,0.2,20)<<endl;
}
