#include "SampleAnalyzer/User/Analyzer/Recasting_SUS16041.h"
using namespace MA5;
using namespace std;

// -- user-defined functions -- //
double Calc_MiniIso( const RecLeptonFormat* lepton, const EventFormat& event );
double Calc_PtRatio( const RecLeptonFormat* lepton, const EventFormat& event );
double Calc_PtRel( const RecLeptonFormat* lepton, const EventFormat& event );
std::vector<const RecJetFormat*> Find_MatchedJets( const RecLeptonFormat* lepton, const EventFormat& event );

bool Find_DiLeptonPair( std::vector<const RecLeptonFormat*> &baseLeptons, vector<double> &vec_M, vector<double> &vec_MT, const EventFormat& event );
double Calc_MT( const RecLeptonFormat* lepton, const EventFormat& event );

double Check_FlavorComposition( vector<const RecLeptonFormat*> baseLeptons );


// -----------------------------------------------------------------------------
// Initialize
// function called one time at the beginning of the analysis
// -----------------------------------------------------------------------------
bool Recasting_SUS16041::Initialize(const MA5::Configuration& cfg, const std::map<std::string,std::string>& parameters)
{
  cout << "BEGIN Initialization" << endl;

  nEvent = 0;
  SumWeight = 0;
  // Flag_Verbose = true;
  Flag_Verbose = false;
  // -- define signal regions -- //
  Manager()->AddRegionSelection("SR15b_onZ_HT600toInf_MET150to300_MT120toInf");
  Manager()->AddRegionSelection("SR15b_offZ_HT600toInf_MET150to300_MT120toInf");

  Manager()->AddRegionSelection("SR16a_onZ_MET300toInf_MT0to120");
  Manager()->AddRegionSelection("SR16a_offZ_MET300toInf_MT0to120");

  Manager()->AddRegionSelection("SR16b_onZ_MET300toInf_MT120toInf");
  Manager()->AddRegionSelection("SR16b_offZ_MET300toInf_MT120toInf");

  Manager()->AddRegionSelection("SSR1_offZ_nbjet0to2_HT200toInf_MET_250toInf_MT120toInf");
  Manager()->AddRegionSelection("SSR3_onZ_nbjet0to2_HT200toInf_MET_250toInf_MT120toInf");

  Manager()->AddRegionSelection("SSR2_offZ_nbjet3toInf_HT60toInf_MET_50toInf_MT120toInf");
  Manager()->AddRegionSelection("SSR4_onZ_nbjet3toInf_HT60toInf_MET_50toInf_MT120toInf");

  std::string SR_All[] =
  {
    "SR15b_onZ_HT600toInf_MET150to300_MT120toInf",
    "SR15b_offZ_HT600toInf_MET150to300_MT120toInf",
    "SR16a_onZ_MET300toInf_MT0to120",
    "SR16a_offZ_MET300toInf_MT0to120",
    "SR16b_onZ_MET300toInf_MT120toInf",
    "SR16b_offZ_MET300toInf_MT120toInf",
    "SSR1_offZ_nbjet0to2_HT200toInf_MET_250toInf_MT120toInf",
    "SSR3_onZ_nbjet0to2_HT200toInf_MET_250toInf_MT120toInf",
    "SSR2_offZ_nbjet3toInf_HT60toInf_MET_50toInf_MT120toInf",
    "SSR4_onZ_nbjet3toInf_HT60toInf_MET_50toInf_MT120toInf"
  };

  std::string SR_onZ[] = 
  {
    "SR15b_onZ_HT600toInf_MET150to300_MT120toInf",
    "SR16a_onZ_MET300toInf_MT0to120",
    "SR16b_onZ_MET300toInf_MT120toInf",
    "SSR3_onZ_nbjet0to2_HT200toInf_MET_250toInf_MT120toInf",
    "SSR4_onZ_nbjet3toInf_HT60toInf_MET_50toInf_MT120toInf"
  };

  std::string SR_offZ[] = 
  {
    "SR15b_offZ_HT600toInf_MET150to300_MT120toInf",
    "SR16a_offZ_MET300toInf_MT0to120",
    "SR16b_offZ_MET300toInf_MT120toInf",
    "SSR1_offZ_nbjet0to2_HT200toInf_MET_250toInf_MT120toInf",
    "SSR2_offZ_nbjet3toInf_HT60toInf_MET_50toInf_MT120toInf"
  };

  std::string SR_nbjet0to2[] = 
  {
    "SSR1_offZ_nbjet0to2_HT200toInf_MET_250toInf_MT120toInf",
    "SSR3_onZ_nbjet0to2_HT200toInf_MET_250toInf_MT120toInf"
  };

  std::string SR_nbjet3toInf[] = 
  {
    "SSR2_offZ_nbjet3toInf_HT60toInf_MET_50toInf_MT120toInf",
    "SSR4_onZ_nbjet3toInf_HT60toInf_MET_50toInf_MT120toInf"
  };

  std::string SR_MET150to300[] = 
  {
    "SR15b_onZ_HT600toInf_MET150to300_MT120toInf",
    "SR15b_offZ_HT600toInf_MET150to300_MT120toInf"
  };

  std::string SR_MET300toInf[] = 
  {
    "SR16a_onZ_MET300toInf_MT0to120",
    "SR16a_offZ_MET300toInf_MT0to120",
    "SR16b_onZ_MET300toInf_MT120toInf",
    "SR16b_offZ_MET300toInf_MT120toInf"
  };

  std::string SR_MET250toInf[] = 
  {
    "SSR1_offZ_nbjet0to2_HT200toInf_MET_250toInf_MT120toInf",
    "SSR3_onZ_nbjet0to2_HT200toInf_MET_250toInf_MT120toInf"
  };

  std::string SR_MET50toInf[] = 
  {
    "SSR2_offZ_nbjet3toInf_HT60toInf_MET_50toInf_MT120toInf",
    "SSR4_onZ_nbjet3toInf_HT60toInf_MET_50toInf_MT120toInf"
  };

  std::string SR_HT600toInf[] = 
  {
    "SR15b_onZ_HT600toInf_MET150to300_MT120toInf",
    "SR15b_offZ_HT600toInf_MET150to300_MT120toInf"
  };

  std::string SR_HT200toInf[] = 
  {
    "SSR1_offZ_nbjet0to2_HT200toInf_MET_250toInf_MT120toInf",
    "SSR3_onZ_nbjet0to2_HT200toInf_MET_250toInf_MT120toInf"
  };

  std::string SR_HT60toInf[] = 
  {
    "SSR2_offZ_nbjet3toInf_HT60toInf_MET_50toInf_MT120toInf",
    "SSR4_onZ_nbjet3toInf_HT60toInf_MET_50toInf_MT120toInf"
  };

  std::string SR_MT0to120[] = 
  {
    "SR16a_onZ_MET300toInf_MT0to120",
    "SR16a_offZ_MET300toInf_MT0to120",
  };

  std::string SR_MT120toInf[] = 
  {
    "SR15b_onZ_HT600toInf_MET150to300_MT120toInf",
    "SR15b_offZ_HT600toInf_MET150to300_MT120toInf",
    "SR16b_onZ_MET300toInf_MT120toInf",
    "SR16b_offZ_MET300toInf_MT120toInf",
    "SSR1_offZ_nbjet0to2_HT200toInf_MET_250toInf_MT120toInf",
    "SSR2_offZ_nbjet3toInf_HT60toInf_MET_50toInf_MT120toInf",
    "SSR3_onZ_nbjet0to2_HT200toInf_MET_250toInf_MT120toInf",
    "SSR4_onZ_nbjet3toInf_HT60toInf_MET_50toInf_MT120toInf",
  };

  Manager()->AddCut( "nGoodLepton >= 3", SR_All );
  Manager()->AddCut( "nJet >= 2", SR_All );
  Manager()->AddCut( "MET > 50 (70) GeV", SR_All );

  Manager()->AddCut("onZ", SR_onZ);
  Manager()->AddCut("offZ", SR_offZ);

  Manager()->AddCut("nbjet <= 2", SR_nbjet0to2);
  Manager()->AddCut("nbjet >= 3", SR_nbjet3toInf);

  Manager()->AddCut("150 < MET < 300 GeV", SR_MET150to300 );
  Manager()->AddCut("MET > 300 GeV", SR_MET300toInf );
  Manager()->AddCut("MET > 250 GeV", SR_MET250toInf );
  Manager()->AddCut("MET > 50 GeV", SR_MET50toInf );

  Manager()->AddCut("HT > 600 GeV", SR_HT600toInf );
  Manager()->AddCut("HT > 200 GeV", SR_HT200toInf );
  Manager()->AddCut("HT > 60 GeV", SR_HT60toInf );

  Manager()->AddCut("MT < 120 GeV", SR_MT0to120 );
  Manager()->AddCut("MT > 120 GeV", SR_MT120toInf );


  // -- histogram: for validation -- //
  Manager()->AddHisto("MiniIso_Elec", 1000, 0, 100 );
  Manager()->AddHisto("PtRatio_Elec", 1000, 0, 100 );
  Manager()->AddHisto("PtRel_Elec", 1000, 0, 100 );
  Manager()->AddHisto("MiniIso_Mu", 1000, 0, 100 );
  Manager()->AddHisto("PtRatio_Mu", 1000, 0, 100 );
  Manager()->AddHisto("PtRel_Mu", 1000, 0, 100 );

  // -- histograms: after baseline selection -- //
  Manager()->AddHisto("# jets (on-Z)", 6, 2, 8 );
  Manager()->AddHisto("# b-jets (on-Z)", 5, 0, 5 );
  Manager()->AddHisto("HT (on-Z)", 9, 60, 660 );
  Manager()->AddHisto("MT (on-Z)", 10, 0, 200 );
  Manager()->AddHisto("MET (on-Z)", 18, 50, 500 );
  Manager()->AddHisto("Leading lepton pT (on-Z)", 14, 20, 300 );
  Manager()->AddHisto("Sub-leading lepton pT (on-Z)", 9, 10, 100 );
  Manager()->AddHisto("Trailing lepton pT (on-Z)", 9, 10, 100 );
  Manager()->AddHisto("Flavor composition (on-Z)", 5, 0, 5 );

  Manager()->AddHisto("# jets (off-Z)", 6, 2, 8 );
  Manager()->AddHisto("# b-jets (off-Z)", 5, 0, 5 );
  Manager()->AddHisto("HT (off-Z)", 9, 60, 660 );
  Manager()->AddHisto("MT (off-Z)", 10, 0, 200 );
  Manager()->AddHisto("MET (off-Z)", 18, 50, 500 );
  Manager()->AddHisto("Leading lepton pT (off-Z)", 14, 20, 300 );
  Manager()->AddHisto("Sub-leading lepton pT (off-Z)", 9, 10, 100 );
  Manager()->AddHisto("Trailing lepton pT (off-Z)", 9, 10, 100 );
  Manager()->AddHisto("Flavor composition (off-Z)", 5, 0, 5 );

  cout << "END   Initialization" << endl;

  return true;
}

// -----------------------------------------------------------------------------
// Finalize
// function called one time at the end of the analysis
// -----------------------------------------------------------------------------
void Recasting_SUS16041::Finalize(const SampleFormat& summary, const std::vector<SampleFormat>& files)
{
  cout << "BEGIN Finalization" << endl;
  // saving histos
  cout << "# total events processed: " << nEvent << endl;
  cout << "\tSum of weights: " << SumWeight << endl;

  cout << "END   Finalization" << endl;
}

// -----------------------------------------------------------------------------
// Execute
// function called each time one event is read
// -----------------------------------------------------------------------------
bool Recasting_SUS16041::Execute(SampleFormat& sample, const EventFormat& event)
{
  if(event.rec()!=0)
  {
    double myEventWeight;
    if(Configuration().IsNoEventWeight()) myEventWeight=1.;
    else if(event.mc()->weight()!=0.) myEventWeight = event.mc()->weight();
    else
    {
      WARNING << "Found one event with a zero weight. Skipping..." << endmsg;
      return false;
    }

    myEventWeight = 1.0; // -- do not use weight! ... we will use only LO events (no negative weights) -- //
    Manager()->InitializeForNewEvent(myEventWeight);

    // -- count # events & sum of weights -- //
    nEvent++;
    SumWeight += myEventWeight;

    if( Flag_Verbose )
    {
      cout << "##### " << nEvent << "th event #####" << endl;
      cout << "Sum of weights: " << SumWeight << endl;
      cout << "\tmyEventWeight: " << myEventWeight << endl;
    }

    std::vector<const RecLeptonFormat*> baseElectrons, baseMuons, baseLeptons;
    std::vector<const RecJetFormat*> baseJets;

    // -- baseline jets (anti-kt, r=0.4 simulated @ Delphes) -- //
    double HT = 0; // -- sum of jet pT for jets with pT > 30 GeV -- //
    int nbjet = 0;
    for(unsigned int ii=0; ii<event.rec()->jets().size(); ii++)
    {
      const RecJetFormat *CurrentJet = &(event.rec()->jets()[ii]);
      double pt = CurrentJet->pt();
      double abseta = fabs(CurrentJet->eta());

      if(pt > 30.0 && abseta < 2.4)
        baseJets.push_back(CurrentJet);

      // -- HT -- //
      if( pt > 30.0 ) HT = HT + pt;

      // -- # b-tagged jets -- //
      if( CurrentJet->btag() && pt > 25.0 && abseta < 2.4 ) nbjet++;
    }

    // -- base electron selection -- //
    for(unsigned int ii=0; ii<event.rec()->electrons().size(); ii++)
    {
      const RecLeptonFormat *CurrentElectron = &(event.rec()->electrons()[ii]);
      // double pt = CurrentElectron->pt();
      double abseta = fabs(CurrentElectron->eta());
      double MiniIso = Calc_MiniIso(CurrentElectron, event);
      double PtRatio = Calc_PtRatio(CurrentElectron, event);
      double PtRel = Calc_PtRel(CurrentElectron, event);

      Manager()->FillHisto( "MiniIso_Elec", MiniIso );
      Manager()->FillHisto( "PtRatio_Elec", PtRatio );
      Manager()->FillHisto( "PtRel_Elec", PtRel );

      if( Flag_Verbose )
        cout << "[" << ii << "th electron] MiniIso: " << MiniIso << ", PtRatio: " << PtRatio << ", PtRel: " << PtRel << endl;

      if( abseta < 2.5 && MiniIso < 0.12 && (PtRatio > 0.76 || PtRel > 7.2) )
      {
        baseElectrons.push_back(CurrentElectron);
        baseLeptons.push_back(CurrentElectron);
      }
    }

    // -- base muon selection -- //
    for(unsigned int ii=0; ii<event.rec()->muons().size(); ii++)
    {
      const RecLeptonFormat *CurrentMuon = &(event.rec()->muons()[ii]);
      // double pt = CurrentMuon->pt();
      double abseta = fabs(CurrentMuon->eta());
      double MiniIso = Calc_MiniIso(CurrentMuon, event);
      double PtRatio = Calc_PtRatio(CurrentMuon, event);
      double PtRel = Calc_PtRel(CurrentMuon, event);

      Manager()->FillHisto( "MiniIso_Mu", MiniIso );
      Manager()->FillHisto( "PtRatio_Mu", PtRatio );
      Manager()->FillHisto( "PtRel_Mu", PtRel );

      if( Flag_Verbose )
        cout << "[" << ii << "th muon] MiniIso: " << MiniIso << ", PtRatio: " << PtRatio << ", PtRel: " << PtRel << endl;

      if( abseta < 2.4 && MiniIso < 0.16 && (PtRatio > 0.69 || PtRel > 6.0) )
      {
        baseMuons.push_back(CurrentMuon);
        baseLeptons.push_back(CurrentMuon);
      }
    }

    // -- test whether this event contains at least 3 leptons & passing kinematic cuts & dilepton mass > 12 GeV -- //
    bool Flag_PassAcc = false;
    int nBaseLepton = (int)baseLeptons.size();
    if( baseLeptons.size() >= 3 )
    {
      // -- test pt cuts -- //
      bool Flag_PassPtCut = false;
      SORTER->sort( baseLeptons );

      double PtCut_1st = 0;
      double PtCut_2nd = 0;
      double PtCut_3rd = 0;
      if( HT < 300 )
      {
        PtCut_1st = 25;

        // -- sub-leading cut: depend on the lepton flavor -- // 
        if( baseLeptons[1]->isMuon() ) PtCut_2nd = 10;
        if( baseLeptons[1]->isElectron() ) PtCut_2nd = 15;

        PtCut_3rd = 10;
      }
      else if( HT > 300 )
      {
        //-- leading cut: depend on the lepton flavor -- // 
        if( baseLeptons[0]->isMuon() ) PtCut_1st = 10;
        if( baseLeptons[0]->isElectron() ) PtCut_1st = 15;

        //-- leading cut: depend on the lepton flavor -- // 
        if( baseLeptons[1]->isMuon() ) PtCut_2nd = 10;
        if( baseLeptons[1]->isElectron() ) PtCut_2nd = 15;

        PtCut_3rd = 10;
      }

      if( baseLeptons[0]->pt() > PtCut_1st &&
          baseLeptons[1]->pt() > PtCut_2nd &&
          baseLeptons[2]->pt() > PtCut_3rd ) Flag_PassPtCut = true;

      // -- test the dilepton(OS, SF) mass of all possible pairs: M(ll) > 12 GeV -- //
      bool Flag_OSMass = true; // -- it will be false if one pair below M < 12 GeV is found -- //

      vector<double> vec_M;
      vector<double> vec_MT_temp;
      bool Flag_PairFound = Find_DiLeptonPair( baseLeptons, vec_M, vec_MT_temp, event );
      if( Flag_PairFound )
      {
        for( const auto& M : vec_M )
        {
          if( M <= 12.0 ) // -- reject the event with dilepton pair mass less than 12 GeV -- //
          {
            Flag_OSMass = false;
            break;
          }
        }
      }

      if( Flag_PassPtCut && Flag_OSMass ) Flag_PassAcc = true;
    }

    // -- test on-Z or off-Z -- //
    bool Flag_onZ = false;
    double MT = 0; // -- will be decided in on-Z test (this value is related to on-Z condition) -- //
    vector<double> vec_M;
    vector<double> vec_MT;
    bool Flag_PairFound = Find_DiLeptonPair( baseLeptons, vec_M, vec_MT, event );
    
    double MT_NotInv_Smallest = 999; // -- only used when on-Z case -- //
    if( Flag_PairFound )
    {
      int nPair = (int)vec_M.size();
      for(int i_pair=0; i_pair<nPair; i_pair++)
      {
        double M_pair = vec_M[i_pair];
        double MT_NotInv_temp = vec_MT[i_pair];

        // cout << i_pair << "th pair: M = " << M_pair << ", MT = " << MT_NotInv_temp << ", HT = " << HT << endl;

        if( M_pair > 76 && M_pair < 106 )
        {
          Flag_onZ = true;
          if( MT_NotInv_temp < MT_NotInv_Smallest )
            MT_NotInv_Smallest = MT_NotInv_temp;
        }
      }
    }

    // -- assign MT of this event -- //
    if( Flag_onZ )
      MT = MT_NotInv_Smallest;
    else // -- off-Z: take the smallest MT value -- //
    {
      double MT_Smallest = 999;
      for(int i_lep=0; i_lep<nBaseLepton; i_lep++)
      {
        double MT_temp = Calc_MT(baseLeptons[i_lep], event);
        if( MT_temp < MT_Smallest )
          MT_Smallest = MT_temp;
      }

      MT = MT_Smallest;
    }

    if( Flag_Verbose )
      cout << "MT: "<< MT << endl;

    // -- apply cuts following cut flow order-- //
    double MET = event.rec()->MET().pt();
    unsigned int nJet = baseJets.size();

    if( Flag_Verbose )
      cout << "MET: "<< MET << endl;

    if(!Manager()->ApplyCut(Flag_PassAcc, "nGoodLepton >= 3"))
      return true;

    if(!Manager()->ApplyCut(nJet >= 2, "nJet >= 2"))
      return true;

    double METCut = Flag_onZ ? 70 : 50;
    if(!Manager()->ApplyCut(MET > METCut, "MET > 50 (70) GeV"))
      return true;

    if( Flag_onZ )
    {
      Manager()->FillHisto("# jets (on-Z)", nJet );
      Manager()->FillHisto("# b-jets (on-Z)", nbjet );
      Manager()->FillHisto("HT (on-Z)", HT );
      Manager()->FillHisto("MT (on-Z)", MT );
      Manager()->FillHisto("MET (on-Z)", MET );
      Manager()->FillHisto("Leading lepton pT (on-Z)", baseLeptons[0]->pt() );
      Manager()->FillHisto("Sub-leading lepton pT (on-Z)", baseLeptons[1]->pt() );
      Manager()->FillHisto("Trailing lepton pT (on-Z)", baseLeptons[2]->pt() );
      double Flavor = Check_FlavorComposition( baseLeptons );
      Manager()->FillHisto("Flavor composition (on-Z)", 0.1 ); // -- 1st bin: total -- //
      Manager()->FillHisto("Flavor composition (on-Z)", Flavor );
    }
    else // -- off-Z -- //
    {
      Manager()->FillHisto("# jets (off-Z)", nJet );
      Manager()->FillHisto("# b-jets (off-Z)", nbjet );
      Manager()->FillHisto("HT (off-Z)", HT );
      Manager()->FillHisto("MT (off-Z)", MT );
      Manager()->FillHisto("MET (off-Z)", MET );
      Manager()->FillHisto("Leading lepton pT (off-Z)", baseLeptons[0]->pt() );
      Manager()->FillHisto("Sub-leading lepton pT (off-Z)", baseLeptons[1]->pt() );
      Manager()->FillHisto("Trailing lepton pT (off-Z)", baseLeptons[2]->pt() );
      double Flavor = Check_FlavorComposition( baseLeptons );
      Manager()->FillHisto("Flavor composition (off-Z)", 0.1 ); // -- 1st bin: total -- //
      Manager()->FillHisto("Flavor composition (off-Z)", Flavor );
    }

    // -- signal region -- //
    if(!Manager()->ApplyCut(Flag_onZ, "onZ"))
      return true;

    if(!Manager()->ApplyCut(!Flag_onZ, "offZ"))
      return true;

    if(!Manager()->ApplyCut(nbjet <= 2, "nbjet <= 2"))
      return true;

    if(!Manager()->ApplyCut(nbjet >= 3, "nbjet >= 3"))
      return true;

    if(!Manager()->ApplyCut(MET > 150.0 && MET < 300.0, "150 < MET < 300 GeV"))
      return true;

    if(!Manager()->ApplyCut(MET > 300.0, "MET > 300 GeV"))
      return true;

    if(!Manager()->ApplyCut(MET > 250.0, "MET > 250 GeV"))
      return true;

    if(!Manager()->ApplyCut(MET > 50.0, "MET > 50 GeV"))
      return true;

    if(!Manager()->ApplyCut(HT > 600.0, "HT > 600 GeV"))
      return true;

    if(!Manager()->ApplyCut(HT > 200.0, "HT > 200 GeV"))
      return true;

    if(!Manager()->ApplyCut(HT > 60.0, "HT > 60 GeV"))
      return true;

    if(!Manager()->ApplyCut(MT < 120.0, "MT < 120 GeV"))
      return true;

    if(!Manager()->ApplyCut(MT > 120.0, "MT > 120 GeV"))
      return true;

    // SORTER->sort( baseJets );
    // SORTER->sort(sigElectrons);
    // SORTER->sort(sigMuons);
    // SORTER->sort(sigLeptons);
    if( Flag_Verbose )
      cout << "\n" << endl;
  }

  return true;
}

double Calc_MiniIso( const RecLeptonFormat* lepton, const EventFormat& event )
{
  double RelIso_Combined = 0;
  double pt = lepton->pt();
  double dR = 10.0 / std::min( std::max(pt, 50.0), 200.0 ); 

  double ALLIso = PHYSICS->Isol->eflow->sumIsolation(lepton, event.rec(), dR, 0., IsolationEFlow::ALL_COMPONENTS);

  // RelIso_Combined = (ChargedIso + NeutralIso + PhotonIso ) / pt;
  RelIso_Combined = (ALLIso) / pt;

  // if( RelIso_Combined > 1 )
  // {
  //   double ChargedIso = PHYSICS->Isol->eflow->sumIsolation(lepton, event.rec(), dR, 0., IsolationEFlow::TRACK_COMPONENT);
  //   double NeutralIso = PHYSICS->Isol->eflow->sumIsolation(lepton, event.rec(), dR, 0., IsolationEFlow::NEUTRAL_COMPONENT);
  //   double PhotonIso = PHYSICS->Isol->eflow->sumIsolation(lepton, event.rec(), dR, 0., IsolationEFlow::PHOTON_COMPONENT);

  //   cout << "Lepton pT: " << pt << endl;
  //   cout << "dR: " << dR << endl;
  //   cout << "ChargedIso: " << ChargedIso << endl;
  //   cout << "NeutralIso: " << NeutralIso << endl;
  //   cout << "PhotonIso: " << PhotonIso << endl;
  //   cout << "ALLIso: "<< ALLIso << endl;
  //   cout << "RelIso_Combined: " << RelIso_Combined << endl;
  // }



  return RelIso_Combined;
}

double Calc_PtRatio( const RecLeptonFormat* lepton, const EventFormat& event )
{
  double PtRatio = 0;
  std::vector<const RecJetFormat*> MatchedJets = Find_MatchedJets( lepton, event );

  unsigned int nJets = MatchedJets.size();
  if( nJets == 0 ) // -- there's no matched jets -- //
    PtRatio = 1;

  else if( nJets == 1 )
    PtRatio = lepton->pt() / MatchedJets[0]->pt();
  
  else // -- more than 1 matched jets -- //
  {
    SORTER->sort( MatchedJets );
    PtRatio = lepton->pt() / MatchedJets[0]->pt(); // -- take the largest-pT jets -- //
  }

  return PtRatio;
}

double Calc_PtRel( const RecLeptonFormat* lepton, const EventFormat& event )
{
  double PtRel = 0;
  std::vector<const RecJetFormat*> MatchedJets = Find_MatchedJets( lepton, event );
  unsigned int nJets = MatchedJets.size();
  if( nJets == 0 ) // -- there's no matched jets -- //
    PtRel = 0;

  else if( nJets == 1 )
  {
    MALorentzVector P_lep = lepton->momentum();
    MALorentzVector P_jet = MatchedJets[0]->momentum();
    MALorentzVector P_axis = P_jet - P_lep;
    PtRel = P_axis.Dot(P_lep) / P_axis.Mag();
  }

  else
  {
    SORTER->sort( MatchedJets ); // -- take the jet with largest pT -- //
    MALorentzVector P_lep = lepton->momentum();
    MALorentzVector P_jet = MatchedJets[0]->momentum();
    MALorentzVector P_axis = P_jet - P_lep;
    PtRel = P_axis.Dot(P_lep) / P_axis.Mag();
  }

  return PtRel;
}

std::vector<const RecJetFormat*> Find_MatchedJets( const RecLeptonFormat* lepton, const EventFormat& event )
{
  std::vector<const RecJetFormat*> MatchedJets;

  for(unsigned int ii=0; ii<event.rec()->jets().size(); ii++)
  {
    const RecJetFormat *CurrentJet = &(event.rec()->jets()[ii]);

    double dR_lj = lepton->dr(CurrentJet);
    if( dR_lj < 0.4 && CurrentJet->pt() > 5.0 )
      MatchedJets.push_back( CurrentJet );
  }

  return MatchedJets;
}

bool Find_DiLeptonPair( std::vector<const RecLeptonFormat*> &baseLeptons, vector<double> &vec_M, vector<double> &vec_MT, const EventFormat& event )
{
  bool Flag_PairFound = false;
  int nBaseLepton = (int)baseLeptons.size();
  for(int i_lep=0; i_lep<nBaseLepton; i_lep++)
  {
    const RecLeptonFormat* lepton_ith = baseLeptons[i_lep];
    for(int j_lep=i_lep+1; j_lep<nBaseLepton; j_lep++)
    {
      const RecLeptonFormat* lepton_jth = baseLeptons[j_lep];

      bool Flag_SF = (lepton_ith->isMuon() && lepton_jth->isMuon()) || (lepton_ith->isElectron() && lepton_jth->isElectron());
      bool Flag_OS = lepton_ith->charge() != lepton_jth->charge();

      // cout << "Flag_SF: " << Flag_SF << ", Flag_OS: " << Flag_OS << endl;

      if( Flag_SF && Flag_OS )
      {
        Flag_PairFound = true;
        double M = ( lepton_ith->momentum() + lepton_jth->momentum() ).M();
        vec_M.push_back( M );

        double MT_Smallest = 999; // -- take the smallest one -- //
        for(int k_lep=0; k_lep<nBaseLepton; k_lep++) // -- iteration over the other leptons not involved with Z mass -- //
        {
          if( (k_lep != i_lep) && (k_lep != j_lep) )
          {
            double MT_temp = Calc_MT(baseLeptons[k_lep], event);
            if( MT_temp < MT_Smallest ) MT_Smallest = MT_temp;
          }
        }
        vec_MT.push_back( MT_Smallest );
      }
    } // -- end of j-th lepton iteration -- //
  } // -- end of i-th lepton iteration -- //

  return Flag_PairFound;
}

double Calc_MT( const RecLeptonFormat* lepton, const EventFormat& event )
{
  double MT = 0;
  double lep_pt = lepton->pt();
  double lep_phi = lepton->phi();
  double MET_pt = event.rec()->MET().pt();
  double MET_phi = event.rec()->MET().phi();

  MT = sqrt( 2*lep_pt*MET_pt*(1 - std::cos(lep_phi-MET_phi)) );
  return MT;
}

double Check_FlavorComposition( vector<const RecLeptonFormat*> baseLeptons )
{
  if( baseLeptons.size() < 3 )
  {
    cout << "[Check_FlavorComposition] # leptons (" << baseLeptons.size() << ") should be larger or equal to 3!" <<endl;
    return 999;
  }

  int nMuon = 0;
  int nElectron = 0;
  for(int i=0; i<3; i++)
  {
    if( baseLeptons[i]->isElectron() ) nElectron++;
    if( baseLeptons[i]->isMuon() ) nMuon++;
  }

  if( nMuon == 3 ) return 1 + 0.1; // -- 2nd bin -- //
  if( nMuon == 2 && nElectron == 1 ) return 2 + 0.1; // -- 3rd bin -- //
  if( nMuon == 1 && nElectron == 2 ) return 3 + 0.1; // -- 4th bin -- //
  if( nElectron == 3 ) return 4 + 0.1; // -- 5th bin -- //

  return 999;
}
