(* ************************************************************************* *)
(* FeynRules Model file for a Zee-Babu extension of the Standard Model.	     *)
(* Contains two new scalar fields: k--(Y=-2) and h-(Y=-1) that couple        *)
(* to all SM leptons through Yukawa couplings, Z/A bosons through gauge	     *)
(* couplings, and the SM Higgs through dimensionless couplings.              *)
(* Neutrinos are massless and aligned with flavor states in this model file. *)
(* The Lagrangian is written with Goldstone bosons.		 	     *)
(*									     *)
(* Contact author: R. Ruiz [rruiz [at] ifj.edu.pl]						 *)
(*										     		 *)
(* The original Zee-Babu papers are: 								 *)
(* 1. Zee, PLB ('85) [Zee:1985rj]	  							 *)
(* 2. Zee, NPB ('86) [Zee:1985id]								 *)
(* 3. Babu, PLB ('88) [Babu:1988ki]								 *)
(* At LO and NLO in QCD, this model was implemented in:				 		 *)
(* 1. R. Ruiz [arXiv:2206.14833],								 *)
(* and extends the default FeynRules SM model file.  				 	     	 *)
(* Please cite accordingly.									 *)
(*				    								 *)
(* ************************************************************************* *)

(* ************************** *)
(* *****  Information   ***** *)
(* ************************** *)
M$ModelName = "SM_ZeeBabu";

M$Information = {
  Authors      -> {"R. Ruiz"},
  Version      -> "1.0",
  Date         -> "2022 June 30",
  Institutions -> {"IFJ PAN"},
  Emails       -> {"rruiz@ifj.edu.pl"},
  References   -> {"R. Ruiz [arXiv:2206.14833];",
		   		"Zee, PLB ('85) [Zee:1985rj];",
		   		"Zee, NPB ('86) [Zee:1985id];",
				"Babu, PLB ('88) [Babu:1988ki]"
				},
  URLs         -> {"https://feynrules.irmp.ucl.ac.be/wiki/ZeeBabu"}
};
FeynmanGauge = True;


(* ************************** *)
(* *****  Change  log   ***** *)
(* ************************** *)
(* v1.1: Public release of NLO model file *)

(* ************************** *)
(* *****   Parameters   ***** *)
(* ************************** *)
M$Parameters = {
	(* External Parameters *)

	(* *****   Scalar Sector   ***** *)

	muZB == {
    	ParameterType    -> External, 
		BlockName        -> ZeeBabuPotential, 
    	OrderBlock       -> 1, 
    	Value            -> 1000.0,
		ComplexParameter -> False,
		InteractionOrder -> {NP,1},
		TeX              -> Subscript[mu,ZB],
    	Description      -> "LNV mass scale; h-h-k ZB coupling"
	},

	lamhZB == {
    	ParameterType    -> External, 
		BlockName        -> ZeeBabuPotential, 
    	OrderBlock       -> 2, 
    	Value            -> 1.0,
		ComplexParameter -> False,
		InteractionOrder -> {NP,1},
		TeX              -> Subscript[lambda,hZB],
    	Description      -> "h-h-h-h ZB coupling"
	},

	lamkZB == {
    	ParameterType    -> External, 
		BlockName        -> ZeeBabuPotential, 
    	OrderBlock       -> 3, 
    	Value            -> 1.000000001,
		ComplexParameter -> False,
		InteractionOrder -> {NP,1},
		TeX              -> Subscript[lambda,kZB],
    	Description      -> "k-k-k-k ZB coupling"
	},

	lamhZBkZB == {
    	ParameterType    -> External, 
		BlockName        -> ZeeBabuPotential, 
    	OrderBlock       -> 4, 
    	Value            -> 1.000000002,
		ComplexParameter -> False,
		InteractionOrder -> {NP,1},
		TeX              -> Subscript[lambda,hZBkZB],
    	Description      -> "h-h-k-k ZB coupling"
	},

	lamhZBH == {
    	ParameterType    -> External, 
		BlockName        -> ZeeBabuPotential, 
    	OrderBlock       -> 5, 
    	Value            -> 1.000000003,
		ComplexParameter -> False,
		InteractionOrder -> {{NP,1},{QED,1}},
		TeX              -> Subscript[lambda,hZBH],
    	Description      -> "h-h-H(-H) ZB-SM coupling"
	},

	lamkZBH == {
    	ParameterType    -> External, 
		BlockName        -> ZeeBabuPotential, 
    	OrderBlock       -> 6, 
    	Value            -> 1.000000004,
		ComplexParameter -> False,
		InteractionOrder -> {{NP,1},{QED,1}},
		TeX              -> Subscript[lambda,kZBH],
    	Description      -> "k-k-H(-H) ZB-SM coupling"
	},

	(* *****   Symmetric Couplings   ***** *)

	gee == {
    	ParameterType    -> External, 
		BlockName        -> ZeeBabuSymmetric, 
    	OrderBlock       -> 1, 
    	Value            -> 1.000000010,
		ComplexParameter -> False,
		InteractionOrder -> {NP,1},
		TeX              -> Subscript[g,ee],
    	Description      -> "ee coupling to k--"
	},

	gemu == {
    	ParameterType    -> External, 
		BlockName        -> ZeeBabuSymmetric, 
    	OrderBlock       -> 2, 
    	Value            -> 1.000000011,
		ComplexParameter -> False,
		InteractionOrder -> {NP,1},
		TeX              -> Subscript[g,emu],
    	Description      -> "emu coupling to k--"
	},

	getau == {
    	ParameterType    -> External, 
		BlockName        -> ZeeBabuSymmetric, 
    	OrderBlock       -> 3,
    	Value            -> 1.000000012,
		ComplexParameter -> False,
		InteractionOrder -> {NP,1},
		TeX              -> Subscript[g,etau],
    	Description      -> "etau coupling to k--"
	},

	gmumu == {
    	ParameterType    -> External, 
		BlockName        -> ZeeBabuSymmetric, 
    	OrderBlock       -> 4, 
    	Value            -> 1.000000013,
		ComplexParameter -> False,
		InteractionOrder -> {NP,1},
		TeX              -> Subscript[g,mumu],
    	Description      -> "mumu coupling to k--"
	},

	gmutau == {
    	ParameterType    -> External, 
		BlockName        -> ZeeBabuSymmetric, 
    	OrderBlock       -> 5,
    	Value            -> 1.000000014,
		ComplexParameter -> False,
		InteractionOrder -> {NP,1},
		TeX              -> Subscript[g,mutau],
    	Description      -> "mutau coupling to k--"
	},

	gtautau == {
    	ParameterType    -> External, 
		BlockName        -> ZeeBabuSymmetric, 
    	OrderBlock       -> 6,
    	Value            -> 1.000000015,
		ComplexParameter -> False,
		InteractionOrder -> {NP,1},
		TeX              -> Subscript[g,tautau],
    	Description      -> "tautau coupling to k--"
	},

	(* *****   Anti-symmetric Couplings   ***** *)

	femu == {
    	ParameterType    -> External, 
		BlockName        -> ZeeBabuAntisymmetric, 
    	OrderBlock       -> 1, 
    	Value            -> 1.000000020,
		ComplexParameter -> False,
		InteractionOrder -> {NP,1},
		TeX              -> Subscript[f,emu],
    	Description      -> "emu coupling to h-"
	},

	fetau == {
    	ParameterType    -> External, 
		BlockName        -> ZeeBabuAntisymmetric, 
    	OrderBlock       -> 2,
    	Value            -> 1.000000021,
		ComplexParameter -> False,
		InteractionOrder -> {NP,1},
		TeX              -> Subscript[f,etau],
    	Description      -> "etau coupling to h-"
	},

	fmutau == {
    	ParameterType    -> External, 
		BlockName        -> ZeeBabuAntisymmetric, 
    	OrderBlock       -> 3,
    	Value            -> 1.000000022,
		ComplexParameter -> False,
		InteractionOrder -> {NP,1},
		TeX              -> Subscript[f,mutau],
    	Description      -> "mutau coupling to h-"
	},


	(* Internal Parameters *)
	mhZBTree == {
    	ParameterType    -> Internal, 
    	Value            -> Sqrt[mhZB^2 - lamhZBH*vev^2 /2],
		ComplexParameter -> False,
    	Description      -> "tree-level mass of h- before EWSB"
	},

	mkZBTree == {
    	ParameterType    -> Internal, 
    	Value            -> Sqrt[mkZB^2 - lamhZBH*vev^2 /2],
		ComplexParameter -> False,
    	Description      -> "tree-level mass of k-- before EWSB"
	},


(* ************************** *)
(* *****   Containers   ***** *)
(* ************************** *)

	gll == { 
		ParameterType	->Internal, 
		Indices			->{Index[Generation],Index[Generation]},
		ComplexParameter->False,
		Unitary			->False,
		InteractionOrder->{NP,1},
		Value			->{
			gll[1,1]->gee,   gll[1,2]->gemu,   gll[1,3]->getau,
			gll[2,1]->gemu,  gll[2,2]->gmumu,  gll[2,3]->gmutau,
			gll[3,1]->getau, gll[3,2]->gmutau, gll[3,3]->gtautau},
		TeX				->Subscript[g,ll], 
		Description		->"symmetric ll coupling matrix to k--"
		},

	fll == { 
		ParameterType	->Internal, 
		Indices			->{Index[Generation],Index[Generation]},
		ComplexParameter->False,
		Unitary			->False,
		InteractionOrder->{NP,1},
		Value			->{
			fll[1,1]->0,   	  fll[1,2]->femu,    fll[1,3]->fetau,
			fll[2,1]->-femu,  fll[2,2]->0,  	 fll[2,3]->fmutau,
			fll[3,1]->-fetau, fll[3,2]->-fmutau, fll[3,3]->0},
		TeX				->Subscript[f,ll], 
		Description		->"anti-symmetric ll coupling matrix to h-"
		}            

};


M$InteractionOrderHierarchy = {
	{QCD,1},
	{QED,2},
	{NP,1}
};


(* ************************** *)
(* **** Particle classes **** *)
(* ************************** *)
M$ClassesDescription = {
	(*Zee-Babu Scalars*)
	S[138] == {
		ClassName		-> hZB,
		SelfConjugate	-> False,
		Mass			-> {mhZB,300.},
		Width			-> {whZB,1},
		QuantumNumbers 	-> {Q->-1,LeptonNumber -> 2},	
		PropagatorLabel	-> "h-",
		PropagatorType  -> ScalarDash,
		PropagatorArrow -> True,
		ParticleName    -> "h-",
		AntiParticleName-> "h+",
		PDG             ->  38,
		FullName        -> "h-"
	},

	S[161] == {
		ClassName		-> kZB,
		SelfConjugate	-> False,
		Mass			-> {mkZB,500.},
		Width			-> {wkZB,1},
		QuantumNumbers 	-> {Q->-2,LeptonNumber -> 2},	
		PropagatorLabel	-> "k--",
		PropagatorType  -> ScalarDash,
		PropagatorArrow -> True,
		ParticleName    -> "k--",
		AntiParticleName-> "k++",
		PDG             ->  61,
		FullName        -> "k--"
	},

  	S[238] == { 
		ClassName     -> hGauge,
		Unphysical    -> True,
		SelfConjugate -> False,
		QuantumNumbers-> {Y->-1,LeptonNumber -> 2},	 
		Definitions   -> { hGauge -> hZB }
	},
  
  	S[261] == { 
		ClassName     -> kGauge,
		Unphysical    -> True,
		SelfConjugate -> False,
		QuantumNumbers-> {Y->-2,LeptonNumber -> 2},	 
		Definitions   -> { kGauge -> kZB }
	}

};
 
(* ************************** *)
(* *****   Lagrangian   ***** *)
(* ************************** *)

(* hZB and kZB covariant derivatives and masses before EWSB *)
LZBKin := DC[hGaugebar,mu] DC[hGauge,mu] + DC[kGaugebar,mu] DC[kGauge,mu] - mhZBTree^2 hGaugebar hGauge - mkZBTree^2 kGaugebar kGauge;

(* hZB and kZB self-couplings *)
LZBScalarSelf := lamhZB * (hGaugebar hGauge)^2 + lamkZB * (kGaugebar kGauge)^2 + lamhZBkZB * (hGaugebar hGauge) (kGaugebar kGauge);

(* hZB and kZB couplings to SM Higgs *)
LZBHiggs	 := lamhZBH (hGaugebar hGauge) * Phibar[ii] * Phi[ii] +	lamkZBH (kGaugebar kGauge) * Phibar[ii] * Phi[ii];

(* hZB and kZB LNV coupling *)
LlnvBare 	:= muZB hGauge hGauge kGaugebar;
Llnv 		:= LlnvBare + HC[LlnvBare];

(* kZB couplings to leptons *)
LkllBare 	:= kGaugebar * CC[lRbar][sp1,ff1].gll[ff1,ff2].lR[sp1,ff2];
Lkll		:= LkllBare + HC[LkllBare];

(* hZB couplings to leptons *)
(* ii,jj = isospin; ff1,ff2 = flavor; sp = spin *)
LhllBare 	:= hGaugebar * CC[LLbar][sp, ii, ff1].fll[ff1,ff2].Eps[ii, jj].LL[sp, jj, ff2];
Lhll		:= LhllBare + HC[LhllBare];

(* Full ZB Lagrangian *)
LScalar := LZBScalarSelf + LZBHiggs + Llnv;
LZB:= LZBKin + Lkll + Lhll - LScalar;

(* Full SM+ZB Lagrangian *)
LFull := LSM + LZB;