(* ************************************************************************************ *)
(* FeynRules Model file for a heavy neutrino extension of the Standard Model.	     	*)
(* Contains three massive Dirac neutrinos that couples to all SM leptons through     	*)
(* mixing between the active (flavor) eigenstates and mass eigenstates.              	*)
(* The Lagrangian is written in the Feynman Gauge.			     	     	*)
(*										     	*)
(* Contact author: R. Ruiz [richard.ruiz [at] uclouvain.be]			     	*)
(*										     	*)
(* The Lagrangian is based on Atre, et. al. [arXiv:0901.3589].				*)
(* The model was first implemented in Pascoli, et. al. [arXiv:1805.09335].  	     	*)
(* Model files are extensions of default FeynRules SM model file			*)
(* and the HeavyN libraries by Degrande, et al. [arXiv:1602.06957] 			*)
(* Please cite accordingly.					     		     	*)
(*										     	*)
(* Neutrino mass (mN) and mixing parameters (VlN) between heavy mass eigenstate and  	*)
(* flavor eigenstates are taken to be independent, phenomenological parameters.      	*)
(* Mixing parameters (VeN, VmuN, VtaN) are taken to be real.			     	*)
(* This allows for maximum flexibility and model independence when calculating rates. 	*)
(*										     	*) 
(* Feynman gauge is set to true.						     	*)
(* ************************************************************************************ *)

(* ************************** *)
(* *****  Information   ***** *)
(* ************************** *)
M$ModelName = "SM_HeavyN_Dirac";

M$Information = {
  Authors      -> {"R. Ruiz"},
  Version      -> "3.1",
  Date         -> "14 November 2018",
  Institutions -> {"CP3 / Universite Catholique de Louvain"},
  Emails       -> {"richard.ruiz@uclouvain.be"},
  References   -> {"Atre, et al, JHEP0905, 030 (2009) [arXiv:0901.3589]",
		   "Degrande, et al, PRD94, 053002 (2016) [arXiv:1602.06957]",
		   "Pascoli, et al, PLB786, 106 (2018) [arXiv:1805.09335]"},
  URLs         -> {"https://feynrules.irmp.ucl.ac.be/wiki/HeavyN"}
};
FeynmanGauge = True;


(* ************************** *)
(* *****  Change  log   ***** *)
(* ************************** *)
(* v1.1: Public release of LO model file 					*)
(* v2.1: Added Goldstone couplings for Feynman Gauge and NLO implementation 	*)
(* v2.2: Corrected relative sign between Yukawa and gauge couplings. 		*)
(* v3.1: Public release of Dirac NLO model file. 				*)

(* ************************** *)
(* *****   Parameters   ***** *)
(* ************************** *)
M$Parameters = {
	(* External Parameters *)

	VeN1 == {
    	ParameterType    -> External, 
	BlockName        -> NUMIXING, 
    	OrderBlock       -> 1, 
    	Value            -> 1.0,
	ComplexParameter -> False,
	TeX              -> Subscript[V,eN1],
    	Description      -> "Mixing between ve flavor/gauge state and N1 mass state"
	},

	VeN2 == {
    	ParameterType    -> External, 
	BlockName        -> NUMIXING, 
    	OrderBlock       -> 2, 
    	Value            -> 0.0,
	ComplexParameter -> False,
	TeX              -> Subscript[V,eN2],
    	Description      -> "Mixing between ve flavor/gauge state and N2 mass state"
	},

	VeN3 == {
    	ParameterType    -> External, 
	BlockName        -> NUMIXING, 
    	OrderBlock       -> 3, 
    	Value            -> 0.0,
	ComplexParameter -> False,
	TeX              -> Subscript[V,eN3],
    	Description      -> "Mixing between ve flavor/gauge state and N3 mass state"
	},

	VmuN1 == {
    	ParameterType    -> External, 
	BlockName        -> NUMIXING, 
    	OrderBlock       -> 4, 
    	Value            -> 0.0,
	ComplexParameter -> False,
	TeX              -> Subscript[V,muN1],
    	Description      -> "Mixing between vm flavor/gauge state and N1 mass state"
	},

	VmuN2 == {
    	ParameterType    -> External, 
	BlockName        -> NUMIXING, 
    	OrderBlock       -> 5, 
    	Value            -> 1.0,
	ComplexParameter -> False,
	TeX              -> Subscript[V,muN2],
    	Description      -> "Mixing between vm flavor/gauge state and N2 mass state"
	},

	VmuN3 == {
    	ParameterType    -> External, 
	BlockName        -> NUMIXING, 
    	OrderBlock       -> 6, 
    	Value            -> 0.0,
	ComplexParameter -> False,
	TeX              -> Subscript[V,muN3],
    	Description      -> "Mixing between vm flavor/gauge state and N3 mass state"
	},

	VtaN1 == {
    	ParameterType    -> External, 
	BlockName        -> NUMIXING, 
    	OrderBlock       -> 7, 
    	Value            -> 0.0,
	ComplexParameter -> False,
	TeX              -> Subscript[V,taN1],
    	Description      -> "Mixing between vt flavor/gauge state and N1 mass state"
	},

	VtaN2 == {
    	ParameterType    -> External, 
	BlockName        -> NUMIXING, 
    	OrderBlock       -> 8, 
    	Value            -> 0.0,
	ComplexParameter -> False,
	TeX              -> Subscript[V,taN2],
    	Description      -> "Mixing between vt flavor/gauge state and N2 mass state"
	},

	VtaN3 == {
    	ParameterType    -> External, 
	BlockName        -> NUMIXING, 
    	OrderBlock       -> 9, 
    	Value            -> 1.0,
	ComplexParameter -> False,
	TeX              -> Subscript[V,taN3],
    	Description      -> "Mixing between vt flavor/gauge state and N3 mass state"
	},

	(* Internal Parameters *)
	gN   == {
		ParameterType	 -> Internal,
		Definitions      -> {gN->ee/sw},
		InteractionOrder -> {NP,1},
		TeX		 -> Subscript[g,N]}
};

M$InteractionOrderHierarchy = {
	{QCD,1},
	{QED,2}
};

(* ************************** *)
(* **** Particle classes **** *)
(* ************************** *)
M$ClassesDescription = {
	(*Dirac Neutrino*)
	F[131] == {
		ClassName	-> N1,
		SelfConjugate	-> False,
		QuantumNumbers  -> {LeptonNumber -> 1},
		Mass		-> {mN1,300.},
		Width		-> {WN1,0.303},	
		PropagatorLabel	-> "N1",
		PropagatorType  -> Straight,
		PropagatorArrow -> True,
		ParticleName    -> "N1",
		PDG             -> {9990012},
		FullName        -> "N1"},

	F[132] == {
		ClassName	-> N2,
		SelfConjugate	-> False,
		QuantumNumbers  -> {LeptonNumber -> 1},
		Mass		-> {mN2,500.},
		Width		-> {WN2,1.50},	
		PropagatorLabel	-> "N2",
		PropagatorType  -> Straight,
		PropagatorArrow -> True,
		ParticleName    -> "N2",
		PDG             -> {9990014},
		FullName        -> "N2"},

	F[133] == {
		ClassName	-> N3,
		SelfConjugate	-> False,
		QuantumNumbers  -> {LeptonNumber -> 1},
		Mass		-> {mN3,1000.},
		Width		-> {WN3,12.3},	
		PropagatorLabel	-> "N3",
		PropagatorType  -> Straight,
		PropagatorArrow -> True,
		ParticleName    -> "N3",
		PDG             -> {9990016},
		FullName        -> "N3"}
};


(* ************************** *)
(* *****   Lagrangian   ***** *)
(* ************************** *)
LNKin :=   I N1bar[s1].Ga[v,s1,s2].del[N1[s2],v] - mN1 N1bar[s1]N1[s1] \
	+  I N2bar[s1].Ga[v,s1,s2].del[N2[s2],v] - mN2 N2bar[s1]N2[s1] \
	+  I N3bar[s1].Ga[v,s1,s2].del[N3[s2],v] - mN3 N3bar[s1]N3[s1];

(* Charge Current *)
LNCCbare :=   gN/Sqrt[2] *(  VeN1 * N1bar.W[m].ProjM[m].e	\
           	 	  + VmuN1 * N1bar.W[m].ProjM[m].mu	\
			  + VtaN1 * N1bar.W[m].ProjM[m].ta )	\
	    + gN/Sqrt[2] *(  VeN2 * N2bar.W[m].ProjM[m].e	\
           	 	  + VmuN2 * N2bar.W[m].ProjM[m].mu	\
			  + VtaN2 * N2bar.W[m].ProjM[m].ta )	\
	    + gN/Sqrt[2] *(  VeN3 * N3bar.W[m].ProjM[m].e	\
           	 	  + VmuN3 * N3bar.W[m].ProjM[m].mu	\
			  + VtaN3 * N3bar.W[m].ProjM[m].ta );
LNCC := LNCCbare + HC[LNCCbare];

(* Neutral Current *)
LNNCBare :=   1/2 * gN/cw *(  VeN1 * N1bar.Z[m].ProjM[m].ve	\
           	  	   + VmuN1 * N1bar.Z[m].ProjM[m].vm	\
           		   + VtaN1 * N1bar.Z[m].ProjM[m].vt )	\
	    + 1/2 * gN/cw *(  VeN2 * N2bar.Z[m].ProjM[m].ve	\
           	  	   + VmuN2 * N2bar.Z[m].ProjM[m].vm	\
           		   + VtaN2 * N2bar.Z[m].ProjM[m].vt )	\
	    + 1/2 * gN/cw *(  VeN3 * N3bar.Z[m].ProjM[m].ve	\
           	  	   + VmuN3 * N3bar.Z[m].ProjM[m].vm	\
           		   + VtaN3 * N3bar.Z[m].ProjM[m].vt );
LNNC := LNNCBare + HC[LNNCBare];

(* Higgs Interaction *)
LNHbare := 	- gN*mN1/(2*MW) *( VeN1 * N1bar.ProjM.ve H	\
				+ VmuN1 * N1bar.ProjM.vm H	\
				+ VtaN1 * N1bar.ProjM.vt H )	\
		- gN*mN2/(2*MW) *( VeN2 * N2bar.ProjM.ve H	\
				+ VmuN2 * N2bar.ProjM.vm H	\
				+ VtaN2 * N2bar.ProjM.vt H )	\
		- gN*mN3/(2*MW) *( VeN3 * N3bar.ProjM.ve H	\
				+ VmuN3 * N3bar.ProjM.vm H	\
				+ VtaN3 * N3bar.ProjM.vt H );
LNH := LNHbare + HC[LNHbare];

(* Goldstone Interaction *)
LNGbare :=	  I *gN*mN1/(2*MW) 	*(	VeN1   * vebar.ProjP.N1 G0	\
					+	VmuN1  * vmbar.ProjP.N1 G0	\
					+	VtaN1  * vtbar.ProjP.N1 G0)	\
		+ I *gN*mN2/(2*MW) 	*(	VeN2   * vebar.ProjP.N2 G0	\
					+	VmuN2  * vmbar.ProjP.N2 G0	\
					+	VtaN2  * vtbar.ProjP.N2 G0)	\
		+ I *gN*mN3/(2*MW) 	*(	VeN3   * vebar.ProjP.N3 G0	\
					+	VmuN3  * vmbar.ProjP.N3 G0	\
					+	VtaN3  * vtbar.ProjP.N3 G0)	\
		+ I *gN*mN1/(Sqrt[2]*MW) *(	VeN1   *  ebar.ProjP.N1 GPbar	\
					+	VmuN1  * mubar.ProjP.N1 GPbar	\
					+	VtaN1  * tabar.ProjP.N1 GPbar)	\
		+ I *gN*mN2/(Sqrt[2]*MW) *(	VeN2   *  ebar.ProjP.N2 GPbar	\
					+	VmuN2  * mubar.ProjP.N2 GPbar	\
					+	VtaN2  * tabar.ProjP.N2 GPbar)	\
		+ I *gN*mN3/(Sqrt[2]*MW) *(	VeN3   *  ebar.ProjP.N3 GPbar	\
					+	VmuN3  * mubar.ProjP.N3 GPbar	\
					+	VtaN3  * tabar.ProjP.N3 GPbar);
LNG := LNGbare + HC[LNGbare];

(* Full N Lagrangian *)
LN:= LNKin + LNCC + LNNC + LNH + LNG;

(* Combine Lagrangian *)
LFull := LSM + LN;