(* ********************************************************************************* *)
(* FeynRules Model file for a heavy neutrino extension of the Standard Model.	     *)
(* Contains three massive Majorana neutrinos that couples to all SM leptons through  *)
(* mixing between the active (flavor) eigenstates and mass eigenstates.              *)
(* The Lagrangian is written in the Feynman Gauge.		 	     	     *)
(*										     *)
(* Contact author: R. Ruiz [richard.ruiz [at] durham.ac.uk]			     *)
(*										     *)
(* The Lagrangian is based on Atre, et. al. [arXiv:0901.3589]. 	     		     *)
(* At LO, the model was implemented in Alva, et. al. [arXiv:1411.7305].    	     *)
(* At NLO, the model was implemented in Degrande, et al. [arXiv:1602.06957].	 *)
(* Both model files are extensions of default FeynRules SM model file.   	     *)
(* At NLO, a subset of vSMEFT operators at dimension six were                    *) 
(*   implemented in Cirigliano, et al. [arXiv:2105.11462]   *)
(* Please cite accordingly.					     		     *)
(*										     *)
(* Neutrino mass (mN) and mixing parameters (VlN) between heavy mass eigenstate and  *)
(* flavor eigenstates are taken to be independent, phenomenological parameters.      *)
(* Mixing parameters (VeN, VmuN, VtaN) are taken to be real.			             *)
(* Wilson coefficients and vSMEFT cutoff scale are also real, independent parameters *)
(* This allows for maximum flexibility and model independence when calculating	     *)
(* rates. 									                                         *) 
(*										     *) 
(* Feynman gauge is set to true.						     *)
(* ********************************************************************************* *)

(* ************************** *)
(* *****  Information   ***** *)
(* ************************** *)
M$ModelName = "SM_HeavyN_vSMEFTdim6";

M$Information = {
  Authors      -> {"R. Ruiz"},
  Version      -> "2.5.branch1.2",
  Date         -> "2021 May 26",
  Institutions -> {"IFJ PAN"},
  Emails       -> {"rruiz@ifj.edu.pl"},
  References   -> {"Atre, et al, JHEP0905, 030 (2009) [arXiv:0901.3589]",
		   		"Alva, et al, JHEP1502, 072 (2015) [arXiv:1411.7305]",
		   		"Degrande, et al, PRD [arXiv:1602.06957]",
		   		"Cirigliano, et al,  [arXiv:2105.11462]"},
  URLs         -> {"https://feynrules.irmp.ucl.ac.be/wiki/HeavyN"}
};
FeynmanGauge = True;


(* ************************** *)
(* *****  Change  log   ***** *)
(* ************************** *)
(* v1.1: Public release of LO model file *)
(* v2.1: Added Goldstone couplings for Feynman Gauge and NLO implementation. *)
(* v2.2: Corrected relative sign between Yukawa and gauge couplings. *)
(* v2.5: Defined NuArray and NuMix for coupling to mesons. *)
(* v2.5.branch1: branched from v2.5 to add dimension six operator *)
(* v2.5.branch1.2: updated to include arXiv information *)

IndexRange[Index[NuGeneration]] = Range[6];
IndexStyle[NuGeneration, kk];

(*IndexStyle[NuGeneration, ff1];*)
(*IndexStyle[NuGeneration, ff2];*)


(* ************************** *)
(* *****   Parameters   ***** *)
(* ************************** *)
M$Parameters = {
	(* External Parameters *)

	VeN1 == {
    	ParameterType    -> External, 
		BlockName        -> NUMIXING, 
    	OrderBlock       -> 1, 
    	Value            -> 1.0,
		ComplexParameter -> False,
		TeX              -> Subscript[V,eN1],
    	Description      -> "Mixing between ve flavor/gauge state and N1 mass state"
	},

	VeN2 == {
    	ParameterType    -> External, 
		BlockName        -> NUMIXING, 
    	OrderBlock       -> 2, 
    	Value            -> 0.0,
		ComplexParameter -> False,
		TeX              -> Subscript[V,eN2],
    	Description      -> "Mixing between ve flavor/gauge state and N2 mass state"
	},

	VeN3 == {
    	ParameterType    -> External, 
		BlockName        -> NUMIXING, 
		OrderBlock       -> 3, 
		Value            -> 0.0,
		ComplexParameter -> False,
		TeX              -> Subscript[V,eN3],
			Description      -> "Mixing between ve flavor/gauge state and N3 mass state"
	},

	VmuN1 == {
    	ParameterType    -> External, 
		BlockName        -> NUMIXING, 
		OrderBlock       -> 4, 
		Value            -> 0.0,
		ComplexParameter -> False,
		TeX              -> Subscript[V,muN1],
    	Description      -> "Mixing between vm flavor/gauge state and N1 mass state"
	},

	VmuN2 == {
    	ParameterType    -> External, 
		BlockName        -> NUMIXING, 
		OrderBlock       -> 5, 
		Value            -> 1.0,
		ComplexParameter -> False,
		TeX              -> Subscript[V,muN2],
    	Description      -> "Mixing between vm flavor/gauge state and N2 mass state"
	},

	VmuN3 == {
    	ParameterType    -> External, 
		BlockName        -> NUMIXING, 
		OrderBlock       -> 6, 
		Value            -> 0.0,
		ComplexParameter -> False,
		TeX              -> Subscript[V,muN3],
    	Description      -> "Mixing between vm flavor/gauge state and N3 mass state"
	},

	VtaN1 == {
    	ParameterType    -> External, 
		BlockName        -> NUMIXING, 
    	OrderBlock       -> 7, 
    	Value            -> 0.0,
		ComplexParameter -> False,
		TeX              -> Subscript[V,taN1],
    	Description      -> "Mixing between vt flavor/gauge state and N1 mass state"
	},

	VtaN2 == {
    	ParameterType    -> External, 
		BlockName        -> NUMIXING, 
    	OrderBlock       -> 8, 
    	Value            -> 0.0,
		ComplexParameter -> False,
		TeX              -> Subscript[V,taN2],
    	Description      -> "Mixing between vt flavor/gauge state and N2 mass state"
	},

	VtaN3 == {
    	ParameterType    -> External, 
		BlockName        -> NUMIXING, 
    	OrderBlock       -> 9, 
    	Value            -> 1.0,
		ComplexParameter -> False,
		TeX              -> Subscript[V,taN3],
    	Description      -> "Mixing between vt flavor/gauge state and N3 mass state"
	},

	(*   vSMEFT Block   *)
	Lambda == {
    	ParameterType    -> External, 
		BlockName        -> vSMEFTBLOCK, 
    	OrderBlock       -> 1,
    	Value            -> 1000.0,
		ComplexParameter -> False,
		InteractionOrder -> {QED,-1},
		TeX              -> \[CapitalLambda],
    	Description      -> "EFT cutoff scale [GeV]"
	},

	CeN1 == {
    	ParameterType    -> External, 
		BlockName        -> vSMEFTBLOCK, 
    	OrderBlock       -> 2, 
    	Value            -> 1.0,
		ComplexParameter -> False,
		TeX              -> Subscript[C,eN1],
    	Description      -> "Wilson coeff. between ve flavor/gauge state and N1 mass state"
	},

	CeN2 == {
    	ParameterType    -> External, 
		BlockName        -> vSMEFTBLOCK, 
    	OrderBlock       -> 3, 
    	Value            -> 0.0,
		ComplexParameter -> False,
		TeX              -> Subscript[C,eN2],
    	Description      -> "Wilson coeff. between ve flavor/gauge state and N2 mass state"
	},

	CeN3 == {
    	ParameterType    -> External, 
		BlockName        -> vSMEFTBLOCK, 
		OrderBlock       -> 4, 
		Value            -> 0.0,
		ComplexParameter -> False,
		TeX              -> Subscript[C,eN3],
			Description      -> "Wilson coeff. between ve flavor/gauge state and N3 mass state"
	},

	CmuN1 == {
    	ParameterType    -> External, 
		BlockName        -> vSMEFTBLOCK, 
		OrderBlock       -> 5, 
		Value            -> 0.0,
		ComplexParameter -> False,
		TeX              -> Subscript[C,muN1],
    	Description      -> "Wilson coeff. between vm flavor/gauge state and N1 mass state"
	},

	CmuN2 == {
    	ParameterType    -> External, 
		BlockName        -> vSMEFTBLOCK, 
		OrderBlock       -> 6, 
		Value            -> 1.0,
		ComplexParameter -> False,
		TeX              -> Subscript[C,muN2],
    	Description      -> "Wilson coeff. between vm flavor/gauge state and N2 mass state"
	},

	CmuN3 == {
    	ParameterType    -> External, 
		BlockName        -> vSMEFTBLOCK, 
		OrderBlock       -> 7, 
		Value            -> 0.0,
		ComplexParameter -> False,
		TeX              -> Subscript[C,muN3],
    	Description      -> "Wilson coeff. between vm flavor/gauge state and N3 mass state"
	},

	CtaN1 == {
    	ParameterType    -> External, 
		BlockName        -> vSMEFTBLOCK, 
    	OrderBlock       -> 8, 
    	Value            -> 0.0,
		ComplexParameter -> False,
		TeX              -> Subscript[C,taN1],
    	Description      -> "Wilson coeff. between vt flavor/gauge state and N1 mass state"
	},

	CtaN2 == {
    	ParameterType    -> External, 
		BlockName        -> vSMEFTBLOCK, 
    	OrderBlock       -> 9, 
    	Value            -> 0.0,
		ComplexParameter -> False,
		TeX              -> Subscript[C,taN2],
    	Description      -> "Wilson coeff. between vt flavor/gauge state and N2 mass state"
	},

	CtaN3 == {
    	ParameterType    -> External, 
		BlockName        -> vSMEFTBLOCK, 
    	OrderBlock       -> 10, 
    	Value            -> 1.0,
		ComplexParameter -> False,
		TeX              -> Subscript[C,taN3],
    	Description      -> "Wilson coeff. between vt flavor/gauge state and N3 mass state"
	},


	(* Internal Parameters *)
	gN   == {
		ParameterType	 -> Internal,
		Definitions      -> {gN->ee/sw},
		InteractionOrder -> {NP,1},
		TeX		 -> Subscript[g,N]
	},
	gvSMEFT == {
		ParameterType	 -> Internal,
		Definitions      -> {gvSMEFT->ee*vev*vev/Lambda/Lambda/sw/2}
	},

(* ************************** *)
(* *****   Containers   ***** *)
(* ************************** *)

	PMNS == { 
		ParameterType	-> Internal,
		Indices			-> {Index[Generation], Index[NuGeneration]},
		Unitary			-> False,
		Value			-> {	PMNS[1,1]->1, PMNS[1,2]->0, PMNS[1,3]->0, PMNS[1,4]->VeN1,  PMNS[1,5]->VeN2,  PMNS[1,6]->VeN3,
								PMNS[2,1]->0, PMNS[2,2]->1, PMNS[2,3]->0, PMNS[2,4]->VmuN1, PMNS[2,5]->VmuN2, PMNS[2,6]->VmuN3,
								PMNS[3,1]->0, PMNS[3,2]->0, PMNS[3,3]->1, PMNS[3,4]->VtaN1, PMNS[3,5]->VtaN2, PMNS[3,6]->VtaN3},
		TeX				-> Subscript[V,PMNS],
		Description		-> "3x6 v-Nu Mixing Matrix"
	},  

  	numass == {
    	ParameterType	-> Internal,
    	Indices			-> {Index[NuGeneration], Index[NuGeneration]},
    	Value			-> {
			numass[1,1]->0, numass[1,2]->0, numass[1,3]->0, numass[1,4]->0,   numass[1,5]->0,   numass[1,6]->0, 
			numass[2,1]->0, numass[2,2]->0, numass[2,3]->0, numass[2,4]->0,   numass[2,5]->0,   numass[2,6]->0, 
			numass[3,1]->0, numass[3,2]->0, numass[3,3]->0, numass[3,4]->0,   numass[3,5]->0,   numass[3,6]->0, 
			numass[4,1]->0, numass[4,2]->0, numass[4,3]->0, numass[4,4]->mN1, numass[4,5]->0,   numass[4,6]->0, 
			numass[5,1]->0, numass[5,2]->0, numass[5,3]->0, numass[5,4]->0,   numass[5,5]->mN2, numass[5,6]->0, 
			numass[6,1]->0, numass[6,2]->0, numass[6,3]->0, numass[6,4]->0,   numass[6,5]->0,   numass[6,6]->mN3}, 
   		InteractionOrder -> {QED, 0},
    	TeX				-> Superscript[M, vN],
    	Description		-> "Neutrino mass matrix"
	}

};


M$InteractionOrderHierarchy = {
	{QCD,1},
	{QED,2}
};


(* ************************** *)
(* **** Particle classes **** *)
(* ************************** *)
M$ClassesDescription = {
	(*Majorana Neutrino*)
	F[131] == {
		ClassName		-> N1,
		SelfConjugate	-> True,
		Mass			-> {mN1,300.},
		Width			-> {WN1,0.303},	
		PropagatorLabel	-> "N1",
		PropagatorType  -> Straight,
		PropagatorArrow -> False,
		ParticleName    -> "N1",
		PDG             -> {9900012},
		FullName        -> "N1"
	},

	F[132] == {
		ClassName		-> N2,
		SelfConjugate	-> True,
		Mass			-> {mN2,500.},
		Width			-> {WN2,1.50},	
		PropagatorLabel	-> "N2",
		PropagatorType  -> Straight,
		PropagatorArrow -> False,
		ParticleName    -> "N2",
		PDG             -> {9900014},
		FullName        -> "N2"
	},

	F[133] == {
		ClassName		-> N3,
		SelfConjugate	-> True,
		Mass			-> {mN3,1000.},
		Width			-> {WN3,12.3},	
		PropagatorLabel	-> "N3",
		PropagatorType  -> Straight,
		PropagatorArrow -> False,
		ParticleName    -> "N3",
		PDG             -> {9900016},
		FullName        -> "N3"
	},

	(* auxiliary container *)
	F[140] == { 
		ClassName      	-> vNu, 
		Unphysical     	-> True, 
		Indices        	-> {Index[NuGeneration]},
		FlavorIndex    	-> NuGeneration,
		SelfConjugate	-> False,
		Definitions    	-> {vNu[sp1_,1]->ve[sp1],
							vNu[sp1_,2]->vm[sp1],
							vNu[sp1_,3]->vt[sp1],
							vNu[sp1_,4]->N1[sp1],
							vNu[sp1_,5]->N2[sp1],
							vNu[sp1_,6]->N3[sp1]}
	}

};
 
(* ************************** *)
(* *****   Lagrangian   ***** *)
(* ************************** *)
LNKin :=   I/2 N1bar[s1].Ga[v,s1,s2].del[N1[s2],v] - 1/2 mN1 N1bar[s1]N1[s1] \
 	    +  I/2 N2bar[s1].Ga[v,s1,s2].del[N2[s2],v] - 1/2 mN2 N2bar[s1]N2[s1] \
	    +  I/2 N3bar[s1].Ga[v,s1,s2].del[N3[s2],v] - 1/2 mN3 N3bar[s1]N3[s1];

(* Charge Current *)
LNCCbare :=   gN/Sqrt[2] *(  VeN1 * N1bar.W[m].ProjM[m].e	\
           	 	  		  + VmuN1 * N1bar.W[m].ProjM[m].mu	\
						  + VtaN1 * N1bar.W[m].ProjM[m].ta )	\
	  	  	+ gN/Sqrt[2] *(  VeN2 * N2bar.W[m].ProjM[m].e	\
           	 			  + VmuN2 * N2bar.W[m].ProjM[m].mu	\
						  + VtaN2 * N2bar.W[m].ProjM[m].ta )	\
		    + gN/Sqrt[2] *(  VeN3 * N3bar.W[m].ProjM[m].e	\
           	 			  + VmuN3 * N3bar.W[m].ProjM[m].mu	\
						  + VtaN3 * N3bar.W[m].ProjM[m].ta );
LNCC := LNCCbare + HC[LNCCbare];

(* Charge Current at dimension six*)
LNRHbare :=	gvSMEFT/Sqrt[2] *( CeN1 * N1bar.W[m].ProjP[m].e	\
						+   CmuN1 * N1bar.W[m].ProjP[m].mu	\
						+   CtaN1 * N1bar.W[m].ProjP[m].ta )*(1 + 2/vev * H + 1/vev/vev * H H) \
		+ 	gvSMEFT/Sqrt[2] *( CeN2 * N2bar.W[m].ProjP[m].e	\
						+   CmuN2 * N2bar.W[m].ProjP[m].mu	\
						+   CtaN2 * N2bar.W[m].ProjP[m].ta )*(1 + 2/vev * H + 1/vev/vev * H H) \
		+ 	gvSMEFT/Sqrt[2] *( CeN3 * N3bar.W[m].ProjP[m].e	\
						+   CmuN3 * N3bar.W[m].ProjP[m].mu	\
						+   CtaN3 * N3bar.W[m].ProjP[m].ta )*(1 + 2/vev * H + 1/vev/vev * H H) ;
LND6 := LNRHbare + HC[LNRHbare];

(* Neutral Current *)
LNNCBare :=   1/2 * gN/cw *(  VeN1 * N1bar.Z[m].ProjM[m].ve	\
           	  			   + VmuN1 * N1bar.Z[m].ProjM[m].vm	\
        		   		   + VtaN1 * N1bar.Z[m].ProjM[m].vt )	\
		    + 1/2 * gN/cw *(  VeN2 * N2bar.Z[m].ProjM[m].ve	\
		           	  	   + VmuN2 * N2bar.Z[m].ProjM[m].vm	\
        		   		   + VtaN2 * N2bar.Z[m].ProjM[m].vt )	\
	    	+ 1/2 * gN/cw *(  VeN3 * N3bar.Z[m].ProjM[m].ve	\
		           	  	   + VmuN3 * N3bar.Z[m].ProjM[m].vm	\
        		   		   + VtaN3 * N3bar.Z[m].ProjM[m].vt );
LNNC := LNNCBare + HC[LNNCBare];

(* Higgs Interaction *)
LNHbare := 	- gN*mN1/(2*MW) *( VeN1 * N1bar.ProjM.ve H	\
							+ VmuN1 * N1bar.ProjM.vm H	\
							+ VtaN1 * N1bar.ProjM.vt H )	\
			- gN*mN2/(2*MW) *( VeN2 * N2bar.ProjM.ve H	\
							+ VmuN2 * N2bar.ProjM.vm H	\
							+ VtaN2 * N2bar.ProjM.vt H )	\
			- gN*mN3/(2*MW) *( VeN3 * N3bar.ProjM.ve H	\
							+ VmuN3 * N3bar.ProjM.vm H	\
							+ VtaN3 * N3bar.ProjM.vt H );
LNH := LNHbare + HC[LNHbare];

(* Goldstone Interaction *)
LNGbare :=	  I *gN*mN1/(2*MW) 	*(	VeN1   * vebar.ProjP.N1 G0	\
								+	VmuN1  * vmbar.ProjP.N1 G0	\
								+	VtaN1  * vtbar.ProjP.N1 G0)	\
			+ I *gN*mN2/(2*MW) 	*(	VeN2   * vebar.ProjP.N2 G0	\
								+	VmuN2  * vmbar.ProjP.N2 G0	\
								+	VtaN2  * vtbar.ProjP.N2 G0)	\
			+ I *gN*mN3/(2*MW) 	*(	VeN3   * vebar.ProjP.N3 G0	\
								+	VmuN3  * vmbar.ProjP.N3 G0	\
								+	VtaN3  * vtbar.ProjP.N3 G0)	\
		+ I *gN*mN1/(Sqrt[2]*MW) *(	VeN1   *  ebar.ProjP.N1 GPbar	\
								+	VmuN1  * mubar.ProjP.N1 GPbar	\
								+	VtaN1  * tabar.ProjP.N1 GPbar)	\
		+ I *gN*mN2/(Sqrt[2]*MW) *(	VeN2   *  ebar.ProjP.N2 GPbar	\
								+	VmuN2  * mubar.ProjP.N2 GPbar	\
								+	VtaN2  * tabar.ProjP.N2 GPbar)	\
		+ I *gN*mN3/(Sqrt[2]*MW) *(	VeN3   *  ebar.ProjP.N3 GPbar	\
								+	VmuN3  * mubar.ProjP.N3 GPbar	\
								+	VtaN3  * tabar.ProjP.N3 GPbar);
LNG := LNGbare + HC[LNGbare];

(* Full N Lagrangian *)
LN:= LNKin + LNCC + LNNC + LNH + LNG + LND6;

(* Combine Lagrangian *)
LFull := LSM + LN;