(***************************************************************************************************************)
(******                       This is the FeynRules mod-file for the minimal Zp models                    ******)
(******                                                                                                   ******)
(******     Author: L. Basso                                                                              ******)
(******                                                                                                   ******)
(****** Choose whether Feynman gauge is desired.                                                          ******)
(****** If set to False, unitary gauge is assumed.                                                          ****)
(****** Feynman gauge is especially useful for CalcHEP/CompHEP where the calculation is 10-100 times faster. ***)
(****** Feynman gauge is not supported in MadGraph and Sherpa.                                              ****)
(***************************************************************************************************************)

M$ModelName = "minimal_Zp";


M$Information = {Authors -> "L. Basso", 
             Version -> "2.0",
             Date -> "09-06-2011",
             Institutions -> {"University of Southampton", "RAL-PPD, Didcot, UK", "Albert-Ludwigs-Universität Freiburg"},
             Emails -> {"lorenzo.basso@physik.uni-freiburg.de"},
	     References -> {"L.~Basso, G.M.~Pruna and S.~Moretti, \"A Renormalisation Group Equation Study of the Scalar Sector of the Minimal B-L Extension of the Standard Model,\",  Phys.,Rev. D 82, 055018 (2010) [arXiv:1004.3039 [hep-ph]]", "L.~Basso, G.M.~Pruna and S.~Moretti, \"A Theoretical constraints on the couplings of non-exotic minimal Z' bosons,\",  JHEP 1108 122 (2011) [arXiv:1106.4762 [hep-ph]]"},
             URLs -> "http://feynrules.phys.ucl.ac.be/..."};

FeynmanGauge = True;


(******* Index definitions ********)

IndexRange[ Index[Generation] ] = Range[3]

IndexRange[ Index[Colour] ] = NoUnfold[Range[3]]

IndexRange[ Index[Gluon] ] = NoUnfold[Range[8]]

IndexRange[ Index[SU2W] ] = Unfold[Range[3]]


IndexStyle[Colour, i]

IndexStyle[Generation, f]

IndexStyle[Gluon ,a]

IndexStyle[SU2W ,k]



(******* Gauge parameters (for FeynArts) ********)

GaugeXi[ V[1] ] = GaugeXi[A];
GaugeXi[ V[2] ] = GaugeXi[Z];
GaugeXi[ V[3] ] = GaugeXi[W];
GaugeXi[ V[4] ] = GaugeXi[G];
GaugeXi[ V[7] ] = GaugeXi[Zp];
GaugeXi[ S[1] ] = 1;
GaugeXi[ S[2] ] = GaugeXi[Z];
GaugeXi[ S[3] ] = GaugeXi[W];
GaugeXi[ S[4] ] = 1;
GaugeXi[ S[5] ] = GaugeXi[Zp];
GaugeXi[ U[1] ] = GaugeXi[A];
GaugeXi[ U[2] ] = GaugeXi[Z];
GaugeXi[ U[31] ] = GaugeXi[W];
GaugeXi[ U[32] ] = GaugeXi[W];
GaugeXi[ U[4] ] = GaugeXi[G];
GaugeXi[ U[7] ] = GaugeXi[Zp];


(***** Setting for interaction order (as e.g. used by MadGraph 5)  ******)

M$InteractionOrderHierarchy = {
     {QCD, 1},
     {QED, 2}
    };


(****************  Parameters *************)

M$Parameters = {

  (* External parameters *)


  \[Alpha]EWM1== {
        ParameterType -> External,
        BlockName -> BLINPUTS,
        ParameterName -> aEWM1,
        InteractionOrder -> {QED, -2},
        Value -> 127.9,
        Description -> "Inverse of the electroweak coupling constant at Z-pole"},

  Gf == {
        ParameterType -> External,
        BlockName -> BLINPUTS,
        InteractionOrder -> {QED, 2},
        Value -> 1.16637 * 10^(-5),
        Description -> "Fermi constant"},

  \[Alpha]S == {
        ParameterType -> External,
        BlockName -> BLINPUTS,
        TeX -> Subscript[\[Alpha], s],
        ParameterName -> aS,
        InteractionOrder -> {QCD, 2},
        Value -> 0.1184,
        Description -> "Strong coupling constant at the Z pole."},

  MW == {
        ParameterType -> External,
        BlockName -> BLINPUTS,
        Value -> 80.292,
        Description -> "W mass"},

  MZp == {
        ParameterType -> External,
        BlockName -> BLINPUTS,
        Value -> 1500.00,
        Description -> "Zp mass"},
	 
  g1p == {
        ParameterType -> External,
        BlockName -> BLINPUTS,
	InteractionOrder -> {QED, 1},
        Value -> 0.2,
        Description -> "Zp coupling"},
 
  gt == {
        ParameterType -> External,
        BlockName -> BLINPUTS,
	InteractionOrder -> {QED, 1},
        Value -> -0.1,
        Description -> "Z-Zp mixing coupling"},

  ymdo == {
        ParameterType -> External,
        BlockName -> YUKAWA,
        Value -> 5.04*10^(-3),
	OrderBlock -> {1},
	Description -> "Down Yukawa mass"},


  ymup == {
        ParameterType -> External,
        BlockName -> YUKAWA,
        Value -> 2.55*10^(-3),
        OrderBlock -> {2},
        Description -> "Up Yukawa mass"},

  yms == {
        ParameterType -> External,
        BlockName -> YUKAWA,
        Value -> 0.101,
	OrderBlock -> {3},
	Description -> "Strange Yukawa mass"},
	
  ymc == {
        ParameterType -> External,
        BlockName -> YUKAWA,
        Value -> 1.27,
        OrderBlock -> {4},
        Description -> "Charm Yukawa mass"},

  ymb == {
        ParameterType -> External,
        BlockName -> YUKAWA,
        Value -> 4.7,
        OrderBlock -> {5},
        Description -> "Bottom Yukawa mass"},

  ymt == {
        ParameterType -> External,
        BlockName -> YUKAWA,
        Value -> 172.0,
        OrderBlock -> {6},
        Description -> "Top Yukawa mass"},

  yme == {
        ParameterType -> External,
        BlockName -> YUKAWA,
        Value -> 0.000511,
        OrderBlock -> {11},
        Description -> "Electron Yukawa mass"},

  ymmu == {
        ParameterType -> External,
        BlockName -> YUKAWA,
        Value -> 0.1057,
        OrderBlock -> {13},
        Description -> "Muon Yukawa mass"},

  ymtau == {
        ParameterType -> External,
        BlockName -> YUKAWA,
        Value -> 1.777,
        OrderBlock -> {15},
        Description -> "Tau Yukawa mass"},

  MH1 == {
        ParameterType -> External,
        BlockName -> BLINPUTS,
        Value -> 120.00,
        Description -> "H1 mass"},

  MH2 == {
        ParameterType -> External,
        BlockName -> BLINPUTS,
        Value -> 450.00,
        Description -> "H2 mass"},


  Sa == {
        ParameterType -> External,
        BlockName -> BLINPUTS,
	Value -> 0.1,
        Description -> "Sine of Higgses mixing angle"},

  sw2 == {
        ParameterType -> External,
        BlockName -> BLINPUTS,
        Value -> 0.232,
        Description -> "Squared Sin of the Weinberg angle"},



   (* Internal Parameters *)
			
  \[Alpha]EW == {
        ParameterType -> Internal,
        Value -> 1/\[Alpha]EWM1,
        TeX -> Subscript[\[Alpha], EW],
        ParameterName -> aEW,
        InteractionOrder -> {QED, 2},
        Description -> "Electroweak coupling contant"},


   ee == {
        TeX -> e,
        ParameterType -> Internal,
        Value -> Sqrt[4 Pi \[Alpha]EW],
        InteractionOrder -> {QED, 1},
        Description -> "Electric coupling constant"},

   cw == {
        TeX -> Subscript[c, w],
        ParameterType -> Internal,
        Value -> Sqrt[1 - sw2],
        Description -> "Cos of the Weinberg angle"},  

   sw == {
        TeX -> Subscript[s, w],
        ParameterType -> Internal,
        Value -> Sqrt[sw2],
        Description -> "Sin of the Weinberg angle"},  

   gw == {
        TeX -> Subscript[g, w],
        ParameterType -> Internal,
        Value -> ee / sw,
        Definitions -> {gw -> ee / sw},
        InteractionOrder -> {QED, 1},
        Description -> "Weak coupling constant"},

   g1 == {
        TeX -> Subscript[g, 1],
        ParameterType -> Internal,
        Value -> ee / cw,
        Definitions -> {g1 -> ee / cw},
        InteractionOrder -> {QED, 1},
        Description -> "U(1)Y coupling constant"},

   gs == {
        TeX -> Subscript[g, s],
        ParameterType -> Internal,
        Value -> Sqrt[4 Pi \[Alpha]S],
        InteractionOrder -> {QCD, 1},
        ParameterName -> G,
        Description -> "Strong coupling constant"},

   v == {
        ParameterType -> Internal,
        BlockName -> VEV,
	Value -> 2*MW*sw/ee,
	InteractionOrder -> {QED, -1},
        Description -> "H1 VEV"},


   x == {
        ParameterType -> Internal,
        BlockName -> VEV,
	Value -> MZp/(2*g1p)*Sqrt[1-gt^2*v^2/(4*MZp^2-v^2*(gw^2+g1^2))],
	InteractionOrder -> {QED, -1},
        Description -> "H2 VEV"},


  Ca == {
        ParameterType -> Internal,
	Value -> Sqrt[1-Sa^2],
	ParameterName -> Ca,
        Description -> "Cosine of Higgses mixing angle"},


   yl == {
        TeX -> Superscript[y, l],
        Indices -> {Index[Generation]},
	AllowSummation -> True,
        ParameterType -> Internal,
        Value -> {yl[1] -> Sqrt[2] yme / v, yl[2] -> Sqrt[2] ymmu / v, yl[3] -> Sqrt[2] ymtau / v},
        ParameterName -> {yl[1] -> ye, yl[2] -> ym, yl[3] -> ytau},
        InteractionOrder -> {QED, 1},
        ComplexParameter -> False,
        Description -> "Lepton Yukawa coupling"},

   yu == {
        Indices -> {Index[Generation]},
	AllowSummation -> True,
        AllowSummation -> True,
        ParameterType -> Internal,
        Value -> {yu[1] -> Sqrt[2] ymup / v, yu[2] -> Sqrt[2] ymc / v, yu[3] -> Sqrt[2] ymt / v},
        ParameterName -> {yu[1] -> yup, yu[2] -> yc, yu[3] -> yt},
        InteractionOrder -> {QED, 1},
        ComplexParameter -> False,
        Description -> "U-quark Yukawa coupling"},

   yd == {
        TeX -> Superscript[y, d],
        Indices -> {Index[Generation]},
	AllowSummation -> True,
        ParameterType -> Internal,
        Value -> {yd[1] -> Sqrt[2] ymdo / v, yd[2] -> Sqrt[2] yms / v, yd[3] -> Sqrt[2] ymb / v},
        ParameterName -> {yd[1] -> ydo, yd[2] -> ys, yd[3] -> yb},
        InteractionOrder -> {QED, 1},
        ComplexParameter -> False,
        Description -> "D-quark Yukawa coupling"},

   ynd == {
        Indices -> {Index[Generation]},
	AllowSummation -> True,
        ParameterType -> Internal,
        Value -> {ynd[1] -> Sqrt[2*MnL1*MnH1]/v,
                  ynd[2] -> Sqrt[2*MnL2*MnH2]/v, 
                  ynd[3] -> Sqrt[2*MnL3*MnH3]/v},
        ParameterName -> {ynd[1] -> ynd1,
                          ynd[2] -> ynd2,
                          ynd[3] -> ynd3},
        InteractionOrder -> {QED, 1},
        ComplexParameter -> False,
        Description -> "Dirac neutrino Yukawa coupling"},

   ynm == {
        Indices -> {Index[Generation]},
        AllowSummation -> True,
        ParameterType -> Internal,
        Value -> {ynm[1] -> (MnH1-MnL1)/Sqrt[2]/x,
	          ynm[2] -> (MnH2-MnL2)/Sqrt[2]/x,
		  ynm[3] -> (MnH3-MnL3)/Sqrt[2]/x},
        ParameterName -> {ynm[1] -> ynm1, ynm[2] -> ynm2, ynm[3] -> ynm3},
        InteractionOrder -> {QED, 1},
        ComplexParameter -> False,
        Description -> "Majorana neutrino Yukawa coupling"},

   Mdd == {
        Indices -> {Index[Generation]},
        AllowSummation -> True,
        ParameterType -> Internal,
        Value -> {Mdd[1] -> ynd1*v/Sqrt[2],
                  Mdd[2] -> ynd2*v/Sqrt[2],
                  Mdd[3] -> ynd3*v/Sqrt[2]},
        ParameterName -> {Mdd[1] -> Mdd1, Mdd[2] -> Mdd2, Mdd[3] -> Mdd3},
        ComplexParameter -> False,
        Description -> "Neutrino Dirac Mass"},

   s12 == {
        TeX -> Subscript[S\[Theta], 12],
        ParameterType -> External,
        BlockName -> CKMBLOCK,
        Value -> 0.221,
        Description -> "Sin(theta_12), PDG-94"},

   s23 == {
        TeX -> Subscript[S\[Theta], 23],
        ParameterType -> External,
        BlockName -> CKMBLOCK,
        Value -> 0.040,
        Description -> "Sin(theta_23), PDG-94"},

   s13 == {
        TeX -> Subscript[S\[Theta], 13],
        ParameterType -> External,
        BlockName -> CKMBLOCK,
        Value -> 0.0035,
        Description -> "Sin(theta_13), PDG-94"},

   c12 == {
        TeX -> Subscript[C\[Theta], 12],
        ParameterType -> Internal,
        BlockName -> CKMBLOCK,
        Value -> Sqrt[1-s12^2],
        Description -> "Cos(theta_12)"},

   c23 == {
        TeX -> Subscript[C\[Theta], 23],
        ParameterType -> Internal,
        BlockName -> CKMBLOCK,
        Value -> Sqrt[1-s23^2],
        Description -> "Cos(theta_23)"},

   c13 == {
        TeX -> Subscript[C\[Theta], 13],
        ParameterType -> Internal,
        BlockName -> CKMBLOCK,
        Value -> Sqrt[1-s13^2],
	Description -> "Cos(theta_13)"},

  CKM == {
       Indices -> {Index[Generation], Index[Generation]},
       TensorClass -> CKM,
       Unitary -> True,
       Value -> {CKM[1,1] -> c12*c13,
                   CKM[1,2] -> s12*c13,
		   CKM[1,3] -> s13,
                   CKM[2,1] -> -s12*c23-c12*s23*s13,
                   CKM[2,2] -> c12*c23-s12*s23*s13,
		   CKM[2,3] -> s23*c13,
                   CKM[3,1] -> s12*s23-c12*c23*s13,
                   CKM[3,2] -> -c12*s23-s12*c23*s13,
		   CKM[3,3] -> c23*c13},		   
       Description -> "CKM-Matrix"},

   San == {
        Indices -> {Index[Generation]},
        AllowSummation -> True,
        ParameterType -> Internal,
        Value -> {San[1] -> Sin[ArcSin[-2*Mdd1/Sqrt[4*Mdd1^2+(MnH1-MnL1)^2]]/2],
	      	  San[2] -> Sin[ArcSin[-2*Mdd2/Sqrt[4*Mdd2^2+(MnH2-MnL2)^2]]/2],
		  San[3] -> Sin[ArcSin[-2*Mdd3/Sqrt[4*Mdd3^2+(MnH3-MnL3)^2]]/2]},
        ParameterName -> {San[1] -> San1, San[2] -> San2, San[3] -> San3},
        ComplexParameter -> False,
        Description -> "Sin-array of neutrino mass-eigenstates"},

   Can == {
        Indices -> {Index[Generation]},
        AllowSummation -> True,
        ParameterType -> Internal,
        Value -> {Can[1] -> Sqrt[1-San1^2],
	      	  Can[2] -> Sqrt[1-San2^2],
		  Can[3] -> Sqrt[1-San3^2]},
        Definitions -> {Can[1]*San1-> Sa2n1/2,
	      	  Can[2]*San2-> Sa2n2/2,
		  Can[3]*San3-> Sa2n3/2,
                  Can[1]^2 -San1^2-> Ca2n1,
	      	  Can[2]^2 -San2^2-> Ca2n2,
		  Can[3]^2 -San3^2-> Ca2n3},
        ParameterName -> {Can[1] -> Can1, Can[2] -> Can2, Can[3] -> Can3},
        ComplexParameter -> False,
        Description -> "Cos-array of neutrino mass-eigenstates"},

	

   \[Lambda]1 == {
        ParameterType -> Internal,
        Value -> MH1^2 /(2*v^2)*Ca^2 + MH2^2 /(2*v^2)*Sa^2,
        ParameterName -> lam1,
        InteractionOrder -> {QED, 2},
        Description -> "Lambda 1"},

   \[Lambda]2 == {
        ParameterType -> Internal,
        Value -> MH1^2 /(2*x^2)*Sa^2 + MH2^2 /(2*x^2)*Ca^2,
        ParameterName -> lam2,
        InteractionOrder -> {QED, 2},
        Description -> "Lambda 2"},

   \[Lambda]3 == {
        ParameterType -> Internal,
        Value -> (MH2^2 - MH1^2)/(x*v)*Sa*Ca,
        ParameterName -> lam3,
        InteractionOrder -> {QED, 2},
        Description -> "Lambda 3, mixing parameter"},

   mu2H1 == {
        ParameterType -> Internal,
	Value -> - \[Lambda]1 * v^2 - \[Lambda]3 /2 * x^2,
        TeX -> m^2,
        Description -> "Coefficient of the quadratic piece of the H1 potential"},

   mu2H2 == {
        ParameterType -> Internal,
	Value -> - \[Lambda]3 /2 * v^2 - \[Lambda]2 * x^2,
        TeX -> \[Mu]^2,
        Description -> "Coefficient of the quadratic piece of the H2 potential"},


   Sp2num == {
        ParameterType -> Internal,
        Value -> 2*gt*Sqrt[(ee/sw)^2+(ee/cw)^2]},

   Cp2num == {
        ParameterType -> Internal,
        Value -> gt^2+16*(x/v)^2*g1p^2-(ee/sw)^2-(ee/cw)^2},

   Sp == {
        AllowSummation -> True,
        ParameterType -> Internal,
        Value -> Sin[ArcSin[Sp2num/Sqrt[Sp2num^2+Cp2num^2]]/2],
        ComplexParameter -> False,
        Description -> "Sin-array of neutrino mass-eigenstates"},

   Cp == {
        AllowSummation -> True,
        ParameterType -> Internal,
        Value -> Sqrt[1-Sp^2],
        ComplexParameter -> False,
        Description -> "Cos-array of neutrino mass-eigenstates"},


   Cn == {
        ParameterType -> Internal,
        ComplexParameter -> False,
        Value -> (ee/sw)^2+(ee/cw)^2+gt^2+16*(x/v)^2*g1p^2},

   Dn == {
        ParameterType -> Internal,
        ComplexParameter -> False,
        Value -> 64*((ee/sw)^2+(ee/cw)^2)*g1p^2*v^2*x^2},
	

   MZ == {
        ParameterType -> Internal,
        Value -> Sqrt[(Cn*v^2-Sqrt[-Dn+v^4*Cn^2])/8],
        Description -> "Z mass"},

   S2gNum == {
        ParameterType -> Internal,
        ComplexParameter -> False,
        Value -> 8*x/v*gt*g1p},

   C2gNum == {
        ParameterType -> Internal,
        ComplexParameter -> False,
        Value -> (ee/sw)^2+(ee/cw)^2+gt^2-16*(x/v)^2*g1p^2},    

   
   sg == {
        ParameterType -> Internal,
        ComplexParameter -> False,
        Value -> Sin[ArcSin[-S2gNum/Sqrt[S2gNum^2+C2gNum^2]]/2],
        Description -> "cosine of Z-Zp goldostone mixing angle"},
	
   cg == {
        ParameterType -> Internal,
        ComplexParameter -> False,
	Value -> Sqrt[1-sg^2],
        Description -> "sine of Z-Zp goldstone mixing angle"}
(*        Value -> Cos[ArcCos[-C2gNum/Sqrt[S2gNum^2+C2gNum^2]]/2], *)
}

(************** Gauge Groups ******************)

M$GaugeGroups = {

  U1BL == {
        Abelian -> True,
        GaugeBoson -> Bp,
        Charge -> BL,
        CouplingConstant -> g1p},
  
  U1Y == {
        Abelian -> True,
        GaugeBoson -> B,
        Charge -> Y,
        CouplingConstant -> g1},

  SU2L == {
        Abelian -> False,
        GaugeBoson -> Wi,
        StructureConstant -> Eps,
        CouplingConstant -> gw},

  SU3C == {
        Abelian -> False,
        GaugeBoson -> G,
        StructureConstant -> f,
        SymmetricTensor -> dSUN,
        Representations -> {T, Colour},
        CouplingConstant -> gs}
}

(********* Particle Classes **********)

M$ClassesDescription = {

(********** Fermions ************)

	(* Mass-Eigenstate light neutrino: Q = 0, BL= -1 *)

  F[11] == {
        ClassName -> nL,
        ClassMembers -> {nL1, nL2, nL3},
        FlavorIndex -> Generation,
	SelfConjugate -> True,
	Indices -> {Index[Generation]},
	Mass -> {MnL, {MnL1, 10^(-9)}, {MnL2, 10^(-9)}, {MnL3, 10^(-9)}},
        Width -> 0,
	PropagatorLabel -> {"nL", "nul1", "nul2", "nul3"},
	PropagatorType -> Straight,
        ParticleName -> {"n1", "n2", "n3"},
	PropagatorArrow -> Forward,
	PDG -> {12, 14, 16},
        FullName -> {"Light neutrino 1", "Light neutrino 2", "Light neutrino 3"} },

	(* Mass-Eigenstate heavy neutrino: Q = 0, BL= -1 *)

  F[12] == {
        ClassName -> nH,
        ClassMembers -> {nH1, nH2, nH3},
        FlavorIndex -> Generation,
	SelfConjugate -> True,
	Indices -> {Index[Generation]},
	Mass -> {MnH, {MnH1, 200.00}, {MnH2, 200.00}, {MnH3, 200.00}},
        Width -> 10^(-13),
	PropagatorLabel -> {"nH", "nuh1", "nuh2", "nuh3"},
	PropagatorType -> Straight,
        ParticleName -> {"~n1", "~n2", "~n3"},
	PropagatorArrow -> Forward,
        PDG -> {9100012, 9100014, 9100016},
        FullName -> {"Heavy neutrino 1", "Heavy neutrino 2", "Heavy neutrino 3"} },

	(* Left-handed neutrino: unphysical *)
  F[13] == {
        ClassName -> nF,
        ClassMembers -> {nF1,nF2,nF3},
        FlavorIndex -> Generation,
	SelfConjugate -> True,
	Indices -> {Index[Generation]},
        Unphysical -> True,
	Definitions -> {nF[s_,i_] -> Can[i] nL[s,i]-San[i] nH[s,i]},
        FullName -> {"Majorana LH component of Dirac neutrino 1",
		    "Majorana LH component of Dirac neutrino 2", "Majorana LH component of Dirac neutrino 3"} },

	(* Right-handed neutrino: unphysical *)
  F[14] == {
        ClassName -> nR,
        ClassMembers -> {nR1,nR2,nR3},
        FlavorIndex -> Generation,
	SelfConjugate -> True,
	Indices -> {Index[Generation]},
        Unphysical -> True,
	Definitions -> {nR[s_,i_] -> San[i] nL[s,i]+Can[i] nH[s,i]},
        FullName -> {"Majorana LH component of Dirac neutrino 1", "Majorana LH component of Dirac neutrino 2", "Majorana LH component of Dirac neutrino 3"} },


	(* Flavour-eigenstate neutrino: unphysical *)
  F[15] == {
        ClassName -> vl,
        ClassMembers -> {vle,vlm,vlt},
        FlavorIndex -> Generation,
	SelfConjugate -> False,
	Indices -> {Index[Generation]},
	QuantumNumbers -> {Q -> 0, LeptonNumber -> 1, BarionLepton -> -1},
        Unphysical -> True,
	Definitions -> {vl[s_,i_] -> left[nF[s,i]]+right[nR[s,i]]},
        ParticleName -> {"nue", "num", "nut"},
        AntiParticleName -> {"nue-bar", "num-bar", "nut-bar"},
        FullName -> {"Electron-neutrino", "Mu-neutrino", "Tau-neutrino"} },
	

	(* Leptons (electron): I_3 = -1/2, Q = -1, BL= -1 *)
  F[2] == {
        ClassName -> l,
        ClassMembers -> {e, m, tt},
        FlavorIndex -> Generation,
	SelfConjugate -> False,
	Indices -> {Index[Generation]},
	Mass -> {Ml, {ME, 0.000511}, {MM, 0.1057}, {MTA, 1.777}},
        Width -> 0,
	QuantumNumbers -> {Q -> -1, LeptonNumber -> 1, BarionLepton -> -1},
	PropagatorLabel -> {"l", "e", "m", "tt"},
	PropagatorType -> Straight,
        ParticleName -> {"e", "m", "l"},
        AntiParticleName -> {"E", "M", "L"},
	PropagatorArrow -> Forward,
        PDG -> {11, 13, 15},
        FullName -> {"Electron", "Muon", "Tau"} },

	(* Quarks (u): I_3 = +1/2, Q = +2/3, BL=1/3 *)
  F[3] == {
        ClassMembers -> {u, c, t},
        ClassName -> uq,
        FlavorIndex -> Generation,
	SelfConjugate -> False,
	Indices -> {Index[Generation], Index[Colour]},
	Mass -> {Mu, {MU, 2.55*10^(-3)}, {MC, 1.27}, {MT, 172.0}},
        Width -> {0, 0, {WT, 1.50833649}},
	QuantumNumbers -> {Q -> 2/3, BarionLepton -> 1/3},
	PropagatorLabel -> {"uq", "u", "c", "t"},
        ParticleName -> {"u", "c", "t"},
        AntiParticleName -> {"U", "C", "T"},	
	PropagatorType -> Straight,
	PropagatorArrow -> Forward,
        PDG -> {2, 4, 6}, 
        FullName -> {"u-quark", "c-quark", "t-quark"}},

	(* Quarks (d): I_3 = -1/2, Q = -1/3, BL=1/3 *) 
  F[4] == {
        ClassMembers -> {d, s, b},
        ClassName -> dq,
        FlavorIndex -> Generation,
	SelfConjugate -> False,
	Indices -> {Index[Generation], Index[Colour]},
	Mass -> {Md, {MD,  5.04*10^(-3)}, {MS, 0.101}, {MB, 4.7}},
        Width -> 0,
	QuantumNumbers -> {Q -> -1/3, BarionLepton -> 1/3},
        ParticleName -> {"d", "s", "b"},
        AntiParticleName -> {"D", "S", "B"},
	PropagatorLabel -> {"dq", "d", "s", "b"},
	PropagatorType -> Straight, 
	PropagatorArrow -> Forward,
        PDG -> {1,3,5},
        FullName -> {"d-quark", "s-quark", "b-quark"} },

	
(********** Ghosts **********)
	U[1] == {
       ClassName -> ghA,
       SelfConjugate -> False,
       Indices -> {},
       Ghost -> A,
       Mass -> 0,
       QuantumNumbers -> {GhostNumber -> 1},
       PropagatorLabel -> uA,
       PropagatorType -> GhostDash,
       PropagatorArrow -> Forward},

	U[2] == {
       ClassName -> ghZ,
       SelfConjugate -> False,
       Indices -> {},
       Mass -> {MZ, Internal},
       Ghost -> Z,
       QuantumNumbers -> {GhostNumber -> 1},
       PropagatorLabel -> uZ,
       PropagatorType -> GhostDash,
       PropagatorArrow -> Forward},

	U[31] == {
       ClassName -> ghWp,
       SelfConjugate -> False,
       Indices -> {},
       Mass -> {MW, Internal},
       Ghost -> W,
       QuantumNumbers -> {Q-> 1, GhostNumber -> 1},
       PropagatorLabel -> uWp,
       PropagatorType -> GhostDash,
       PropagatorArrow -> Forward},

   U[32] == {
       ClassName -> ghWm,
       SelfConjugate -> False,
       Indices -> {},
       Mass -> {MW, Internal},
       Ghost -> Wbar,
       QuantumNumbers -> {Q-> -1, GhostNumber -> 1},
       PropagatorLabel -> uWm,
       PropagatorType -> GhostDash,
       PropagatorArrow -> Forward},

 	U[4] == {
       ClassName -> ghG,
       SelfConjugate -> False,
       Indices -> {Index[Gluon]},
       Ghost -> G,
       Mass -> 0,
       QuantumNumbers -> {GhostNumber -> 1},
       PropagatorLabel -> uG,
       PropagatorType -> GhostDash,
       PropagatorArrow -> Forward},

	U[5] == {
        ClassName -> ghWi,
        Unphysical -> True,
        Definitions -> {ghWi[1] -> (ghWp + ghWm)/Sqrt[2],
                        ghWi[2] -> (ghWm - ghWp)/Sqrt[2]/I,
                        ghWi[3] -> cw*Cp ghZ + sw ghA - cw*Sp ghZp},
        SelfConjugate -> False,
        Indices -> {Index[SU2W]},
        FlavorIndex -> SU2W,
	Ghost -> Wi},

	U[6] == {
        ClassName -> ghB,
	SelfConjugate -> False,
        Definitions -> {ghB -> -sw*Cp ghZ + cw ghA + sw*Sp ghZp},
	Indices -> {},
        Unphysical -> True,
	Ghost -> B},

	U[7] == {
       ClassName -> ghZp,
       SelfConjugate -> False,
       Indices -> {},
       Mass -> {MZp, Internal},
       Ghost -> Zp,
       QuantumNumbers -> {GhostNumber -> 1},
       PropagatorLabel -> uZp,
       PropagatorType -> GhostDash,
       PropagatorArrow -> Forward},

	U[8] == {
        ClassName -> ghBp,
	SelfConjugate -> False,
        Definitions -> {ghBp -> Sp ghZ + Cp ghZp},
	Indices -> {},
        Unphysical -> True,
	Ghost -> Bp},

(************ Gauge Bosons ***************)
	(* Gauge bosons: Q = 0 *)
  V[1] == {
        ClassName -> A,
	SelfConjugate -> True,
	Indices -> {},
	Mass -> 0,
        Width -> 0,
	PropagatorLabel -> "a",
	PropagatorType -> W,
	PropagatorArrow -> None,
        PDG -> 22,
        FullName -> "Photon" },

  V[2] == {
        ClassName -> Z,
	SelfConjugate -> True, 
	Indices -> {},
	Mass -> {MZ, Internal},
        Width -> {WZ, 2.4952},
	PropagatorLabel -> "Z",
	PropagatorType -> Sine,
	PropagatorArrow -> None,
        PDG -> 23,
        FullName -> "Z" },

	(* Gauge bosons: Q = -1 *)
  V[3] == {
        ClassName -> W,
	SelfConjugate -> False,
	Indices -> {},
	Mass -> {MW, Internal},
        Width -> {WW, 2.085},
	QuantumNumbers -> {Q -> 1},
	PropagatorLabel -> "W",
	PropagatorType -> Sine,
	PropagatorArrow -> Forward,
	ParticleName ->"W+",
	AntiParticleName ->"W-",
        PDG -> 24,
        FullName -> "W" },

V[4] == {
        ClassName -> G,
	SelfConjugate -> True,
	Indices -> {Index[Gluon]},
	Mass -> 0,
        Width -> 0,
        PropagatorLabel -> G,
	PropagatorType -> C,
	PropagatorArrow -> None,
        PDG -> 21,
        FullName -> "G" },

V[5] == {
        ClassName -> Wi,
        Unphysical -> True,
        Definitions -> {Wi[mu_, 1] -> (W[mu] + Wbar[mu])/Sqrt[2],
                        Wi[mu_, 2] -> (Wbar[mu] - W[mu])/Sqrt[2]/I,
                        Wi[mu_, 3] -> sw A[mu] + cw*Cp Z[mu] - cw*Sp Zp[mu]},
        SelfConjugate -> True,
        Indices -> {Index[SU2W]},
        FlavorIndex -> SU2W,
        Mass -> 0,
        PDG -> {1,2,3}},

V[6] == {
        ClassName -> B,
	SelfConjugate -> True,
        Definitions -> {B[mu_] -> cw A[mu] - sw*Cp Z[mu] + sw*Sp Zp[mu]},
	Indices -> {},
	Mass -> 0,
        Unphysical -> True},

V[7] == {
        ClassName -> Zp,
	SelfConjugate -> True, 
	Indices -> {},
	Mass -> {MZp, Internal},
        Width -> {WZp, 80.00},
	PropagatorLabel -> "Zp",
	PropagatorType -> Sine,
	PropagatorArrow -> None,
        PDG -> 9900032,
        FullName -> "Zp" },

V[8] == {
        ClassName -> Bp,
	SelfConjugate -> True,
        Definitions -> {Bp[mu_] ->  Sp Z[mu] + Cp Zp[mu]},
	Indices -> {},
        Unphysical -> True},


(************ Scalar Fields **********)
	(* physical Higgs: Q = 0 *) 
  S[1] == {
        ClassName -> H1,
	SelfConjugate -> True,
	Mass -> {MH1, Internal},
        Width -> {WH1, 1.5},
	PropagatorLabel -> "H1",
	PropagatorType -> D,
	PropagatorArrow -> None,
        PDG -> 9900025,
        FullName -> "H1" },

  S[2] == {
        ClassName -> phiZ,
	SelfConjugate -> True,
	Mass -> {MZ, Internal},
        Width -> Wphi,
	PropagatorLabel -> "PhiZ",
	PropagatorType -> D,
	PropagatorArrow -> None,
	ParticleName ->"phiZ",
        PDG -> 9900250,
        FullName -> "PhiZ",
        Goldstone -> Z },

  S[3] == {
        ClassName -> phi2,
	SelfConjugate -> False,
	Mass -> {MW, Internal},
        Width -> Wphi2,
	PropagatorLabel -> "Phi2",
	PropagatorType -> D,
	PropagatorArrow -> None,
	ParticleName ->"phi+",
	AntiParticleName ->"phi-",
        PDG -> 9900251,
        FullName -> "Phi2",
        Goldstone -> W,
        QuantumNumbers -> {Q -> 1}},

  S[4] == {
        ClassName -> H2,
	SelfConjugate -> True,
	Mass -> {MH2, Internal}, 
        Width -> {WH2, 10},
	PropagatorLabel -> "H2",
	PropagatorType -> D,
	PropagatorArrow -> None,
        PDG -> 9900026,
        FullName -> "H2" },
   
  S[5] == {
        ClassName -> phiZp,
	SelfConjugate -> True,
	Mass -> {MZp, Internal},
        Width -> WphiZp,
	PropagatorLabel -> "PhiZp",
	PropagatorType -> D,
	PropagatorArrow -> None,
	ParticleName ->"phiZp",
        PDG -> 9900252,
        FullName -> "PhiZp",
        Goldstone -> Zp },
	
  S[6] == {
        ClassName -> phi,
	Unphysical -> True,
        Definitions -> {phi -> cg phiZ - sg phiZp},
	SelfConjugate -> True},

  S[7] == {
        ClassName -> phip,
	Unphysical -> True,
        Definitions -> {phip -> sg phiZ + cg phiZp},
	SelfConjugate -> True},

  S[8] == {
        ClassName -> phic,
	Unphysical -> True,
        Definitions -> {phic[1] -> (phi2 + phi2bar)/Sqrt[2],
                        phic[2] -> (phi2bar - phi2)/Sqrt[2]/I},
        SelfConjugate -> False}
	
}


(*****************************************************************************************)

(* mZp Lagrangian *)

(******************** Gauge F^2 Lagrangian terms*************************)
(*Sign convention from Lagrangian in between Eq. (A.9) and Eq. (A.10) of Peskin & Schroeder.*)
 LGauge = -1/4 (del[Wi[nu, i1], mu] - del[Wi[mu, i1], nu] + gw Eps[i1, i2, i3] Wi[mu, i2] Wi[nu, i3])*
 	(del[Wi[nu, i1], mu] - del[Wi[mu, i1], nu] + gw Eps[i1, i4, i5] Wi[mu, i4] Wi[nu, i5]) -
	
	1/4 (del[B[nu], mu] - del[B[mu], nu])^2 - 1/4 (del[Bp[nu], mu] - del[Bp[mu], nu])^2 -
	
	1/4 (del[G[nu, a1], mu] - del[G[mu, a1], nu] + gs f[a1, a2, a3] G[mu, a2] G[nu, a3])*
 		 (del[G[nu, a1], mu] - del[G[mu, a1], nu] + gs f[a1, a4, a5] G[mu, a4] G[nu, a5]);


(********************* Fermion Lagrangian terms*************************)
(*Sign convention from Lagrangian in between Eq. (A.9) and Eq. (A.10) of Peskin & Schroeder.*)
 LFermions = Module[{Lkin, LQCD, LEWleft, LEWright},

    Lkin = I uqbar.Ga[mu].del[uq, mu] + 
    	I dqbar.Ga[mu].del[dq, mu] + 
    	I lbar.Ga[mu].del[l, mu] + 
	I left[anti[vl]].Ga[mu].del[left[vl],mu] +
	I right[anti[vl]].Ga[mu].del[right[vl],mu];

    LQCD = gs (uqbar.Ga[mu].T[a].uq + 
    	dqbar.Ga[mu].T[a].dq)G[mu, a];

    LBright = 
       -2ee/cw B[mu]/2 lbar.Ga[mu].ProjP.l +           (*Y_lR=-2*)
    	4ee/3/cw B[mu]/2 uqbar.Ga[mu].ProjP.uq -       (*Y_uR=4/3*)
    	2ee/3/cw B[mu]/2 dqbar.Ga[mu].ProjP.dq;        (*Y_dR=-2/3*)

    LBleft = 
       -ee/cw B[mu]/2 left[anti[vl]].Ga[mu].ProjM.vl -          (*Y_LL=-1*)
    	ee/cw B[mu]/2 lbar.Ga[mu].ProjM.l  +           (*Y_LL=-1*)
    	ee/3/cw B[mu]/2 uqbar.Ga[mu].ProjM.uq +        (*Y_QL=1/3*)
    	ee/3/cw B[mu]/2 dqbar.Ga[mu].ProjM.dq ;        (*Y_QL=1/3*)
	
    LWleft = ee/sw/2(
	   left[anti[vl]].Ga[mu].ProjM.vl Wi[mu, 3] -              (*sigma3 = ( 1   0 )*)
    	   lbar.Ga[mu].ProjM.l Wi[mu, 3] +                (*         ( 0  -1 )*)
    	
    	Sqrt[2] left[anti[vl]].Ga[mu].ProjM.l W[mu] +
    	Sqrt[2] lbar.Ga[mu].ProjM.vl Wbar[mu]+
    	
    	uqbar.Ga[mu].ProjM.uq Wi[mu, 3] -              (*sigma3 = ( 1   0 )*)
    	dqbar.Ga[mu].ProjM.dq Wi[mu, 3] +              (*         ( 0  -1 )*)
    	
    	Sqrt[2] uqbar.Ga[mu].ProjM.CKM.dq W[mu] +
    	Sqrt[2] dqbar.Ga[mu].ProjM.HC[CKM].uq Wbar[mu]
	);

    LBpright =       
    	     - g1p Bp[mu] right[anti[vl]].Ga[mu].ProjP.vl +          (*Y_vlR=0, BL_vlR=-1*)
      	     (-1*gt -g1p) Bp[mu] lbar.Ga[mu].ProjP.l +           (*Y_lR=-1, BL_lR=-1*)
      	     (2/3*gt + g1p/3) Bp[mu] uqbar.Ga[mu].ProjP.uq +       (*Y_uR=2/3, BL_uR=1/3*)
      	     (-1/3*gt + g1p/3) Bp[mu] dqbar.Ga[mu].ProjP.dq;        (*Y_dR=-1/3, BL_dR=1/3*)

    LBpleft = 
   	   - (gt/2 + g1p) Bp[mu] left[anti[vl]].Ga[mu].ProjM.vl -          (*Y_lL=-1/2, BL_vlL=-1*)
      	     (gt/2 + g1p) Bp[mu] lbar.Ga[mu].ProjM.l +           (*Y_lL=-1/2, BL_lL=-1*)
      	     (gt/3/2 + g1p/3) Bp[mu] uqbar.Ga[mu].ProjM.uq +       (*Y_qL=1/6, BL_uL=1/3*)
      	     (gt/3/2 + g1p/3) Bp[mu] dqbar.Ga[mu].ProjM.dq         (*Y_qL=1/6, BL_dL=1/3*)
	     ;

    Lkin + LQCD + LBright + LBleft + LWleft + LBpright + LBpleft ];

(******************** Higgs Lagrangian terms****************************)
 Phi := If[FeynmanGauge, {-I phi2, (v + Ca*H1+Sa*H2 + I phi)/Sqrt[2]}, {0, (v + Ca*H1+Sa*H2)/Sqrt[2]}];
 Phibar := If[FeynmanGauge, {I phi2bar, (v + Ca*H1+Sa*H2 - I phi)/Sqrt[2]} ,{0, (v + Ca*H1+Sa*H2)/Sqrt[2]}];
 
 Chi := If[FeynmanGauge, {(x -Sa*H1+Ca*H2 + I phip)/Sqrt[2]}, {(x -Sa*H1+Ca*H2)/Sqrt[2]}];
 Chibar := If[FeynmanGauge, {(x -Sa*H1+Ca*H2 - I phip)/Sqrt[2]}, {(x -Sa*H1+Ca*H2)/Sqrt[2]}];
    
 LHiggs := Block[{PMVec, WVec, Dc, Dcbar, Vphichi, Dcp, Dcpbar},
    
    PMVec = Table[PauliSigma[i], {i, 3}];   
    Wvec[mu_] := {Wi[mu, 1], Wi[mu, 2], Wi[mu, 3]};

	(*Y_phi=1/2*)
    Dc[f_, mu_] := I del[f, mu] + ee/cw B[mu]/2 f + gt/2 Bp[mu] f + ee/sw/2 (Wvec[mu].PMVec).f;
    Dcbar[f_, mu_] := -I del[f, mu] + ee/cw B[mu]/2 f + gt/2 Bp[mu] f + ee/sw/2 f.(Wvec[mu].PMVec);

	(*BL_phi=2*)
    Dcp[f_, mu_] := I del[f, mu] + 2*g1p Bp[mu] f ;
    Dcpbar[f_, mu_] := -I del[f, mu] + 2*g1p Bp[mu] f ; 

    Vphichi[Phi_, Phibar_, Chi_, Chibar_] := mu2H1 Phibar.Phi + mu2H2 Chibar.Chi +
    	\[Lambda]1 (Phibar.Phi)^2 + \[Lambda]2 (Chibar.Chi)^2 + \[Lambda]3 (Phibar.Phi)*(Chibar.Chi);

    (Dcbar[Phibar, mu]).Dc[Phi, mu] + (Dcpbar[Chibar, mu]).Dcp[Chi, mu] - Vphichi[Phi, Phibar, Chi, Chibar]  

   ];
    


(*************** Yukawa Lagrangian***********************)
(*NOTE: Neutrino states have been expanded on the LH/RH components and the LH mass terms have been changed signs*)

LYuk := If[FeynmanGauge,

      Module[{s,r,n,m,i},                                             			 -
	      yd[m] CKM[n,m]     uqbar[s,n,i].ProjP[s,r].dq[r,m,i] (-I phi2)              -
	      yd[n]              dqbar[s,n,i].ProjP[s,r].dq[r,n,i] (v+Ca*H1+Sa*H2 +I phi)/Sqrt[2]   -
	 
	      yu[n]              uqbar[s,n,i].ProjP[s,r].uq[r,n,i] (v+Ca*H1+Sa*H2 -I phi)/Sqrt[2]   + (*This sign from eps matrix*)
	      yu[m] Conjugate[CKM[m,n]] dqbar[s,n,i].ProjP[s,r].uq[r,m,i] ( I phi2bar)    
	
  	     - yl[n]    Can[n]    anti[nL][s,n].ProjP[s,r].l[r,n]      (-I phi2) 
  	     + yl[n]    San[n]    anti[nH][s,n].ProjP[s,r].l[r,n]      (-I phi2) 

	     - yl[n]               lbar[s,n].ProjP[s,r].l[r,n]      (v+Ca*H1+Sa*H2 +I phi)/Sqrt[2] +
	      ynd[n]  San[n] Can[n] anti[nL][s,n].ProjP[s,r].nL[r,n]         (v+Ca*H1+Sa*H2 - I phi)/Sqrt[2]+
	      ynd[n]  San[n] Can[n] anti[nH][s,n].ProjP[s,r].nH[r,n]         (v+Ca*H1+Sa*H2 - I phi)/Sqrt[2]-

	      ynd[n]  (Can[n] Can[n] - San[n] San[n]) anti[nL][s,n].ProjP[s,r].nH[r,n]         (v+Ca*H1+Sa*H2 - I phi)/Sqrt[2]

	    - ynd[n]	San[n]	  lbar[s,n].ProjP[s,r].nL[r,n]     		     (I phi2bar)  +
	      ynd[n]	Can[n]	  lbar[s,n].ProjP[s,r].nH[r,n]     		     (I phi2bar)  +

	      ynm[n] San[n] San[n]  anti[nL][s,n].ProjP[s,r].nL[r,n]          (x-Sa*H1+Ca*H2 + I phip)/Sqrt[2]-
	      ynm[n] Can[n] Can[n]   anti[nH][s,n].ProjP[s,r].nH[r,n]          (x-Sa*H1+Ca*H2 + I phip)/Sqrt[2]-

	      ynm[n] 2 San[n] Can[n]  anti[nL][s,n].ProjP[s,r].nH[r,n]         (x-Sa*H1+Ca*H2 + I phip)/Sqrt[2]


           ],
           
	   Module[{s,r,n,m,i},                                                              -
	      yd[n]              dqbar[s,n,i].ProjP[s,r].dq[r,n,i] (v+Ca*H1+Sa*H2)/Sqrt[2]  -
	      yu[n]              uqbar[s,n,i].ProjP[s,r].uq[r,n,i] (v+Ca*H1+Sa*H2)/Sqrt[2]  

	     - yl[n]               lbar[s,n].ProjP[s,r].l[r,n]      (v+Ca*H1+Sa*H2)/Sqrt[2] +
	      ynd[n]  San[n] Can[n] anti[nL][s,n].ProjP[s,r].nL[r,n]         (v+Ca*H1+Sa*H2)/Sqrt[2]+
	      ynd[n]  San[n] Can[n] anti[nH][s,n].ProjP[s,r].nH[r,n]         (v+Ca*H1+Sa*H2)/Sqrt[2]-

	      ynd[n]  (Can[n] Can[n] - San[n] San[n]) anti[nL][s,n].ProjP[s,r].nH[r,n]         (v+Ca*H1+Sa*H2)/Sqrt[2]+



	      ynm[n] San[n] San[n]  anti[nL][s,n].ProjP[s,r].nL[r,n]          (x-Sa*H1+Ca*H2)/Sqrt[2]-
	      ynm[n] Can[n] Can[n]   anti[nH][s,n].ProjP[s,r].nH[r,n]          (x-Sa*H1+Ca*H2)/Sqrt[2]-

	      ynm[n] 2 San[n] Can[n]  anti[nL][s,n].ProjP[s,r].nH[r,n]         (x-Sa*H1+Ca*H2)/Sqrt[2]

           ]
         ];

LYukawa := LYuk + HC[LYuk];



(**************Ghost terms**************************)
(* Now we need the ghost terms which are of the form:             *)
(* - g * antighost * d_BRST G                                     *)
(* where d_BRST G is BRST transform of the gauge fixing function. *)

LGhost := If[FeynmanGauge, 
		Block[{dBRSTG,LGhostG,dBRSTWi,LGhostWi,dBRSTB,LGhostB, dBRSTBp, LGhostBp},
		
	(***********First the pure gauge piece.**********************) 	
	dBRSTG[mu_,a_] := 1/gs Module[{a2, a3}, del[ghG[a], mu] + gs f[a,a2,a3] G[mu,a2] ghG[a3]];
		LGhostG := - gs ghGbar[a].del[dBRSTG[mu,a],mu];
 	
	dBRSTWi[mu_,i_] := sw/ee Module[{i2, i3}, del[ghWi[i], mu] + ee/sw Eps[i,i2,i3] Wi[mu,i2] ghWi[i3] ];
		LGhostWi := - ee/sw ghWibar[a].del[dBRSTWi[mu,a],mu];	
 	
	dBRSTB[mu_] := cw/ee del[ghB, mu];
		LGhostB := - ee/cw ghBbar.del[dBRSTB[mu],mu];

	dBRSTBp[mu_] := 1/g1p del[ghBp, mu];
		LGhostBp := - g1p ghBpbar.del[dBRSTBp[mu],mu];
	
	(***********Next the piece from the scalar field.************)
	LGhostphi := 
	(*-   ee/(2*sw*cw) MW ( - I phi2 ( (cw^2-sw^2)ghWpbar.ghZ + 2sw*cw ghWpbar.ghA )  +
		    I phi2bar ( (cw^2-sw^2)ghWmbar.ghZ + 2sw*cw ghWmbar.ghA ) ) -
		     ee/(2*sw) MW ( ( (v+Ca*H1+Sa*H2) + I phi) ghWpbar.ghWp +
		  ( (v+Ca*H1+Sa*H2) - I phi) ghWmbar.ghWm   ) -
		I ee/(2*sw) MZ ( - phi2bar ghZbar.ghWp + phi2 ghZbar.ghWm ) -
				  ee/(2*sw*cw) MZ (v+Ca*H1+Sa*H2) ghZbar.ghZ *)

1/4*gw*v (-gw*(v+Ca*H1+Sa*H2) ghWibar[1].ghWi[1] +g1 phic[2] ghWibar[1].ghB +gw phic[2] ghWibar[1].ghWi[3] -gw phi ghWibar[1].ghWi[2] + gt phic[2] ghWibar[1].ghBp) +
1/4*gw*v (-gw*(v+Ca*H1+Sa*H2) ghWibar[2].ghWi[2] -g1 phic[1] ghWibar[2].ghB -gw phic[1] ghWibar[2].ghWi[3] +gw phi ghWibar[2].ghWi[1] -gt phic[1] ghWibar[2].ghBp) +
1/4*gw*v (g1*(v+Ca*H1+Sa*H2) ghWibar[3].ghB -gw*(v+Ca*H1+Sa*H2) ghWibar[3].ghWi[3] +gw phic[1] ghWibar[3].ghWi[2] 
-gw phic[2] ghWibar[3].ghWi[1]   +gt (v+Ca*H1+Sa*H2) ghWibar[3].ghBp ) + 
1/4*g1*v (-g1*(v+Ca*H1+Sa*H2) ghBbar.ghB +gw*(v+Ca*H1+Sa*H2) ghBbar.ghWi[3] -gw phic[1] ghBbar.ghWi[2] +gw phic[2] ghBbar.ghWi[1] -gt*(v+Ca*H1+Sa*H2) ghBbar.ghBp) +
1/4*gt*v (-g1*(v+Ca*H1+Sa*H2) ghBpbar.ghB +gw*(v+Ca*H1+Sa*H2) ghBpbar.ghWi[3] -gw phic[1] ghBpbar.ghWi[2] +gw phic[2] ghBpbar.ghWi[1] -gt (v+Ca*H1+Sa*H2) ghBpbar.ghBp) -
4*g1p^2*x*(x-Sa*H1+Ca*H2) ghBpbar.ghBp 
;
		 	
	(***********Now add the pieces together.********************)
	LGhostG + LGhostWi + LGhostB + LGhostphi + LGhostBp ]

, 

        (*If unitary gauge, only include the gluonic ghost.*)
                Block[{dBRSTG,LGhostG},
		
	(***********First the pure gauge piece.**********************) 	
	dBRSTG[mu_,a_] := 1/gs Module[{a2, a3}, del[ghG[a], mu] + gs f[a,a2,a3] G[mu,a2] ghG[a3]];
		LGhostG := - gs ghGbar[a].del[dBRSTG[mu,a],mu];		 	
		 	
	(***********Now add the pieces together.********************)
	LGhostG]

];
		
(*********Total SM Lagrangian*******)	 	
LmZp := LGauge + LHiggs + LFermions + LYukawa  + LGhost;
