
/** \class ExRootCandidate
 *
 *  Class implementing particle candidate model.
 *
 *  $Date: 2008-06-04 13:57:54 $
 *  $Revision: 1.1 $
 *
 *
 *  \author P. Demin - UCL, Louvain-la-Neuve
 *
 */

#include "ExRootAnalysis/ExRootCandidate.h"
#include "ExRootAnalysis/ExRootFactory.h"

#include "TDatabasePDG.h"
#include "TBrowser.h"
#include "TClass.h"

#include <map>
#include <iostream>

using namespace std;

ExRootCompare *ExRootCandidate::fgCompare = 0;

ExRootCandidate::ExRootCandidate() :
  fIsResonance(kFALSE),
  fCharge(0),
  fLorentzVector(0.0, 0.0, 0.0, 0.0),
  fMother(0)
{
}

//------------------------------------------------------------------------------

ExRootCandidate::ExRootCandidate(const ExRootCandidate &object) :
  fIsResonance(kFALSE),
  fCharge(0),
  fLorentzVector(0.0, 0.0, 0.0, 0.0),
  fMother(0)
{
  object.Copy(*this);
}

//------------------------------------------------------------------------------

Bool_t ExRootCandidate::IsCloneOf(const ExRootCandidate *object, Bool_t checkType) const
{
  // Original behaviour of ExRootCandidate::IsCloneOf()
  if(object->GetUniqueID() == GetUniqueID() && !checkType) return kTRUE;

  if((IsComposite() && !object->IsComposite()) ||
     (!IsComposite() && object->IsComposite())) return kFALSE;

  // forsingle tracks and clusters, it is enough to compare
  // UIDs and PDT types
  if(!IsComposite() && !object->IsComposite())
  {
    return (object->GetUniqueID() == GetUniqueID() &&
            (!checkType || GetType() == object->GetType()));
  }

  // if we got here, must be true
  return kTRUE;
}

//------------------------------------------------------------------------------

const ExRootCandidate *ExRootCandidate::FindCloneInTree(const ExRootCandidate *object) const
{
  if(IsCloneOf(object)) return this;
  const ExRootCandidate *daughter;
  ExRootCandConstIter itDaughters(Iterator());
  while((daughter = itDaughters.Next()))
  {
    daughter = daughter->FindCloneInTree(object);
    if(daughter) return daughter;
  }
  return 0;
}

//------------------------------------------------------------------------------

Bool_t ExRootCandidate::Overlaps(const ExRootCandidate *object) const
{
  if(object->GetUniqueID() == GetUniqueID()) return kTRUE;

  const ExRootCandidate *daughter;

  ExRootCandConstIter itDaughters(Iterator());
  while((daughter = itDaughters.Next()))
  {
    if(daughter->Overlaps(object)) return kTRUE;
  }

  itDaughters = object->Iterator();
  while((daughter = itDaughters.Next()))
  {
    if(daughter->Overlaps(this)) return kTRUE;
  }

  return kFALSE;
}

//------------------------------------------------------------------------------

Bool_t ExRootCandidate::Equals(const ExRootCandidate *object) const
{
  if(object->GetUniqueID() != GetUniqueID()) return kFALSE;

  const ExRootCandidate *daughter;

  ExRootCandConstIter itDaughters(Iterator());
  while((daughter = itDaughters.Next()))
  {
    if(!daughter->Equals(object)) return kFALSE;
  }

  itDaughters = object->Iterator();
  while((daughter = itDaughters.Next()))
  {
    if(!daughter->Equals(this)) return kFALSE;
  }

  return kTRUE;
}

//------------------------------------------------------------------------------

void ExRootCandidate::SetMass(Double_t mass)
{
  fLorentzVector.SetVectM(fLorentzVector.Vect(), mass);
}

//------------------------------------------------------------------------------

void ExRootCandidate::SetP4(const TLorentzVector &p4)
{
  fLorentzVector = p4;
}

//------------------------------------------------------------------------------

void ExRootCandidate::SetMomentum(Double_t momentum)
{
  // this implementation leaves mass unchanged; subclasses may differ
  Double_t scale = 0.0;
  Double_t p = fLorentzVector.P();
  if(p != 0.0) scale = momentum / p;
  fLorentzVector.SetVect(scale*fLorentzVector.Vect());
}

//------------------------------------------------------------------------------

void ExRootCandidate::SetType(TParticlePDG *particle)
{
  const TParticlePDG *pdg = GetType();

  if(pdg == particle || particle == 0) return;

  SetInfo(particle);
  SetName(particle->GetName());

  //
  // by default:
  //   if the proper lifetime multiplied by light velocity is less
  //   than a nanometer, the object is considered a resonance
  //   (a state that does not fly)
  //
  fIsResonance = kFALSE;
  if(particle->Width() > 1.0e-15) fIsResonance = kTRUE; // Lifetime() < 1.0e-08

  if(!IsComposite())
  {
    // the mass has changed since the type has changed
    SetMass(GetMass());

    // set the charge
    SetCharge(particle->Charge());
  }
  else
  {
    if(GetCharge() != particle->Charge())
    {
      cout
      << "** ERROR: attempt to call ExRootCandidate::SetType(\""
      << particle->ParticleClass() << "\") for a composite" << endl
      << " ExRootCandidate whose daughters have total charge "
      << GetCharge() << endl;
    }
  }
}

//------------------------------------------------------------------------------

void ExRootCandidate::SetType(const char *name)
{
  TDatabasePDG *pdg = TDatabasePDG::Instance();
  TParticlePDG *particle;
  if((particle = pdg->GetParticle(name))) SetType(particle);
}

//------------------------------------------------------------------------------

void ExRootCandidate::SetType(Int_t pdgCode)
{
  TDatabasePDG *pdg = TDatabasePDG::Instance();
  TParticlePDG *particle;
  if((particle = pdg->GetParticle(pdgCode))) SetType(particle);
}

//------------------------------------------------------------------------------
// Geneology functions, no longer in a separate class
//------------------------------------------------------------------------------

void ExRootCandidate::AddDaughter(const ExRootCandidate *object)
{
  Add(object);
}

//------------------------------------------------------------------------------

void ExRootCandidate::AddDaughter(ExRootCandidate *object)
{
  Add(object);
}

//------------------------------------------------------------------------------

void ExRootCandidate::Add(const ExRootCandidate *object)
{
  Add(static_cast<ExRootCandidate *>(object->Clone()));
}

//------------------------------------------------------------------------------

void ExRootCandidate::Add(ExRootCandidate *object)
{
  ExRootCandList::Add(object);

  // as soon as there are daughters, the charge is
  // given by the sum of the daughter charges
  if(Size() == 0) fCharge = 0;

  fCharge += object->fCharge;
  fLorentzVector += object->fLorentzVector;

  // set the daughter's mother link
  object->fMother = this;
}

//------------------------------------------------------------------------------
// Access functions
//------------------------------------------------------------------------------

Double_t ExRootCandidate::GetMass() const
{
  const TParticlePDG *pdg = GetType();
  if(!IsComposite() && pdg)
    return pdg->Mass();
  else
    return fLorentzVector.M();
}

//------------------------------------------------------------------------------

TObject *ExRootCandidate::Clone(const char *newname) const
{
  ExRootCandidate *object = fFactory->NewCandidate();
  Copy(*object);
  return object;
}

//------------------------------------------------------------------------------

void ExRootCandidate::Copy(TObject &obj) const
{
  ExRootCandidate &object = (ExRootCandidate &) obj;

  ExRootCandList::Copy(obj);

  object.fIsResonance = fIsResonance;
  object.fCharge = fCharge;
  object.fLorentzVector = fLorentzVector;

  object.fMother = 0;

  const ExRootCandidate *daughterOld;
  ExRootCandidate *daughterNew;
  ExRootCandConstIter itDaughters(Iterator());
  while((daughterOld = itDaughters.Next()))
  {
    daughterNew = static_cast<ExRootCandidate *>(daughterOld->Clone());
    object.ExRootCandList::Add(daughterNew);
    daughterNew->fMother = static_cast<ExRootCandidate *>(&object);
  }

  map<const TClass *, TObject *>::const_iterator it;
  for(it = fInfo.begin(); it != fInfo.end(); ++it)
  {
    object.SetInfo(it->second->Clone());
  }
}

//------------------------------------------------------------------------------

void ExRootCandidate::Clear()
{
  SetUniqueID(0);
  fCharge = 0;
  fLorentzVector.SetXYZT(0.0, 0.0, 0.0, 0.0);
  fInfo.clear();
  ExRootCandList::Clear();
}

//------------------------------------------------------------------------------

const TObject *ExRootCandidate::GetInfo(const TClass *cl) const
{
  map<const TClass *, TObject *>::const_iterator it = fInfo.find(cl);

  return (it != fInfo.end() ? it->second : 0);
}

//------------------------------------------------------------------------------

TObject *ExRootCandidate::GetInfo(const TClass *cl)
{
  map<const TClass *, TObject *>::const_iterator it = fInfo.find(cl);

  return (it != fInfo.end() ? it->second : 0);
}

//------------------------------------------------------------------------------

void ExRootCandidate::SetInfo(TObject *info)
{
  fInfo[info->IsA()] = info;
}

//------------------------------------------------------------------------------

void ExRootCandidate::Browse(TBrowser *b)
{
/*
  map<const TClass *, TObject *>::const_iterator it = fInfo.find(ExRootDSTInfo::Class());

  if(it != fInfo.end()) b->Add(it->second);
*/
  ExRootCandList::Browse(b);
}

//------------------------------------------------------------------------------
