
#include <iostream>
#include <fstream>
#include <sstream>
#include <map>

#include "TROOT.h"
#include "TApplication.h"

#include "TFile.h"
#include "TChain.h"
#include "TString.h"

#include "TH2.h"
#include "THStack.h"
#include "TLegend.h"
#include "TPaveText.h"
#include "TLorentzVector.h"

#include "ExRootAnalysis/ExRootClasses.h"

#include "ExRootAnalysis/ExRootTreeWriter.h"
#include "ExRootAnalysis/ExRootTreeBranch.h"

#include "ExRootAnalysis/ExRootUtilities.h"
#include "ExRootAnalysis/ExRootProgressBar.h"

using namespace std;

struct MatchingTreeObject
{
  enum {maxDblParam = 58};
  Double_t dblParam[maxDblParam];
};

//------------------------------------------------------------------------------

class MatchingTreeConverter
{
public:
  MatchingTreeConverter(const char *outputFileName);
  ~MatchingTreeConverter();

  void ProcessObject();
  void Write();

  Long64_t GetNumberOfObjects(ifstream &inputFileStream);
  Bool_t ReadObject(ifstream &inputFileStream);

private:

  void AnalyseEvent(ExRootTreeBranch *branch);

  istringstream fBufferStream;
  string fBuffer;

  MatchingTreeObject fCurrentObject;

  TFile *fOutputFile;
  ExRootTreeWriter *fTreeWriter;

  ExRootTreeBranch *fBranchMatch;

};

//------------------------------------------------------------------------------

MatchingTreeConverter::MatchingTreeConverter(const char *outputFileName) :
  fOutputFile(0), fTreeWriter(0)
{
  fOutputFile = TFile::Open(outputFileName, "RECREATE");
  fTreeWriter = new ExRootTreeWriter(fOutputFile, "Match");

  // information about reconstructed event
  fBranchMatch = fTreeWriter->NewBranch("Match", ExRootGenMatch::Class());
}

//------------------------------------------------------------------------------

MatchingTreeConverter::~MatchingTreeConverter()
{
  if(fTreeWriter) delete fTreeWriter;
  if(fOutputFile) delete fOutputFile;
}

//------------------------------------------------------------------------------

Long64_t MatchingTreeConverter::GetNumberOfObjects(ifstream &inputFileStream)
{
  Long64_t counter = 0;
  Bool_t canReadNumber, canReadFile = kTRUE;
  Int_t number;
  int position = inputFileStream.tellg();
  inputFileStream.seekg(0, std::ios::beg);

  inputFileStream.clear();

  while(canReadFile)
  {
    do
    {
      getline(inputFileStream, fBuffer);
  
      if(!inputFileStream.good())
      {
        canReadFile = kFALSE;
        break;
      }

      fBufferStream.clear();
      fBufferStream.str(fBuffer);
      
      canReadNumber = (fBufferStream >> number);
    }
    while(!canReadNumber);

    ++counter;
  }

  inputFileStream.clear();

  inputFileStream.seekg(position, std::ios::beg);

  return (counter - 1);
}

//------------------------------------------------------------------------------

Bool_t MatchingTreeConverter::ReadObject(ifstream &inputFileStream)
{
  Int_t i;
  Bool_t canReadNumber;

  do
  {
    getline(inputFileStream, fBuffer);

    if(!inputFileStream.good()) return kFALSE;

    fBufferStream.clear();
    fBufferStream.str(fBuffer);

    canReadNumber = kTRUE;

    for(i = 0; canReadNumber && i < MatchingTreeObject::maxDblParam; ++i)
    {
      canReadNumber = (fBufferStream >> fCurrentObject.dblParam[i]);
    }
  }
  while(!canReadNumber);

  return kTRUE;
}

//---------------------------------------------------------------------------

void MatchingTreeConverter::Write()
{
  if(fTreeWriter) fTreeWriter->Write();
}

//---------------------------------------------------------------------------

void MatchingTreeConverter::ProcessObject()
{
  fTreeWriter->Clear();

  AnalyseEvent(fBranchMatch);

  fTreeWriter->Fill();
}


//---------------------------------------------------------------------------

void MatchingTreeConverter::AnalyseEvent(ExRootTreeBranch *branch)
{
  ExRootGenMatch *element;

  element = static_cast<ExRootGenMatch*>(branch->NewEntry());

  element->Npart = fCurrentObject.dblParam[0];
  element->Qjet1 = fCurrentObject.dblParam[1];
  element->Qjet2 = fCurrentObject.dblParam[2];
  element->Qjet3 = fCurrentObject.dblParam[3];
  element->Qjet4 = fCurrentObject.dblParam[4];
  element->Ptcjet1 = fCurrentObject.dblParam[5];
  element->Ptcjet2 = fCurrentObject.dblParam[6];
  element->Ptcjet3 = fCurrentObject.dblParam[7];
  element->Ptcjet4 = fCurrentObject.dblParam[8];
  element->Etacjet1 = fCurrentObject.dblParam[9];
  element->Etacjet2 = fCurrentObject.dblParam[10];
  element->Etacjet3 = fCurrentObject.dblParam[11];
  element->Etacjet4 = fCurrentObject.dblParam[12];
  element->Phicjet1 = fCurrentObject.dblParam[13];
  element->Phicjet2 = fCurrentObject.dblParam[14];
  element->Phicjet3 = fCurrentObject.dblParam[15];
  element->Phicjet4 = fCurrentObject.dblParam[16];
  element->Ptjet1 = fCurrentObject.dblParam[17];
  element->Ptjet2 = fCurrentObject.dblParam[18];
  element->Ptjet3 = fCurrentObject.dblParam[19];
  element->Ptjet4 = fCurrentObject.dblParam[20];
  element->Etajet1 = fCurrentObject.dblParam[21];
  element->Etajet2 = fCurrentObject.dblParam[22];
  element->Etajet3 = fCurrentObject.dblParam[23];
  element->Etajet4 = fCurrentObject.dblParam[24];
  element->Phijet1 = fCurrentObject.dblParam[25];
  element->Phijet2 = fCurrentObject.dblParam[26];
  element->Phijet3 = fCurrentObject.dblParam[27];
  element->Phijet4 = fCurrentObject.dblParam[28];
  element->Idres1 = fCurrentObject.dblParam[29];
  element->Ptres1 = fCurrentObject.dblParam[30];
  element->Etares1 = fCurrentObject.dblParam[31];
  element->Phires1 = fCurrentObject.dblParam[32];
  element->Idres2 = fCurrentObject.dblParam[33];
  element->Ptres2 = fCurrentObject.dblParam[34];
  element->Etares2 = fCurrentObject.dblParam[35];
  element->Phires2 = fCurrentObject.dblParam[36];
  element->Ptlep1 = fCurrentObject.dblParam[37];
  element->Etmiss = fCurrentObject.dblParam[38];
  element->Htjets = fCurrentObject.dblParam[39];
  element->Ptb = fCurrentObject.dblParam[40];
  element->Etab = fCurrentObject.dblParam[41];
  element->Ptbbar = fCurrentObject.dblParam[42];
  element->Etabbar = fCurrentObject.dblParam[43];
  element->Ptbj = fCurrentObject.dblParam[44];
  element->Etabj = fCurrentObject.dblParam[45];
  element->Qpar1 = fCurrentObject.dblParam[46];
  element->Qpar2 = fCurrentObject.dblParam[47];
  element->Qpar3 = fCurrentObject.dblParam[48];
  element->Qpar4 = fCurrentObject.dblParam[49];
  element->Ptpar1 = fCurrentObject.dblParam[50];
  element->Ptpar2 = fCurrentObject.dblParam[51];
  element->Ptpar3 = fCurrentObject.dblParam[52];
  element->Ptpar4 = fCurrentObject.dblParam[53];
  element->Ncjets = fCurrentObject.dblParam[54];
  element->Njets = fCurrentObject.dblParam[55];
  element->Nfile = fCurrentObject.dblParam[56];
  element->Nclus = fCurrentObject.dblParam[57];
}

//---------------------------------------------------------------------------

int main(int argc, char *argv[])
{
  char *appName = "ExRootLHEFConverter";

  if(argc != 3)
  {
    cout << " Usage: " << appName << " input_file" << " output_file" << endl;
    cout << " input_file - input file in LHEF format," << endl;
    cout << " output_file - output file in ROOT format." << endl;
    return 1;
  }

  gROOT->SetBatch();

  int appargc = 1;
  char *appargv[] = {appName};
  TApplication app(appName, &appargc, appargv);

  // Open a stream connected to an event file:
  ifstream inputFileStream(argv[1]);

  if(!inputFileStream.is_open())
  {
    cerr << "** ERROR: Can't open '" << argv[1] << "' for input" << endl;
    return 1;
  }

  // Create LHC Olympics converter:
  MatchingTreeConverter *converter = new MatchingTreeConverter(argv[2]);

  cout << "** Calculating number of objects to process. Please wait..." << endl;
  Long64_t allEntries = converter->GetNumberOfObjects(inputFileStream);
  cout << "** Input file contains " << allEntries << " objects" << endl;

  if(allEntries > 0)
  {
    ExRootProgressBar progressBar(allEntries);

    // Loop over all objects
    Long64_t entry = 0;
    while(converter->ReadObject(inputFileStream))
    {
      converter->ProcessObject();

      progressBar.Update(entry);
      
      ++entry;
    }
    progressBar.Finish();

    converter->Write();
  }

  cout << "** Exiting..." << endl;

  delete converter;
}


