
/** \class ExRootEventLoop
 *
 *  Analysis steering class.
 *  Implements events loop and modules management.
 *
 *  $Date: 2008-06-04 13:57:55 $
 *  $Revision: 1.1 $
 *
 *
 *  \author P. Demin - UCL, Louvain-la-Neuve
 *
 */

#include "ExRootAnalysis/ExRootEventLoop.h"
#include "ExRootAnalysis/ExRootFactory.h"

#include "ExRootAnalysis/ExRootConfReader.h"
#include "ExRootAnalysis/ExRootTreeReader.h"
#include "ExRootAnalysis/ExRootTreeWriter.h"

#include "ExRootAnalysis/ExRootUtilities.h"
#include "ExRootAnalysis/ExRootProgressBar.h"

#include "TROOT.h"
#include "TClass.h"
#include "TSystem.h"
#include "TFolder.h"
#include "TObjArray.h"

#include <iostream>

#include <string.h>
#include <stdio.h>

using namespace std;

ExRootEventLoop::ExRootEventLoop() :
  fAllEntries(0), fEventLoop(0)
{
  fChains = new TObjArray;
  fChains->SetOwner();
}

//------------------------------------------------------------------------------

ExRootEventLoop::~ExRootEventLoop()
{
  delete fChains;
  if(fEventLoop) delete fEventLoop;
}

//------------------------------------------------------------------------------

Long64_t ExRootEventLoop::GetEntries()
{
  ExRootTreeReader *reader = GetTreeReader();
  return reader ? reader->GetEntries() : 0;
}

//------------------------------------------------------------------------------

Bool_t ExRootEventLoop::ReadEvent(Long64_t entry)
{
  ExRootTreeReader *reader = GetTreeReader();
  return reader ? reader->ReadEntry(entry) : kFALSE;
}

//------------------------------------------------------------------------------

void ExRootEventLoop::Init()
{
  fEventLoop = NewTask(ExRootTask::Class(), "EventLoop");

  ExRootTask *task;
  const ExRootConfReader::ExRootTaskMap *modules = GetModules();
  ExRootConfReader::ExRootTaskMap::const_iterator itModules;

  ExRootConfParam param = GetParam("TaskList");
  Long_t i, size;
  TString name;

  size = param.GetSize();

  for(i = 0; i < size; ++i)
  {
    name = param[i].GetString();
    itModules = modules->find(name);
    if(itModules != modules->end())
    {
      cout << itModules->second << " \t " <<  itModules->first << endl;
      task = NewTask(itModules->second, itModules->first);
      fEventLoop->Add(task);
    }
    else
    {
      cout << "** ERROR: module '" << name;
      cout << "' is specified in TaskList but not configured.";
      return;
    }

  }

  param = GetParam("InputCollection");
  TChain *chain = 0, *firstChain = 0;
  size = param.GetSize();
  if(size > 0)
  {
    for(i = 0; i < size; ++i)
    {
      chain = new TChain("", "");
      fChains->Add(chain);
      name = param[i][0].GetString();
      chain->SetName(name);
      FillChain(chain, param[i][1].GetString());
      if(i == 0)
      {
        firstChain = chain;
      }
      else
      {
        firstChain->AddFriend(chain, name + i);
      }
    }
    GetTreeReader()->SetTree(firstChain);
  }

  cout << "** Calculating number of events to process. Please wait..." << endl;
  fAllEntries = GetEntries();
  cout << "** Chain contains " << fAllEntries << " events" << endl;

  if(fAllEntries <= 0)
  {
    cout << "** ERROR: cannot read any event for analysis" << endl;
    return;
  }

  fEventLoop->CleanTasks();
  fEventLoop->InitSubTasks();

}

//------------------------------------------------------------------------------

void ExRootEventLoop::Process()
{
  Long64_t entry;

  if(fAllEntries > 0)
  {
    ExRootProgressBar progressBar(fAllEntries);
    // Loop over all events
    for(entry = 0; entry < fAllEntries; ++entry)
    {
      if(!ReadEvent(entry))
      {
        cout << "** ERROR: cannot read event " << entry << endl;
        break;
      }
      
      Clear();

      fEventLoop->CleanTasks();
      fEventLoop->ProcessSubTasks();

      if(fTreeWriter) fTreeWriter->Fill();

      progressBar.Update(entry);
    }
    progressBar.Finish();
  }

}

//------------------------------------------------------------------------------

void ExRootEventLoop::Finish()
{
  fEventLoop->CleanTasks();
  fEventLoop->FinishSubTasks();
}

//------------------------------------------------------------------------------

void ExRootEventLoop::Clear()
{
  if(fTreeWriter) fTreeWriter->Clear();
  if(fFactory) fFactory->Clear();
}

//------------------------------------------------------------------------------

