#!/usr/bin/env tclsh

set prefix "tmp/"
set suffix " \\\n\t"

set srcSuf {.$(SrcSuf)}
set objSuf {.$(ObjSuf)}
set exeSuf {$(ExeSuf)}

proc dependencies {fileName firstLine {force 1} {command {}}} {
  global suffix headerFiles sourceFiles

  if {[info exists sourceFiles($fileName)]} return

  set sourceFiles($fileName) 1

  set list {}
  set fid [open $fileName]
  while {! [eof $fid]} {
    set line [gets $fid]
    if [regexp -- {^\s*#include\s*"((\w+/)+\w+\.(h|hh))"} $line] {
      set elements [split $line {"}]
      set file [lindex $elements 1]
      if [file exists $file] {
        lappend list $file
        set headerFiles($file) 1
      }
    }
  }
  
  if {[llength $list] > 0} {
    puts -nonewline $firstLine
    foreach file $list {puts -nonewline $suffix$file}
    if {$command != {}} {
      puts ""
      puts $command
    }
    puts ""
  } elseif {$force} {
    puts -nonewline $firstLine
    if {$command != {}} {
      puts ""
      puts $command
    }
    puts ""
  }

  close $fid  
}

proc dictDeps {dictVar args} {

  global prefix suffix srcSuf objSuf

  set dict [eval glob -nocomplain $args]
  
  set dictSrcFiles {}
  set dictObjFiles {}

  foreach fileName $dict {
    regsub {LinkDef\.h} $fileName {Dict} dictName
    set dictName $prefix$dictName
  
    lappend dictSrcFiles $dictName$srcSuf
    lappend dictObjFiles $dictName$objSuf
  
    dependencies $fileName "$dictName$srcSuf:$suffix$fileName"
  }
  
  puts -nonewline "${dictVar} = $suffix"
  puts [join $dictSrcFiles $suffix]
  puts ""

  puts -nonewline "${dictVar}_OBJ = $suffix"
  puts [join $dictObjFiles $suffix]
  puts ""

}

proc sourceDeps {srcPrefix args} {

  global prefix suffix srcSuf objSuf
  
  set source [eval glob -nocomplain $args]
    
  set srcObjFiles {}
  
  foreach fileName $source {
    regsub {\.cc} $fileName {} srcName
    set srcObjName $prefix$srcName
  
    lappend srcObjFiles $srcObjName$objSuf
  
    dependencies $fileName "$srcObjName$objSuf:$suffix$srcName$srcSuf"
  }

  puts -nonewline "${srcPrefix}_OBJ = $suffix"
  puts [join $srcObjFiles $suffix]
  puts ""
}

proc tclDeps {} {

  global prefix suffix srcSuf objSuf
   
  set source [glob -nocomplain {tcl/*.c}]
  
  set srcObjFiles {}
  
  foreach fileName $source {
    if {$fileName == "tcl/tclc.c" || $fileName == "tcl/tcl.c"} continue
 
    regsub {\.c} $fileName {} srcName
    set srcObjName $prefix$srcName
  
    lappend srcObjFiles $srcObjName$objSuf
  
    dependencies $fileName "$srcObjName$objSuf:$suffix$fileName"
  }
  
  puts -nonewline "TCL_OBJ = $suffix"
  puts [join $srcObjFiles $suffix]
  puts ""
}

proc stdhepDeps {} {

  global prefix suffix srcSuf objSuf
   
  set source [glob -nocomplain {mcfio/*.c} {stdhep/*.c}]
  
  set srcObjFiles {}

  foreach fileName $source {
    regsub {\.c} $fileName {} srcName
    set srcObjName $prefix$srcName
  
    lappend srcObjFiles $srcObjName$objSuf
  
    dependencies $fileName "$srcObjName$objSuf:$suffix$fileName"
  }
  
  puts -nonewline "STDHEP_OBJ = $suffix"
  puts [join $srcObjFiles $suffix]
  puts ""
}

proc stdhepExecutableDeps {} {

  global prefix suffix objSuf exeSuf
   
  set executable [glob -nocomplain {test/ExRootSTDHEPConverter.cpp} \
                                   {test/MatchingSTDHEPConverter.cpp}]

  set exeFiles {}
  
  foreach fileName $executable {
    regsub {\.cpp} $fileName {} exeObjName
    set exeObjName $prefix$exeObjName
    set exeName [file tail $exeObjName]

    lappend exeFiles $exeName$exeSuf
    lappend exeObjFiles $exeObjName$objSuf
    
    puts "$exeName$exeSuf:$suffix$exeObjName$objSuf"
    puts ""
  
    dependencies $fileName "$exeObjName$objSuf:$suffix$fileName"
  }
  
  if [info exists exeFiles] {
    puts -nonewline "STDHEP_EXECUTABLE = $suffix"
    puts [join $exeFiles $suffix]
    puts ""
  }
  if [info exists exeObjFiles] {
    puts -nonewline "STDHEP_EXECUTABLE_OBJ = $suffix"
    puts [join $exeObjFiles $suffix]
    puts ""
  }

}

proc executableDeps {} {

  global prefix suffix objSuf exeSuf
   
  set executable [glob -nocomplain {test/*.cpp}]
  
  set exeFiles {}
  
  foreach fileName $executable {
    if {$fileName == "test/ExRootSTDHEPConverter.cpp"} continue
    if {$fileName == "test/MatchingSTDHEPConverter.cpp"} continue
    regsub {\.cpp} $fileName {} exeObjName
    set exeObjName $prefix$exeObjName
    set exeName [file tail $exeObjName]

    lappend exeFiles $exeName$exeSuf
    lappend exeObjFiles $exeObjName$objSuf
    
    puts "$exeName$exeSuf:$suffix$exeObjName$objSuf"
    puts ""
  
    dependencies $fileName "$exeObjName$objSuf:$suffix$fileName"
  }
  
  if [info exists exeFiles] {
    puts -nonewline "EXECUTABLE = $suffix"
    puts [join $exeFiles $suffix]
    puts ""
  }
  if [info exists exeObjFiles] {
    puts -nonewline "EXECUTABLE_OBJ = $suffix"
    puts [join $exeObjFiles $suffix]
    puts ""
  }

}

proc headerDeps {} {
  global suffix headerFiles
    
  foreach fileName [array names headerFiles] {  
    dependencies $fileName "$fileName:" 0 "\t@touch \$@"
  }
}

puts {
#
# Makefile for ExRootAnalysis
#
# Author: P. Demin - UCL, Louvain-la-Neuve
#
# multi-platform configuration is taken from ROOT (root/test/Makefile)
#

ARCH   = $(shell root-config --arch)

CXX    =
ObjSuf = o
SrcSuf = cc
ExeSuf =
DllSuf = so
LibSuf = a
OutPutOpt     = -o # keep whitespace after "-o"

ROOTCFLAGS   := $(shell root-config --cflags)
ROOTLIBS     := $(shell root-config --libs)
ROOTGLIBS    := $(shell root-config --glibs)

ifeq ($(ARCH),win32)
PLATFORM      = win32
endif
ifeq ($(ARCH),win32gdk)
PLATFORM      = win32
endif

ifeq ($(ARCH),hpux)
# HP-UX with CC
CXX    = CC
CXXFLAGS      = -O +Z
LD     = CC
LDFLAGS= -O +a1 -z
SOFLAGS= -b
DllSuf = sl
endif

ifeq ($(ARCH),hpuxacc)
# HP-UX 10.x with aCC
CXX    = aCC
CXXFLAGS      = -O +Z
LD     = aCC
LDFLAGS= -O -z
SOFLAGS= -b
endif

ifeq ($(ARCH),hpuxia64acc)
# HP-UX 11i 1.5 (IA-64) with aCC
CXX    = aCC
CXXFLAGS      = +DD64 -O +Z
LD     = aCC
LDFLAGS= +DD64 -O -z
SOFLAGS= -b
endif

ifeq ($(ARCH),hpuxegcs)
# HP-UX 10.x with g++
CXXFLAGS      = -O -fPIC
CXX    = g++
LD     = g++
LDFLAGS= -O
SOFLAGS= -fPIC -shared
endif


ifeq ($(ARCH),hurddeb)
# GNU/Hurd
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared
endif

ifeq ($(ARCH),aix)
# IBM AIX xlC 4.x
CXX    = xlC
CXXFLAGS      = -O
LD     = xlC
LDFLAGS= -O
SOFLAGS=
ROOTLIBS     := $(shell root-config --nonew --libs)
ROOTGLIBS    := $(shell root-config --nonew --glibs)
endif

ifeq ($(ARCH),aix5)
# IBM AIX xlC 5.x
CXX    = xlC
CXXFLAGS      = -O
LD     = xlC
LDFLAGS= -O
SOFLAGS=
ROOTLIBS     := $(shell root-config --nonew --libs)
ROOTGLIBS    := $(shell root-config --nonew --glibs)
endif

ifeq ($(ARCH),aixegcs)
# IBM AIX with GCC
CXX    = g++
CXXFLAGS      = -O
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared
endif

ifeq ($(ARCH),solaris)
# Solaris CC
CXX    = /opt/SUNWspro/bin/CC
CXXFLAGS      = -O -KPIC
LD     = /opt/SUNWspro/bin/CC
LDFLAGS= -O
SOFLAGS= -G
endif

ifeq ($(ARCH),solarisCC5)
# Solaris CC 5.0
CXX    = CC
CXXFLAGS      = -O -KPIC
LD     = CC
LDFLAGS= -O
SOFLAGS= -G
endif

ifeq ($(ARCH),solarisegcs)
# Solaris egcs
CXX    = g++
CXXFLAGS      = -O -fPIC
LD     = CC
LDFLAGS= -O
SOFLAGS= -shared
endif

ifeq ($(ARCH),solarisgcc)
# Solaris gcc
CXX    = g++
CXXFLAGS      = -O -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared
endif

ifeq ($(ARCH),solariskcc)
# Solaris kcc
CXX    = KCC --one_instantiation_per_object
CXXFLAGS      = -O4 -KPIC
LD     = KCC
LDFLAGS= -O4
SOFLAGS=
endif

ifeq ($(ARCH),solarisx86)
# Solaris CC on Intel
CXX    = CC
CXXFLAGS      = -O -KPIC
LD     = CC
LDFLAGS= -O
SOFLAGS= -G
endif

ifeq ($(ARCH),sgicc)
# SGI
CXX    = CC -n32  -I/usr/include/CC.sgi
CXXFLAGS      = -O
LD     = CC -n32  -I/usr/include/CC.sgi
LDFLAGS= -O
SOFLAGS= -shared
endif

ifeq ($(ARCH),sgiegcs)
# SGI 6.x with EGCS
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O -Wl,-u,__builtin_new -Wl,-u,__builtin_delete -Wl,-u,__nw__FUiPv
SOFLAGS= -shared
endif

ifeq ($(ARCH),sgin32egcs)
# SGI 6.x with EGCS for n32 ABI
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O -L/usr/lib32 -Wl,-woff,134
SOFLAGS= -shared
endif

ifeq ($(ARCH),sgigcc)
# SGI with GCC
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O -Wl,-u,__builtin_new -Wl,-u,__builtin_delete -Wl,-u,__nw__FUiPv
SOFLAGS= -shared
endif

ifeq ($(ARCH),sgikcc)
# SGI with KCC
CXX    = KCC -n32 --one_instantiation_per_object
CXXFLAGS      = -O
LD     = KCC -n32
LDFLAGS= -O
SOFLAGS=
endif

ifeq ($(ARCH),alphagcc)
# Alpha/OSF with g++
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -Wl,-expect_unresolved,* -shared
endif

ifeq ($(ARCH),alphaegcs)
# Alpha/OSF with egcs
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -Wl,-expect_unresolved,* -shared
endif

ifeq ($(ARCH),alphakcc)
# Alpha/OSF with kai compiler (not yet valid)
CXX    = KCC --one_instantiation_per_object
CXXFLAGS      = -O -fPIC
LD     = KCC
LDFLAGS= -O
SOFLAGS= -Wl,-expect_unresolved,* -shared
endif

ifeq ($(ARCH),alphacxx6)
# Alpha/OSF with cxx6
CXX    = cxx
CXXFLAGS      = -O
LD     = cxx
LDFLAGS= -O
SOFLAGS= -shared -nocxxstd -Wl,-expect_unresolved,*,-msym
endif

ifeq ($(ARCH),alphacxx)
# Alpha/OSF with cxx5
CXX    = cxx
CXXFLAGS      = -O
LD     = cxx
LDFLAGS= -O
SOFLAGS= -Wl,-expect_unresolved,* -call_shared
endif

ifeq ($(ARCH),linuxrh51)
# Linux with gcc 2.7.2.x
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif

ifeq ($(ARCH),linuxrh42)
# Linux with gcc 2.7.2.x (RedHat 4.2)
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif

ifeq ($(ARCH),linuxdeb)
# Linux with gcc 2.7.2.x
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif

ifeq ($(ARCH),linuxdeb2)
# Linux with gcc 2.7.2.x
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif

ifeq ($(ARCH),linuxdeb2ppc)
# Debian/Linux on the PowerPC
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif


ifeq ($(ARCH),linuxsuse6)
# Linux with gcc 2.7.2.x
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif

ifeq ($(ARCH),linux)
# Linux with egcs, gcc 2.9x, gcc 3.x (>= RedHat 5.2)
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif

ifeq ($(ARCH),linuxkcc)
# Linux with the KAI compiler
CXX    = KCC --one_instantiation_per_object
CXXFLAGS      = -fPIC +K0
LD     = KCC
LDFLAGS= -O $(shell root-config --cflags)
SOFLAGS=
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif

ifeq ($(ARCH),linuxicc)
# Linux with Intel icc compiler
CXX    = icc
CXXFLAGS      = -O
LD     = icc
LDFLAGS= -O
SOFLAGS= -shared
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif

ifeq ($(ARCH),linuxppcegcs)
# MkLinux with egcs/glibc
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif

ifeq ($(ARCH),linuxia64gcc)
# Itanium Linux with gcc 2.9x
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif

ifeq ($(ARCH),linuxia64sgi)
# Itanium Linux with sgiCC
CXX    = sgiCC
CXXFLAGS      = -O -Wall -fPIC
LD     = gsgiCC
LDFLAGS= -O
SOFLAGS= -shared
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif

ifeq ($(ARCH),linuxia64ecc)
# Itanium Linux with Intel ecc
CXX    = ecc
CXXFLAGS      = -O
LD     = ecc
LDFLAGS= -O -i_dynamic
SOFLAGS= -shared
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif

ifeq ($(ARCH),linuxx8664gcc)
# AMD Opteron (64 bit mode) Linux with gcc 3.x
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif

ifeq ($(ARCH),linuxalphaegcs)
# Alpha Linux with egcs
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif

ifeq ($(ARCH),linuxarm)
# ARM Linux with egcs
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif

ifeq ($(ARCH),mklinux)
# MkLinux with libc5
CXX    = g++
CXXFLAGS      = -O -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif

ifeq ($(ARCH),freebsd)
# FreeBSD with libc5
CXX    = g++
CXXFLAGS      = -O -pipe -W -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared -Wl,-x
AR = ar
ARFLAGS = cru -crv
RANLIB = ranlib
endif

ifeq ($(ARCH),freebsd4)
# FreeBSD with glibc
CXX    = g++
CXXFLAGS      = -O -pipe -W -Wall -fPIC
LD     = g++
LDFLAGS= -O
SOFLAGS= -shared -Wl,-x
AR = ar
ARFLAGS = cru -crv
RANLIB = ranlib
endif

ifeq ($(ARCH),macosx)
# MacOS X with cc (GNU cc 2.95.2)
CXX    = c++
CXXFLAGS      = -O -pipe -Wall
LD     = c++
LDFLAGS= -O -Xlinker -bind_at_load -flat_namespace
# The SOFLAGS will be used to create the .dylib; the .so will
# be created separately
DllSuf = dylib
SOFLAGS= -dynamiclib -flat_namespace -undefined suppress
AR = ar
ARFLAGS = cru -crv
RANLIB = ranlib
endif

ifeq ($(ARCH),hiux)
# Hitachi HIUX
CXX    = g++
CXXFLAGS      = -O2 -fPIC
LD     = g++
LDFLAGS= -Wl,+s
SOFLAGS= -Wl,-b,-E -nostdlib -nostartfiles
DllSuf = sl
endif

ifeq ($(PLATFORM),win32)
# Windows with the VC++ compiler
ObjSuf = obj
SrcSuf = cxx
ExeSuf = .exe
DllSuf = dll
OutPutOpt     = -out:
CXX    = cl
CXXOPT = -O2
#CXXOPT = -Z7
CXXFLAGS      = $(CXXOPT) -G5 -GR -GX -MD -DWIN32 -D_WINDOWS -nologo \
  -DVISUAL_CPLUSPLUS -D_X86_=1 -D_DLL
LD     = link
LDOPT  = -opt:ref
#LDOPT  = -debug
LDFLAGS= $(LDOPT) -pdb:none -nologo
SOFLAGS= -DLL

ROOTLIBS     := $(shell root-config --nonew --libs)
ROOTGLIBS    := $(shell root-config --nonew --glibs)
EXPLLINKLIBS  = $(ROOTLIBS) $(ROOTGLIBS)
endif

ifeq ($(ARCH),win32gcc)
# Windows with gcc
DllSuf = dll
ExeSuf = .exe
CXX    = g++
CXXFLAGS      = -O -Wall -Woverloaded-virtual -I/usr/X11R6/include
LD     = g++
LDFLAGS= -O -Wl,--enable-auto-import
SOFLAGS= -shared -D_DLL -Wl,--export-all-symbols
EXPLLINKLIBS  = $(ROOTLIBS) $(ROOTGLIBS)
AR = ar
ARFLAGS = cru
RANLIB = ranlib
endif

ifeq ($(CXX),)
$(error $(ARCH) invalid architecture)
endif

CXXFLAGS += $(ROOTCFLAGS) -DDROP_CGAL -I. -Itcl -Imcfio -Istdhep -ICDFCones -ICDFCones/CDFcode
LIBS = $(ROOTLIBS) -lEG $(SYSLIBS)
GLIBS = $(ROOTGLIBS) $(SYSLIBS)
	
###

STATIC = lib/libExRootAnalysisPGS.$(LibSuf)
SHARED = lib/libExRootAnalysis.$(DllSuf)

all:

}

stdhepExecutableDeps

executableDeps

dictDeps {DICT} {src/*LinkDef.h} {modules/*LinkDef.h}

dictDeps {PGS_DICT} {pgs/*LinkDef.h}

sourceDeps {SOURCE} {src/*.cc} {modules/*.cc} {CDFCones/*.cc} {KtJet/*.cc} {CLHEP/src/*.cc}

sourceDeps {PGS_SOURCE} {src/ExRootClasses.cc} {src/ExRootTreeBranch.cc} {src/ExRootTreeWriter.cc}

sourceDeps {PGS} {pgs/*.cc}

stdhepDeps

tclDeps

headerDeps

puts {

###

all: $(SHARED) $(EXECUTABLE) $(STDHEP_EXECUTABLE)

static: $(STATIC)

$(STATIC): $(PGS_DICT_OBJ) $(PGS_SOURCE_OBJ) $(PGS_OBJ)
	@mkdir -p $(@D)
	$(AR) $(ARFLAGS) $@ $^
	$(RANLIB) $@

$(SHARED): $(DICT_OBJ) $(SOURCE_OBJ) $(TCL_OBJ)
		@mkdir -p $(@D)
		@echo ">> Building $@"
ifeq ($(ARCH),aix)
		@/usr/ibmcxx/bin/makeC++SharedLib $(OutPutOpt)$@ $(LIBS) -p 0 $^
else
ifeq ($(ARCH),aix5)
		@/usr/vacpp/bin/makeC++SharedLib $(OutPutOpt)$@ $(LIBS) -p 0 $^
else
ifeq ($(ARCH),macosx)
# We need to make both the .dylib and the .so
		@$(LD) $(SOFLAGS) $^ $(OutPutOpt)$@
		@$(LD) -bundle -undefined suppress $(LDFLAGS) $^ $(LIBS) \
		   $(OutPutOpt)$(subst .$(DllSuf),.so,$@)
else
ifeq ($(PLATFORM),win32)
		@bindexplib $* $^ > $*.def
		@lib -nologo -MACHINE:IX86 $^ -def:$*.def \
		   $(OutPutOpt)$(EVENTLIB)
		@$(LD) $(SOFLAGS) $(LDFLAGS) $^ $*.exp $(LIBS) \
		   $(OutPutOpt)$@
else
		@$(LD) $(SOFLAGS) $(LDFLAGS) $^ $(OutPutOpt) $@ $(EXPLLINKLIBS)
endif
endif
endif
endif

clean:
		@rm -f $(PGS_DICT_OBJ) $(PGS_SOURCE_OBJ) $(PGS_OBJ) $(DICT_OBJ) $(SOURCE_OBJ) $(TCL_OBJ) $(STDHEP_OBJ) core

distclean: clean
		@rm -f $(SHARED) $(STATIC) $(EXECUTABLE) $(STDHEP_EXECUTABLE)

###

.SUFFIXES: .$(SrcSuf) .$(ObjSuf) .$(DllSuf)

%Dict.$(SrcSuf):
	@mkdir -p $(@D)
	@echo ">> Generating $@"
	@rootcint -f $@ -c $<
	@echo "#define private public" > $@.arch
	@echo "#define protected public" >> $@.arch
	@mv $@ $@.base
	@cat $@.arch $< $@.base > $@
	@rm $@.arch $@.base

$(SOURCE_OBJ): tmp/%.$(ObjSuf): %.$(SrcSuf)
	@mkdir -p $(@D)
	@echo ">> Compiling $<"
	@$(CXX) $(CXXFLAGS) -c $< $(OutPutOpt)$@

$(PGS_OBJ): tmp/%.$(ObjSuf): %.$(SrcSuf)
	@mkdir -p $(@D)
	@echo ">> Compiling $<"
	@$(CXX) $(CXXFLAGS) -c $< $(OutPutOpt)$@

$(DICT_OBJ): %.$(ObjSuf): %.$(SrcSuf)
	@mkdir -p $(@D)
	@echo ">> Compiling $<"
	@$(CXX) $(CXXFLAGS) -c $< $(OutPutOpt)$@

$(PGS_DICT_OBJ): %.$(ObjSuf): %.$(SrcSuf)
	@mkdir -p $(@D)
	@echo ">> Compiling $<"
	@$(CXX) $(CXXFLAGS) -c $< $(OutPutOpt)$@

$(TCL_OBJ): tmp/%.$(ObjSuf): %.c
	@mkdir -p $(@D)
	@echo ">> Compiling $<"
	@gcc $(CXXFLAGS) -c $< $(OutPutOpt)$@

$(STDHEP_OBJ): tmp/%.$(ObjSuf): %.c
	@mkdir -p $(@D)
	@echo ">> Compiling $<"
	@gcc $(CXXFLAGS) -c $< $(OutPutOpt)$@

$(STDHEP_EXECUTABLE_OBJ): tmp/%.$(ObjSuf): %.cpp
	@mkdir -p $(@D)
	@echo ">> Compiling $<"
	@$(CXX) $(CXXFLAGS) -c $< $(OutPutOpt)$@

$(STDHEP_EXECUTABLE): %$(ExeSuf): $(DICT_OBJ) $(SOURCE_OBJ) $(TCL_OBJ) $(STDHEP_OBJ)
	@echo ">> Building $@"
	@$(LD) $(LDFLAGS) $^ $(LIBS) $(OutPutOpt)$@

$(EXECUTABLE_OBJ): tmp/%.$(ObjSuf): %.cpp
	@mkdir -p $(@D)
	@echo ">> Compiling $<"
	@$(CXX) $(CXXFLAGS) -c $< $(OutPutOpt)$@

$(EXECUTABLE): %$(ExeSuf): $(DICT_OBJ) $(SOURCE_OBJ) $(TCL_OBJ)
	@echo ">> Building $@"
	@$(LD) $(LDFLAGS) $^ $(LIBS) $(OutPutOpt)$@

###

}
