#ifndef ExRootCandidate_h
#define ExRootCandidate_h

/** \class ExRootCandidate
 *
 *  Class implementing particle candidate model.
 *
 *  $Date: 2008-06-04 13:57:23 $
 *  $Revision: 1.1 $
 *
 *
 *  \author P. Demin - UCL, Louvain-la-Neuve
 *
 */

#include "ExRootAnalysis/ExRootCandList.h"

#include "ExRootAnalysis/ExRootSortableObject.h"

#include "TNamed.h"
#include "TLorentzVector.h"
#include "TParticlePDG.h"

#include <map>

class TClass;
class TBrowser;
class TObjArray;
class TParticlePDG;

class ExRootFactory;

//------------------------------------------------------------------------------

class ExRootCandidate: public ExRootCandList
{
  friend class ExRootFactory;
  friend class ExRootCandList;

public:

  ExRootCandidate();

  ExRootCandidate(const ExRootCandidate &object);

  virtual ~ExRootCandidate() {}

  /** Return the charge of the candidate.
    @return charge of type double
    */
  Double_t GetCharge() const { return fCharge; }

  /** Assign charge to the candidate.
    @param charge of type double
    */
  void SetCharge(Double_t charge) { fCharge = charge; }

  /** Assign particle identification code.
    @param particle -- pointer to TParticlePDG
    */
  void SetType(TParticlePDG *particle);

  /** Assign particle identification code.
    @param name -- particle name
    */
  void SetType(const char *name);

  /** Assign particle identification code.
    @param pdgCode -- particle code
    */
  void SetType(Int_t pdgCode);

  /** Return particle identification informationn.
    @return pointer to TParticlePDG
    */
  const TParticlePDG *GetType() const { return GetInfo<TParticlePDG>(); }

  /** Assign mass to the candidate.
    @param mass of type double
    */
  void SetMass(Double_t mass);

  /** Assign momentum to the candidate (at constant mass).
    @param momentum of type double
    */
  void SetMomentum(Double_t momentum);

  /** Return the mass of the candidate.
    @return mass of type double
    */
  Double_t GetMass() const;

  /** Return the momentum 4-vector of the candidate.
    @return momentum of type TLorentzVector
    */
  const TLorentzVector &GetP4() const { return fLorentzVector; }

  /** Assign momentum 4-vector to the candidate.
    @param momentum of type double
    */

  void SetP4(const TLorentzVector &p4);

  //
  // Genealogy
  //

  /** Return the direct mother of the candidate.
    @return pointer to ExRootCandidate instance
    */
  const ExRootCandidate *Mother() const { return fMother; }

  Bool_t IsComposite() const { return Size() > 0; }

  Bool_t IsResonance() const { return fIsResonance; } // true when c*tau is very small (< 1nm)

  void SetResonance(Bool_t flag = kTRUE) { fIsResonance = flag; }

  //
  // Check for overlap
  //

  /** Candidates overlap if they are identical (same pointers),
    equal (same Base), clones (same UniqueID),
    representing a same reconstructed object,
    or having daughters that are overlapping
    @param object -- pointer to ExRootCandidate
    @return value of type Bool_t
    */
  Bool_t Overlaps(const ExRootCandidate *object) const;

  /** Candidates are equal if they are clones (same UniqueID),
    and all their daughters are equal
    @param object -- pointer to ExRootCandidate
    @return value of type Bool_t
    */
  Bool_t Equals(const ExRootCandidate *object) const;

  /** Candidates are clones if they have same UniqueID,
    same particle identification code and
    if they are compisite or non-composite in the same time.
    @param object -- pointer to ExRootCandidate
    @return value of type Bool_t
    */
  Bool_t IsCloneOf(const ExRootCandidate *object, Bool_t checkType = kFALSE) const;

  /** Return pointer to the first clone of a candidate given as parameter
    found in the decay tree of this candidate.
    @param object -- pointer to ExRootCandidate
    @return pointer to ExRootCandidate
    */
  const ExRootCandidate *FindCloneInTree(const ExRootCandidate *object) const;

  //
  // Accessors to specific information (reconstruction, Monte-Carlo, fit parameters)
  //

  /** Return additional information for this candiate
    in form of instance of class T.
    @return pointer to instance of class T
    */
  template<typename T>
  const T *GetInfo() const { return dynamic_cast<const T *>(GetInfo(T::Class())); }

  template<typename T>
  T *GetInfo() { return dynamic_cast<T *>(GetInfo(T::Class())); }

  /** Return additional information for this candiate
    in form of instance of a given class.
    @param cl -- pointer to TClass
    @return pointer to TObject
    */
  const TObject *GetInfo(const TClass *cl) const;

  TObject *GetInfo(const TClass *cl);

  /** Add additional information for this candiate
    @param info -- pointer to an object with additional information to be attched to this candidate
    */
  void SetInfo(TObject *info);

  /** Add link to a daughter of the candidate.
    @param object -- pointer to ExRootObject instance
    */
  virtual void Add(ExRootCandidate *object);
  virtual void Add(const ExRootCandidate *object);

  void AddDaughter(ExRootCandidate *object);
  void AddDaughter(const ExRootCandidate *object);

  /** Copy this candidate to object of type ExRootCandidate.
    @param object -- reference to TObject
    */
  virtual void Copy(TObject &object) const;

  /** Make a clone of this candidate.
    @return pointer to TObject
    */
  virtual TObject *Clone(const char *newname = "") const;

  virtual void Clear();

  /** Browse candidate.
    */
  virtual void Browse(TBrowser *b);

  /** Returns kTRUE in case candidate contains browsable objects.
    @return value of type Bool_t
    */
  virtual Bool_t IsFolder() const { return kTRUE; }

  /** Returns kTRUE in case this candidate is sortable.
    @return value of type Bool_t
    */
  virtual Bool_t IsSortable() const { return fgCompare ? fgCompare->IsSortable(this) : kFALSE; }

  /** Calls Compare method of an instance of a functor class that inherits from ExRootCompare.
    @return -1 if this is smaller than obj, 0 if candidates are equal and 1 if this is larger than obj.
    */
  virtual Int_t Compare(const TObject *obj) const { return fgCompare->Compare(this, obj); }

private:

  /**  resonance flag
    */
  Bool_t fIsResonance; //

  /** electrical charge
    */
  Double_t fCharge; //

  TLorentzVector fLorentzVector; //

  /** pointer to mother candidate
    */
  ExRootCandidate *fMother; //!

  static ExRootCompare *fgCompare; //!

  std::map<const TClass *, TObject *> fInfo;

  ClassDef(ExRootCandidate, 1)
};

#endif /* ExRootCandidate_h */

