
/** \class ExRootCandidate
 *
 *  A list of ExRootCandidates with iterators.
 *
 *  $Date: 2008-06-04 13:57:53 $
 *  $Revision: 1.1 $
 *
 *
 *  \author P. Demin - UCL, Louvain-la-Neuve
 *
 */

#include "ExRootAnalysis/ExRootCandList.h"
#include "ExRootAnalysis/ExRootCandidate.h"
#include "ExRootAnalysis/ExRootFactory.h"

#include "TObjArray.h"
#include "TBrowser.h"

ExRootCandList::ExRootCandList() :
  TNamed("ExRootCandList", ""),
  fFactory(0),
  fArray(0)
{
}

//------------------------------------------------------------------------------

ExRootCandList::ExRootCandList(const ExRootCandList &object) :
  TNamed("ExRootCandList", ""),
  fFactory(0),
  fArray(0)
{
  object.Copy(*this);
}

//------------------------------------------------------------------------------
// Geneology functions, no longer in a separate class
//------------------------------------------------------------------------------

void ExRootCandList::Add(ExRootCandidate *object)
{
  if(!fArray) fArray = fFactory->NewArray();
  fArray->Add(object);
}

//------------------------------------------------------------------------------

void ExRootCandList::Add(const ExRootCandidate *object)
{
  if(!fArray) fArray = fFactory->NewArray();
  fArray->Add(object->Clone());
}

//------------------------------------------------------------------------------
// Access functions
//------------------------------------------------------------------------------

const ExRootCandidate *ExRootCandList::At(Int_t i) const
{
  if(i >= 0 && i < Size())
    return static_cast<ExRootCandidate *>(fArray->UncheckedAt(i));
  else
    return 0;
}

//------------------------------------------------------------------------------

Int_t ExRootCandList::Size() const
{
  if(!fArray) return 0;
  return fArray->GetEntriesFast();
}

//------------------------------------------------------------------------------

TObject *ExRootCandList::Clone(const char *newname) const
{
  ExRootCandList *object = fFactory->NewCandList();
  Copy(*object);
  return object;
}

//------------------------------------------------------------------------------

void ExRootCandList::Copy(TObject &obj) const
{
  ExRootCandList &object = (ExRootCandList &) obj;

  TNamed::Copy(obj);

  object.fFactory = fFactory;
  object.fArray = 0;
}

//------------------------------------------------------------------------------

void ExRootCandList::Clear()
{
  if(fArray) fArray->Clear();
}

//------------------------------------------------------------------------------

void ExRootCandList::Sort(ExRootCompare *compare)
{
  if(fArray)
  {
    ExRootCompare *backup = ExRootCandidate::fgCompare;
    ExRootCandidate::fgCompare = compare;
    fArray->Sort();
    ExRootCandidate::fgCompare = backup;
  }
}

//------------------------------------------------------------------------------

void ExRootCandList::Browse(TBrowser *b)
{
  if(fArray) fArray->Browse(b);
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------

ExRootCandIter::ExRootCandIter(TObjArray *array)
{
  fArray = array;
  Reset();
}

//------------------------------------------------------------------------------

ExRootCandIter::ExRootCandIter(ExRootCandList *object)
{
  fArray = object->fArray;
  Reset();
}

//------------------------------------------------------------------------------

ExRootCandIter::ExRootCandIter(const ExRootCandIter &iter)
{
  fArray = iter.fArray;
  fCursor = iter.fCursor;
}

//------------------------------------------------------------------------------

ExRootCandIter &ExRootCandIter::operator=(const ExRootCandIter &rhs)
{
  if(this != &rhs)
  {
    fArray = rhs.fArray;
    fCursor = rhs.fCursor;
  }
  return *this;
}

//------------------------------------------------------------------------------

ExRootCandidate *ExRootCandIter::Next(Bool_t direction)
{
  // By default the iteration direction is kIterForward.
  // To go backward use kIterBackward.
  // Return next object in array. Returns 0 when no more objects in array.

  if(!fArray) return 0;

  if(direction == kIterForward)
  {
    Int_t size = fArray->Capacity();
    while(fCursor < size && fArray->UncheckedAt(fCursor) == 0) { ++fCursor; }
    if(fCursor < size) return static_cast<ExRootCandidate *>(fArray->UncheckedAt(fCursor++));
  }
  else
  {
    while(fCursor >= 0 && fArray->UncheckedAt(fCursor) == 0) { --fCursor; }
    if(fCursor >= 0) return static_cast<ExRootCandidate *>(fArray->UncheckedAt(fCursor--));
  }
  return 0;
}

//------------------------------------------------------------------------------

void ExRootCandIter::Reset(Bool_t direction)
{
  if(direction == kIterForward)
  {
    fCursor = 0;
  }
  else
  {
    if(fArray) fCursor = fArray->Capacity() - 1;
  }
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------

ExRootCandConstIter::ExRootCandConstIter(const TObjArray *array)
{
  fArray = array;
  Reset();
}

//------------------------------------------------------------------------------

ExRootCandConstIter::ExRootCandConstIter(const ExRootCandList *object)
{
  fArray = object->fArray;
  Reset();
}

//------------------------------------------------------------------------------

ExRootCandConstIter::ExRootCandConstIter(const ExRootCandIter &iter)
{
  fArray = iter.fArray;
  fCursor = iter.fCursor;
}

//------------------------------------------------------------------------------

ExRootCandConstIter::ExRootCandConstIter(const ExRootCandConstIter &iter)
{
  fArray = iter.fArray;
  fCursor = iter.fCursor;
}

//------------------------------------------------------------------------------

ExRootCandConstIter &ExRootCandConstIter::operator=(const ExRootCandIter &rhs)
{
  fArray = rhs.fArray;
  fCursor = rhs.fCursor;
  return *this;
}

//------------------------------------------------------------------------------

ExRootCandConstIter &ExRootCandConstIter::operator=(const ExRootCandConstIter &rhs)
{
  if(this != &rhs)
  {
    fArray = rhs.fArray;
    fCursor = rhs.fCursor;
  }
  return *this;
}

//------------------------------------------------------------------------------

const ExRootCandidate *ExRootCandConstIter::Next(Bool_t direction)
{
  // By default the iteration direction is kIterForward.
  // To go backward use kIterBackward.
  // Return next object in array. Returns 0 when no more objects in array.

  if(!fArray) return 0;

  if(direction == kIterForward)
  {
    Int_t size = fArray->Capacity();
    while(fCursor < size && fArray->UncheckedAt(fCursor) == 0) { ++fCursor; }
    if(fCursor < size) return static_cast<ExRootCandidate *>(fArray->UncheckedAt(fCursor++));
  }
  else
  {
    while(fCursor >= 0 && fArray->UncheckedAt(fCursor) == 0) { --fCursor; }
    if(fCursor >= 0) return static_cast<ExRootCandidate *>(fArray->UncheckedAt(fCursor--));
  }
  return 0;
}

//------------------------------------------------------------------------------

void ExRootCandConstIter::Reset(Bool_t direction)
{
  if(direction == kIterForward)
  {
    fCursor = 0;
  }
  else
  {
    if(fArray) fCursor = fArray->Capacity() - 1;
  }
}

//------------------------------------------------------------------------------

