#ifndef ExRootClasses_h
#define ExRootClasses_h

/** \class ExRootClasses
 *
 *  Definition of classes to be stored in the root tree.
 *  Function ExRootCompareXYZ sorts objects by the variable XYZ that MUST be
 *  present in the data members of the root tree class of the branch.
 *
 *  $Date: 2008-06-04 13:57:24 $
 *  $Revision: 1.1 $
 *
 *
 *  \author P. Demin - UCL, Louvain-la-Neuve
 *
 */

// Dependencies (#includes)

#include "TRef.h"
#include "TObject.h"
#include "TRefArray.h"

#include "ExRootAnalysis/ExRootSortableObject.h"

//---------------------------------------------------------------------------

class ExRootLHEFEvent: public TObject
{
public:

  Long64_t Number; // event number

  Int_t Nparticles; // number of particles in the event | hepup.NUP
  Int_t ProcessID; // subprocess code for the event | hepup.IDPRUP

  Double_t Weight; // weight for the event | hepup.XWGTUP
  Double_t ScalePDF; // scale in GeV used in the calculation of the PDFs in the event | hepup.SCALUP
  Double_t CouplingQED; // value of the QED coupling used in the event | hepup.AQEDUP
  Double_t CouplingQCD; // value of the QCD coupling used in the event | hepup.AQCDUP

  ClassDef(ExRootLHEFEvent, 2)
};

//---------------------------------------------------------------------------

class ExRootLHEFParticle: public ExRootSortableObject
{
public:

  Int_t PID; // particle HEP ID number | hepup.IDUP[number]
  Int_t Status; // particle status code | hepup.ISTUP[number]
  Int_t Mother1; // index for the particle first mother | hepup.MOTHUP[number][0]
  Int_t Mother2; // index for the particle last mother | hepup.MOTHUP[number][1]
  Int_t ColorLine1; // index for the particle color-line | hepup.ICOLUP[number][0]
  Int_t ColorLine2; // index for the particle anti-color-line | hepup.ICOLUP[number][1]

  Double_t Px; // particle momentum vector (x component) | hepup.PUP[number][0]
  Double_t Py; // particle momentum vector (y component) | hepup.PUP[number][1]
  Double_t Pz; // particle momentum vector (z component) | hepup.PUP[number][2]
  Double_t E; // particle energy | hepup.PUP[number][3]
  Double_t M; // particle mass | hepup.PUP[number][4]

  Double_t PT; // particle transverse momentum
  Double_t Eta; // particle pseudorapidity
  Double_t Phi; // particle azimuthal angle

  Double_t Rapidity; // particle rapidity

  Double_t LifeTime; // particle invariant lifetime
                     // (c*tau, distance from production to decay in mm)
                     // | hepup.VTIMUP[number]

  Double_t Spin; // cosine of the angle between the particle spin vector
                 // and the decaying particle 3-momentum,
                 // specified in the lab frame. | hepup.SPINUP[number]

  static ExRootCompare *fgCompare; //!
  const ExRootCompare *GetCompare() const { return fgCompare; }

  ClassDef(ExRootLHEFParticle, 2)
};

//---------------------------------------------------------------------------

class ExRootGenEvent: public TObject
{
public:

  Long64_t Number; // event number | hepevt.nevhep

  ClassDef(ExRootGenEvent, 1)
};

//---------------------------------------------------------------------------

class ExRootGenParticle: public ExRootSortableObject
{
public:
  Int_t PID; // particle HEP ID number | hepevt.idhep[number]
  Int_t Status; // particle status | hepevt.isthep[number]
  Int_t M1; // particle 1st mother | hepevt.jmohep[number][0] - 1
  Int_t M2; // particle 2nd mother | hepevt.jmohep[number][1] - 1
  Int_t D1; // particle 1st daughter | hepevt.jdahep[number][0] - 1
  Int_t D2; // particle 2nd daughter | hepevt.jdahep[number][1] - 1

  Double_t E; // particle energy | hepevt.phep[number][3]
  Double_t Px; // particle momentum vector (x component) | hepevt.phep[number][0]
  Double_t Py; // particle momentum vector (y component) | hepevt.phep[number][1]
  Double_t Pz; // particle momentum vector (z component) | hepevt.phep[number][2]

  Double_t PT; // particle transverse momentum
  Double_t Eta; // particle pseudorapidity
  Double_t Phi; // particle azimuthal angle

  Double_t Rapidity; // particle rapidity

  Double_t T; // particle vertex position (t component) | hepevt.vhep[number][3]
  Double_t X; // particle vertex position (x component) | hepevt.vhep[number][0]
  Double_t Y; // particle vertex position (y component) | hepevt.vhep[number][1]
  Double_t Z; // particle vertex position (z component) | hepevt.vhep[number][2]

  static ExRootCompare *fgCompare; //!
  const ExRootCompare *GetCompare() const { return fgCompare; }

  ClassDef(ExRootGenParticle, 1)
};

//---------------------------------------------------------------------------

class ExRootGenJet: public ExRootSortableObject
{
public:

  Double_t E; // jet energy
  Double_t Px; // jet momentum vector (x component)
  Double_t Py; // jet momentum vector (y component)
  Double_t Pz; // jet momentum vector (z component)

  Double_t PT; // jet transverse momentum
  Double_t Eta; // jet pseudorapidity
  Double_t Phi; // jet azimuthal angle

  Double_t Rapidity; // jet rapidity

  Double_t Mass; // jet invariant mass

  static ExRootCompare *fgCompare; //!
  const ExRootCompare *GetCompare() const { return fgCompare; }

  ClassDef(ExRootGenJet, 1)
};

//---------------------------------------------------------------------------

class ExRootMatching: public TObject
{
public:
  Double_t DMerge;
  Double_t YMerge;

  ClassDef(ExRootMatching, 1)
};

//---------------------------------------------------------------------------

class ExRootGenMatch: public TObject
{
public:

  Double_t Npart;
  Double_t Qjet1;
  Double_t Qjet2;
  Double_t Qjet3;
  Double_t Qjet4;
  Double_t Ptcjet1;
  Double_t Ptcjet2;
  Double_t Ptcjet3;
  Double_t Ptcjet4;
  Double_t Etacjet1;
  Double_t Etacjet2;
  Double_t Etacjet3;
  Double_t Etacjet4;
  Double_t Phicjet1;
  Double_t Phicjet2;
  Double_t Phicjet3;
  Double_t Phicjet4;
  Double_t Ptjet1;
  Double_t Ptjet2;
  Double_t Ptjet3;
  Double_t Ptjet4;
  Double_t Etajet1;
  Double_t Etajet2;
  Double_t Etajet3;
  Double_t Etajet4;
  Double_t Phijet1;
  Double_t Phijet2;
  Double_t Phijet3;
  Double_t Phijet4;
  Double_t Idres1;
  Double_t Ptres1;
  Double_t Etares1;
  Double_t Phires1;
  Double_t Idres2;
  Double_t Ptres2;
  Double_t Etares2;
  Double_t Phires2;
  Double_t Ptlep1;
  Double_t Etmiss;
  Double_t Htjets;
  Double_t Ptb;
  Double_t Etab;
  Double_t Ptbbar;
  Double_t Etabbar;
  Double_t Ptbj;
  Double_t Etabj;
  Double_t Qpar1;
  Double_t Qpar2;
  Double_t Qpar3;
  Double_t Qpar4;
  Double_t Ptpar1;
  Double_t Ptpar2;
  Double_t Ptpar3;
  Double_t Ptpar4;
  Double_t Ncjets;
  Double_t Njets;
  Double_t Nfile;
  Double_t Nclus;

  ClassDef(ExRootGenMatch, 1)
};

//---------------------------------------------------------------------------

class ExRootEvent: public TObject
{
public:

  Long64_t Number; // event number
  Int_t Trigger; // trigger word

  ClassDef(ExRootEvent, 1)
};

//---------------------------------------------------------------------------

class ExRootMissingET: public TObject
{
public:
  Double_t MET; // mising transverse energy
  Double_t Phi; // mising energy azimuthal angle

  ClassDef(ExRootMissingET, 1)
};

//---------------------------------------------------------------------------

class ExRootPhoton: public ExRootSortableObject
{
public:

  Double_t PT; // photon transverse momentum
  Double_t Eta; // photon pseudorapidity
  Double_t Phi; // photon azimuthal angle

  Double_t EhadOverEem; // ratio of the hadronic versus electromagnetic energy
                        // deposited in the calorimeter

  static ExRootCompare *fgCompare; //!
  const ExRootCompare *GetCompare() const { return fgCompare; }

  ClassDef(ExRootPhoton, 2)
};

//---------------------------------------------------------------------------

class ExRootElectron: public ExRootSortableObject
{
public:

  Double_t PT; // electron transverse momentum
  Double_t Eta; // electron pseudorapidity
  Double_t Phi; // electron azimuthal angle

  Double_t Charge; // electron charge

  Double_t Ntrk; // number of tracks associated with the electron

  Double_t EhadOverEem; // ratio of the hadronic versus electromagnetic energy
                        // deposited in the calorimeter

  static ExRootCompare *fgCompare; //!
  const ExRootCompare *GetCompare() const { return fgCompare; }

  ClassDef(ExRootElectron, 2)
};

//---------------------------------------------------------------------------

class ExRootMuon: public ExRootSortableObject
{
public:

  Double_t PT; // muon transverse momentum
  Double_t Eta; // muon pseudorapidity
  Double_t Phi; // muon azimuthal angle

  Double_t Charge; // muon charge

  Double_t Ntrk; // number of tracks associated with the muon

  Double_t PTiso; // sum of tracks transverse momentum within a cone of radius R=0.4
                  // centered on the muon (excluding the muon itself)

  Double_t ETiso; // ratio of ET in a 3x3 calorimeter cells array around the muon
                  // (including the muon's cell) to the muon PT

  Int_t JetIndex; // index of the closest jet

  static ExRootCompare *fgCompare; //!
  const ExRootCompare *GetCompare() const { return fgCompare; }

  ClassDef(ExRootMuon, 2)
};

//---------------------------------------------------------------------------

class ExRootTau: public ExRootSortableObject
{
public:

  Double_t PT; // tau transverse momentum
  Double_t Eta; // tau pseudorapidity
  Double_t Phi; // tau azimuthal angle

  Double_t Charge; // tau charge

  Double_t Ntrk; // number of charged tracks associated with the tau

  Double_t EhadOverEem; // ratio of the hadronic versus electromagnetic energy
                        // deposited in the calorimeter

  static ExRootCompare *fgCompare; //!
  const ExRootCompare *GetCompare() const { return fgCompare; }

  ClassDef(ExRootTau, 2)
};

//---------------------------------------------------------------------------

class ExRootJet: public ExRootSortableObject
{
public:

  Double_t PT; // jet transverse momentum
  Double_t Eta; // jet pseudorapidity
  Double_t Phi; // jet azimuthal angle

  Double_t Mass; // jet invariant mass

  Double_t Ntrk; // number of tracks associated with the jet

  Double_t BTag; // 1 or 2 for a jet that has been tagged as containing a heavy quark

  Double_t EhadOverEem; // ratio of the hadronic versus electromagnetic energy
                        // deposited in the calorimeter

  Int_t Index; // jet index in the LHC Olympics file

  static ExRootCompare *fgCompare; //!
  const ExRootCompare *GetCompare() const { return fgCompare; }

  ClassDef(ExRootJet, 2)
};

#endif // ExRootClasses_h


